package ru.yandex.travel.api.endpoints.hotels_extranet.tabledata

import io.grpc.StatusRuntimeException
import io.swagger.annotations.ApiOperation
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.ExceptionHandler
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import org.springframework.web.context.request.async.DeferredResult
import ru.yandex.travel.api.endpoints.hotels_extranet.tabledata.req_rsp.GetBankOrderDetailsRspV1
import ru.yandex.travel.api.endpoints.hotels_extranet.tabledata.req_rsp.GetBankOrdersReqV1
import ru.yandex.travel.api.endpoints.hotels_extranet.tabledata.req_rsp.GetBankOrdersRspV1
import ru.yandex.travel.api.endpoints.hotels_extranet.tabledata.req_rsp.GetOrdersReqV1
import ru.yandex.travel.api.endpoints.hotels_extranet.tabledata.req_rsp.GetOrdersRspV1
import ru.yandex.travel.api.exceptions.GrpcError
import ru.yandex.travel.api.infrastucture.ResponseProcessor
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers
import ru.yandex.travel.api.services.extranet.ExtranetClientFactory
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.hotels.extranet.TGetBankOrderDetailsReq
import java.util.concurrent.CompletableFuture
import javax.validation.Valid

@RestController
@RequestMapping(value = ["/api/hotels_extranet"])
class HotelExtranetOrdersController @Autowired constructor(
    private val clientFactory: ExtranetClientFactory,
    private val responseProcessor: ResponseProcessor,
    private val slugService: HotelSlugService,
) {

    @PostMapping(
        value = ["/v1/get_orders"],
        produces = [MediaType.APPLICATION_JSON_VALUE]
    )
    @ApiOperation(value = "Query orders for the extranet app", response = GetOrdersRspV1::class)
    fun requestBankOrder(@Valid @RequestBody request: GetOrdersReqV1): DeferredResult<GetOrdersRspV1> {
        return responseProcessor.replyWithFutureRetrying(
            "ExtranetGetOrders",
            { getOrdersRspV1CompletableFuture(request) },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        )
    }

    private fun getOrdersRspV1CompletableFuture(request: GetOrdersReqV1): CompletableFuture<GetOrdersRspV1> {
        return FutureUtils.buildCompletableFuture(
            clientFactory.createDataServiceFutureStub().getOrders(
                request.toProto()
            )
        ).thenApply { GetOrdersRspV1.fromProto(it, slugService) }
    }

    @PostMapping(
        value = ["/v1/get_bank_orders"],
        produces = [MediaType.APPLICATION_JSON_VALUE]
    )
    @ApiOperation(
        value = "Query BANK orders for the extranet app",
        response = GetOrdersRspV1::class
    )
    fun request(@Valid @RequestBody request: GetBankOrdersReqV1): DeferredResult<GetBankOrdersRspV1> {
        return responseProcessor.replyWithFutureRetrying(
            "ExtranetGetBankOrders",
            { getBankOrdersRspV1CompletableFuture(request) },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        )
    }

    private fun getBankOrdersRspV1CompletableFuture(request: GetBankOrdersReqV1):
        CompletableFuture<GetBankOrdersRspV1> {
        return FutureUtils.buildCompletableFuture(
            clientFactory.createDataServiceFutureStub().getBankOrderInfo(
                request.toProto()
            )
        ).thenApply { GetBankOrdersRspV1.fromProto(it, slugService) }
    }

    @GetMapping(
        value = ["/v1/bank_orders/{id}"],
        produces = [MediaType.APPLICATION_JSON_VALUE]
    )
    @ApiOperation(
        value = "Get details of the specific order",
        response = GetBankOrderDetailsRspV1::class
    )
    fun request(@PathVariable("id") id: String): DeferredResult<GetBankOrderDetailsRspV1> {
        return responseProcessor.replyWithFutureRetrying(
            "ExtranetGetBankOrderDetails",
            { getBankOrderDetailsRspV1CompletableFuture(id) },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        )
    }

    @ExceptionHandler(StatusRuntimeException::class)
    fun handleGrpcErrors(ex: StatusRuntimeException?): ResponseEntity<GrpcError>? {
        val error = GrpcError.fromGrpcStatusRuntimeException(ex)
        return ResponseEntity.status(error.status).contentType(MediaType.APPLICATION_JSON).body(error)
    }

    private fun getBankOrderDetailsRspV1CompletableFuture(paymentBatchId: String):
        CompletableFuture<GetBankOrderDetailsRspV1> {
        return FutureUtils.buildCompletableFuture(
            clientFactory.createDataServiceFutureStub().getBankOrderDetails(
                TGetBankOrderDetailsReq.newBuilder()
                    .setPaymentBatchId(paymentBatchId)
                    .build()
            )
        ).thenApply { GetBankOrderDetailsRspV1.fromProto(it) }
    }
}
