package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.api.models.common.ExtraVisitAndUserParams;
import ru.yandex.travel.api.models.hotels.OfferCacheMetadata;
import ru.yandex.travel.api.models.hotels.PartnerInfo;
import ru.yandex.travel.api.models.hotels.SearchFilterAndTextParams;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.offercache.api.TPrice;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.region_pages.EPageType;

@Slf4j
public class ExtraVisitAndUserParamsUtils {

    private static ExtraVisitAndUserParams create(ExtraVisitAndUserParams.VisitParams.Hotels hotelsParams) {
        ExtraVisitAndUserParams result = new ExtraVisitAndUserParams();
        var visitParams = new ExtraVisitAndUserParams.VisitParams();
        visitParams.setHotels(hotelsParams);
        result.setVisitParams(visitParams);
        return result;
    }

    public static ExtraVisitAndUserParams createForHotelPage(Map<String, String> pageParams) {
        var hotels = new ExtraVisitAndUserParams.VisitParams.Hotels();
        hotels.setHotelPage(pageParams);
        return create(hotels);
    }

    public static ExtraVisitAndUserParams createForSearchPage(Map<String, String> pageParams) {
        var hotels = new ExtraVisitAndUserParams.VisitParams.Hotels();
        hotels.setSearchPage(pageParams);
        return create(hotels);
    }

    public static ExtraVisitAndUserParams createForSeoRegionPage(Map<String, String> pageParams) {
        var hotels = new ExtraVisitAndUserParams.VisitParams.Hotels();
        hotels.setSeoRegionPage(pageParams);
        return create(hotels);
    }

    public static ExtraVisitAndUserParams createForBookingPage(Map<String, String> pageParams) {
        var hotels = new ExtraVisitAndUserParams.VisitParams.Hotels();
        hotels.setBookingPage(pageParams);
        return create(hotels);
    }

    public static Map<String, String> initParamsMap() {
        return new HashMap<>();
    }

    private static String getPartnerName(EPartnerId partnerId) {
        String result = partnerId.toString();
        if (result.startsWith("PI_")) {
            result = result.substring(3);
        }
        return result.substring(0, 1).toUpperCase() + result.substring(1).toLowerCase();
    }

    private static String integerToString(Integer v) {
        return (v == null) ? "" : Integer.toString(v);
    }

    private static String boolToString(boolean v) {
        return v ? "1" : "0";
    }

    private static void fillPartners(Map<String, String> pageParams, OfferCacheMetadata ocMeta,
                                     Collection<Integer> partners, String suffix) {
        boolean anyBoYRef = false;
        boolean anyClickoutRef = false;
        for (Map.Entry<Integer, PartnerInfo> partnerEntry : ocMeta.getPartnerById().entrySet()) {
            boolean present = partners != null && partners.contains(partnerEntry.getKey());
            if (present) {
                if (partnerEntry.getValue().isBookOnYandex()) {
                    anyBoYRef = true;
                } else {
                    anyClickoutRef = true;
                }
            }
            String name = getPartnerName(EPartnerId.forNumber(partnerEntry.getKey()));
            pageParams.put("has" + name + suffix, boolToString(present));
        }
        pageParams.put("hasAnyBookOnYandex" + suffix, boolToString(anyBoYRef));
        pageParams.put("hasAnyClickout" + suffix, boolToString(anyClickoutRef));
    }

    public static void fillHotelRefs(Map<String, String> pageParams, OfferCacheMetadata ocMeta, GeoHotel geoHotel) {
        List<Integer> partners = null;
        if (geoHotel.getOfferCacheResponse() != null) {
            partners = geoHotel.getOfferCacheResponse().getAvailablePartnerIdsList();
        }
        fillPartners(pageParams, ocMeta, partners,  "Ref");
    }

    public static void fillPermalink(Map<String, String> pageParams, Long permalink) {
        pageParams.put("hotelPermalink", permalink.toString());
    }

    public static void fillFilterParams(Map<String, String> pageParams, SearchFilterAndTextParams params) {

        ObjectMapper objectMapper = new ObjectMapper();

        // check filter params what not null
        Map<String, String> filters = new HashMap<>();
        var filterAtoms = params.getFilterAtoms();
        if (filterAtoms != null) {
            filters.put("filterAtoms", filterAtoms.toString());
        }
        var priceFrom = params.getFilterPriceFrom();
        if (priceFrom != null) {
            filters.put("priceFrom", priceFrom.toString());
        }
        var priceTo = params.getFilterPriceTo();
        if (priceTo != null) {
            filters.put("priceTo", priceTo.toString());
        }
        if (!filters.isEmpty()) {
            try {
                pageParams.put("filterParams", objectMapper.writeValueAsString(filters));
            } catch(Exception e) {
                log.error("{}: put filterParams in visitParams", e);
            }
        }
    }

    public static void fillHotelOffers(Map<String, String> pageParams, OfferCacheMetadata ocMeta, GeoHotel geoHotel) {
        Set<Integer> partners = null;
        if (geoHotel.getOfferCacheResponse() != null) {
            partners =
                    geoHotel.getOfferCacheResponse().getPricesList().stream().map(TPrice::getPartnerId).collect(Collectors.toSet());
        }
        fillPartners(pageParams, ocMeta, partners,  "Offers");
    }

    public static void fillSearchParams(Map<String, String> pageParams, OfferSearchParamsProvider p) {
        pageParams.put("paramDaysTillTrip",
                Integer.toString((int) ChronoUnit.DAYS.between(LocalDate.now(), p.getCheckinDate())));
        pageParams.put("paramNightCount",
                Integer.toString((int) ChronoUnit.DAYS.between(p.getCheckinDate(), p.getCheckoutDate())));
        pageParams.put("paramAdultCount",
                Integer.toString(p.getAdults()));
        pageParams.put("paramGuestCount",
                Integer.toString(p.getChildrenAges().size() + p.getAdults()));
        pageParams.put("paramChildCount",
                Integer.toString(p.getChildrenAges().size()));
    }

    public static void fillGeoParams(Map<String, String> pageParams, String domain, Integer geoId, GeoBase geoBase) {
        pageParams.put("geoId", integerToString(geoId));
        Integer cityId = null;
        Integer countryId = null;
        if (geoId != null) {
            cityId = GeoBaseHelpers.getRegionRoundTo(geoBase, geoId, GeoBaseHelpers.CITY_REGION_TYPE, domain);
            countryId = GeoBaseHelpers.getRegionRoundTo(geoBase, geoId, GeoBaseHelpers.COUNTRY_REGION_TYPE, domain);
        }
        pageParams.put("cityGeoId", integerToString(cityId));
        pageParams.put("countryGeoId", integerToString(countryId));
    }

    public static void fillSeoStaticPageParams(Map<String, String> pageParams, int geoId, EPageType pt) {
        pageParams.put("geoId", integerToString(geoId));
        String type = pt.toString();
        if (type.startsWith("PT_")) {
            type = type.substring(3);
        }
        pageParams.put("kind", type.toLowerCase());
    }

    public static void fillPartner(Map<String, String> pageParams, EPartnerId partnerId) {
        pageParams.put("partner", getPartnerName(partnerId));
    }

    public static void fillBoolean(Map<String, String> pageParams, String key, boolean v) {
        pageParams.put(key, boolToString(v));
    }
}
