package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Duration;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsOffersReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsOffersRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSharedFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSharedFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.RemoveFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.RemoveFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.ShareFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.ShareFavoriteHotelsRspV1;
import ru.yandex.travel.api.exceptions.FavoriteHotelLimitExceededException;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.hotels.HotelWithOffers;
import ru.yandex.travel.api.models.hotels.OfferCacheMetadata;
import ru.yandex.travel.api.models.hotels.OfferSearchParams;
import ru.yandex.travel.api.models.hotels.OfferSearchProgress;
import ru.yandex.travel.api.models.hotels.ShortHotelOffersInfo;
import ru.yandex.travel.api.models.hotels.interfaces.DebugOfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.ImageParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.RequestAttributionProvider;
import ru.yandex.travel.api.proto.hotels_portal.TFavoriteShareToken;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.tugc.TugcService;
import ru.yandex.travel.api.services.hotels.tugc.model.FavoriteGeoIdsRsp;
import ru.yandex.travel.api.services.hotels.tugc.model.FavoriteHotelsRsp;
import ru.yandex.travel.api.services.hotels.tugc.model.HotelFavoriteInfosRsp;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryStrategy;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.geosearch.model.OfferCacheRequestParams;
import ru.yandex.travel.hotels.offercache.api.ERespMode;
import ru.yandex.travel.hotels.offercache.api.TReadReq;

@Component
@RequiredArgsConstructor
@Slf4j
public class FavoritesService {
    private static final int MAX_GEOSEARCH_RETRIES = 5;
    private static final String LANG = "ru";
    private static final String ALL_CATEGORY_ID = "all";
    private static final String ALL_CATEGORY_NAME = "Все";
    private static final int MAX_LIMIT_PER_PAGE = 100;

    private final HotelsPortalProperties config;
    private final GeoSearchService geoSearch;
    private final AmenityService amenityService;
    private final GeoBase geoBase;
    private final HotelSlugService hotelSlugService;
    private final TugcService tugcService;
    private final Retry retryHelper;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final HotelImagesService hotelImagesService;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;

    public CompletableFuture<AddFavoriteHotelRspV1> addFavoriteHotel(AddFavoriteHotelReqV1 req,
                                                                     CommonHttpHeaders commonHttpHeaders,
                                                                     UserCredentials userCredentials) {
        return getHotelGeoId(commonHttpHeaders, req.getDomain(), req.getPermalink())
                .thenCompose(geoId -> tugcService.addFavoriteHotel(createLogId(), userCredentials, req.getPermalink(), geoId))
                .thenApply(tugcRsp -> {
                    if (tugcRsp.isHotelLimitExceed()) {
                        throw new FavoriteHotelLimitExceededException();
                    }
                    return new AddFavoriteHotelRspV1();
                });
    }

    public CompletableFuture<RemoveFavoriteHotelsRspV1> removeFavoriteHotels(RemoveFavoriteHotelsReqV1 req,
                                                                             UserCredentials userCredentials) {
        var logId = createLogId();
        TravelPreconditions.checkRequestArgument(req.getPermalink() != null || req.getCategoryId() != null, "Either permalink or category is required");
        TravelPreconditions.checkRequestArgument(req.getPermalink() == null || req.getCategoryId() == null, "Permalink and category can't be used simultaneously");
        if (req.getCategoryId() != null) {
            if (req.getCategoryId().equals(ALL_CATEGORY_ID)) {
                throw new TravelApiBadRequestException("Can't delete 'all' category");
            }
            return tugcService.removeFavoriteHotels(logId, userCredentials, getCategoryAsGeoId(req.getCategoryId()))
                    .thenApply(x -> new RemoveFavoriteHotelsRspV1());
        } else {
            return tugcService.removeFavoriteHotel(logId, userCredentials, req.getPermalink())
                    .thenApply(x -> new RemoveFavoriteHotelsRspV1());
        }
    }

    public CompletableFuture<ShareFavoriteHotelsRspV1> shareFavoriteHotels(ShareFavoriteHotelsReqV1 req,
                                                                           UserCredentials userCredentials) {
        var logId = createLogId();
        return getFavoriteHotelsByCategory(logId, userCredentials, req.getCategoryId())
                .thenApply(tugcRsp -> {
                    var tokenBuilder = TFavoriteShareToken.newBuilder();
                    tokenBuilder.addAllPermalink(tugcRsp.getHotels().stream().map(Permalink::asLong).collect(Collectors.toUnmodifiableList()));
                    tokenBuilder.setCheckInDate(req.getCheckinDate().toString());
                    tokenBuilder.setCheckOutDate(req.getCheckoutDate().toString());
                    tokenBuilder.setAges(Ages.build(req.getAdults(), Objects.requireNonNullElse(req.getChildrenAges(), List.of())).toString());
                    return new ShareFavoriteHotelsRspV1(HotelsPortalUtils.encodeProtoToString(tokenBuilder.build()));
                });
    }

    public CompletableFuture<GetFavoriteHotelsRspV1> getFavoriteHotels(GetFavoriteHotelsReqV1 req, CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        var logId = createLogId();
        var exps = new Experiments(req, config);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, commonHttpHeaders);

        var selectedCategoryId = getSelectedCategoryId(req.getCategoryId());

        var getFavoriteHotelsFuture = getFavoriteHotelsByCategory(logId, userCredentials, selectedCategoryId);
        var getCategoriesFuture = getFavoriteCategories(logId, userCredentials, selectedCategoryId);

        var getHotelsWithGeoSearchFuture = getFavoriteHotelsFuture
                .thenCompose(tugcRsp -> {
                    var hotels = applyPaging(tugcRsp.getHotels(), req.getOffset(), req.getLimit());
                    if (hotels.isEmpty()) {
                        return CompletableFuture.completedFuture(null);
                    }
                    return getHotelsFromGeoSearch(logId, req, req, req, commonHttpHeaders, userCredentials, req.getDomain(), false, hotels, null, uaasSearchExperiments);
                });

        return CompletableFuture.allOf(getHotelsWithGeoSearchFuture, getCategoriesFuture)
                .thenApply(ignored -> {
                    var geoRsp = getHotelsWithGeoSearchFuture.join();
                    var categories = getCategoriesFuture.join();
                    var tugcRsp = getFavoriteHotelsFuture.join();

                    Preconditions.checkState(categories.stream().anyMatch(x -> x.getId().equals(selectedCategoryId)));

                    var rsp = new GetFavoriteHotelsRspV1();
                    rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, exps));
                    rsp.setSelectedCategoryId(selectedCategoryId);
                    rsp.setCategories(categories);
                    rsp.setTotalHotelCount(tugcRsp.getHotels().size());

                    if (geoRsp == null || geoRsp.getOfferCacheResponseMetadata() == null) {
                        var today = LocalDate.now();
                        rsp.setOfferSearchParams(new OfferSearchParams(
                                Objects.requireNonNullElse(req.getCheckinDate(), today.plusDays(1)),
                                Objects.requireNonNullElse(req.getCheckoutDate(), today.plusDays(2)),
                                req.getAdults(),
                                req.getChildrenAges())
                        );
                        rsp.setOfferSearchProgress(new OfferSearchProgress(true, 0, 0, List.of(), List.of()));
                        rsp.setContext(buildPollingContext(""));
                        if (geoRsp != null) {
                            rsp.setHotels(processGeoSearchHotels(geoRsp, null, req, exps, uaasSearchExperiments, null));
                        } else {
                            rsp.setHotels(List.of());
                        }
                    } else {
                        var ocMeta = extractOcMeta(geoRsp, req);
                        rsp.setOfferSearchParams(ocMeta.getOfferSearchParams());
                        rsp.setOfferSearchProgress(ocMeta.getProgress());
                        rsp.setContext(buildPollingContext(geoRsp.getResponseInfo().getReqid()));
                        rsp.setHotels(processGeoSearchHotels(geoRsp, ocMeta, req, exps, uaasSearchExperiments, null));
                    }

                    log.info("{}: Polling context is {}", logId, rsp.getContext());
                    return rsp;
                });
    }

    public CompletableFuture<GetSharedFavoriteHotelsRspV1> getSharedFavoriteHotels(GetSharedFavoriteHotelsReqV1 req, CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        var logId = createLogId();
        var exps = new Experiments(req, config);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, commonHttpHeaders);

        TFavoriteShareToken token;
        try {
            token = TFavoriteShareToken.parseFrom(HotelsPortalUtils.decodeBytesFromString(req.getToken()));
        } catch (Exception e) {
            log.warn("{}: Invalid share token ('{}')", logId, req.getToken(), e);
            throw new TravelApiBadRequestException("Invalid share token");
        }
        TravelPreconditions.checkRequestArgument(token.hasCheckInDate(), "Invalid token (no checkin date)");
        TravelPreconditions.checkRequestArgument(token.hasCheckOutDate(), "Invalid token (no checkout date)");
        TravelPreconditions.checkRequestArgument(token.hasAges(), "Invalid token (no ages)");

        var permalinks = applyPaging(
                token.getPermalinkList().stream().map(Permalink::of).collect(Collectors.toUnmodifiableList()),
                req.getOffset(),
                req.getLimit());

        var offerSearchParams = new OfferSearchParams();
        offerSearchParams.setCheckinDate(LocalDate.parse(token.getCheckInDate()));
        offerSearchParams.setCheckoutDate(LocalDate.parse(token.getCheckOutDate()));
        offerSearchParams.setAdults(Ages.fromString(token.getAges()).getAdults());
        offerSearchParams.setChildrenAges(Ages.fromString(token.getAges()).getChildrenAges());
        if (req.getCheckinDate() != null || req.getCheckoutDate() != null) {
            offerSearchParams.setCheckinDate(req.getCheckinDate());
            offerSearchParams.setCheckoutDate(req.getCheckoutDate());
            offerSearchParams.setAdults(req.getAdults());
            offerSearchParams.setChildrenAges(req.getChildrenAges());
        }

        if (permalinks.isEmpty()) {
            var rsp = new GetSharedFavoriteHotelsRspV1();
            rsp.setOfferSearchParams(offerSearchParams);
            rsp.setOfferSearchProgress(new OfferSearchProgress(true, 0, 0, List.of(), List.of()));
            rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, exps));
            rsp.setContext(buildPollingContext(""));
            rsp.setHotels(List.of());
            rsp.setTotalHotelCount(token.getPermalinkCount());

            log.info("{}: Polling context is {}", logId, rsp.getContext());

            return CompletableFuture.completedFuture(rsp);
        }

        CompletableFuture<HotelFavoriteInfosRsp> tugcFuture;
        if (HotelsPortalUtils.needLikes(commonHttpHeaders, userCredentials)) {
            tugcFuture = tugcService.getHotelFavoriteInfos(logId, userCredentials, permalinks);
        } else {
            tugcFuture = CompletableFuture.completedFuture(null);
        }

        var getHotelsWithGeoSearchFuture = getHotelsFromGeoSearch(logId, offerSearchParams, req, req, commonHttpHeaders, userCredentials, req.getDomain(), false, permalinks, null, uaasSearchExperiments);

        return CompletableFuture.allOf(getHotelsWithGeoSearchFuture, tugcFuture)
                .handle((ignored, ignoredT) -> {
                    GeoSearchRsp geoRsp = getHotelsWithGeoSearchFuture.join();
                    HotelFavoriteInfosRsp tugcRsp = tugcFuture.join();

                    var rsp = new GetSharedFavoriteHotelsRspV1();
                    if (geoRsp == null || geoRsp.getOfferCacheResponseMetadata() == null) {
                        var today = LocalDate.now();
                        rsp.setOfferSearchParams(new OfferSearchParams(
                                Objects.requireNonNullElse(req.getCheckinDate(), today.plusDays(1)),
                                Objects.requireNonNullElse(req.getCheckoutDate(), today.plusDays(2)),
                                req.getAdults(),
                                req.getChildrenAges())
                        );
                        rsp.setOfferSearchProgress(new OfferSearchProgress(true, 0, 0, List.of(), List.of()));
                        rsp.setContext(buildPollingContext(""));
                        if (geoRsp != null) {
                            rsp.setContext(buildPollingContext(geoRsp.getResponseInfo().getReqid()));
                            rsp.setHotels(processGeoSearchHotels(geoRsp, null, req, exps, uaasSearchExperiments, null));
                        } else {
                            rsp.setContext(buildPollingContext(""));
                            rsp.setHotels(List.of());
                        }
                    } else {
                        rsp.setContext(buildPollingContext(geoRsp.getResponseInfo().getReqid()));
                        var ocMeta = extractOcMeta(geoRsp, req);
                        rsp.setOfferSearchParams(ocMeta.getOfferSearchParams());
                        rsp.setOfferSearchProgress(ocMeta.getProgress());
                        rsp.setContext(buildPollingContext(geoRsp.getResponseInfo().getReqid()));
                        rsp.setHotels(processGeoSearchHotels(geoRsp, ocMeta, req, exps, uaasSearchExperiments, null));
                    }
                    rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, exps));
                    rsp.setTotalHotelCount(token.getPermalinkCount());

                    log.info("{}: Polling context is {}", logId, rsp.getContext());

                    return rsp;
                });
    }

    public CompletableFuture<GetFavoriteHotelsOffersRspV1> getFavoriteHotelsOffers(GetFavoriteHotelsOffersReqV1 req, CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        TravelPreconditions.checkRequestArgument(!req.getPermalinks().isEmpty(), "At least one permalink is required");

        var logId = createLogId();
        var exps = new Experiments(req, config);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, commonHttpHeaders);
        var parentReqId = getParentReqIdFromContext(req.getContext());

        var permalinks = req.getPermalinks();

        return getHotelsFromGeoSearch(logId, req, req, req, commonHttpHeaders, userCredentials, req.getDomain(), true, permalinks, parentReqId, uaasSearchExperiments)
                .thenApply(geoRsp -> {
                    var rsp = new GetFavoriteHotelsOffersRspV1();
                    rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, exps));
                    rsp.setContext(req.getContext());
                    Map<Permalink, ShortHotelOffersInfo> geoOffers;
                    if (geoRsp == null || geoRsp.getOfferCacheResponseMetadata() == null) {
                        var today = LocalDate.now();
                        rsp.setOfferSearchParams(new OfferSearchParams(
                                Objects.requireNonNullElse(req.getCheckinDate(), today.plusDays(1)),
                                Objects.requireNonNullElse(req.getCheckoutDate(), today.plusDays(2)),
                                req.getAdults(),
                                req.getChildrenAges())
                        );
                        rsp.setOfferSearchProgress(new OfferSearchProgress(true, 0, 0, List.of(), List.of()));
                        geoOffers = new HashMap<>();
                    } else {
                        var ocMeta = extractOcMeta(geoRsp, req);
                        rsp.setOfferSearchParams(ocMeta.getOfferSearchParams());
                        rsp.setOfferSearchProgress(ocMeta.getProgress());
                        geoOffers = geoRsp.getHotels()
                                .stream()
                                .collect(Collectors.toUnmodifiableMap(
                                        GeoHotel::getPermalink,
                                        geoHotel -> HotelsPortalUtils.extractOnlyOffers(ocMeta, geoHotel,
                                                config.getHotelBadgePriorities(), config.getOfferBadgePriorities(), exps)));
                    }
                    var offers = permalinks.stream()
                            .collect(Collectors.toUnmodifiableMap(
                                    Function.identity(),
                                    permalink -> {
                                        var defaultShortHotelOffersInfo = new ShortHotelOffersInfo();
                                        defaultShortHotelOffersInfo.setOffers(List.of());
                                        defaultShortHotelOffersInfo.setSearchIsFinished(true);
                                        return geoOffers.getOrDefault(permalink, defaultShortHotelOffersInfo);
                                    }
                            ));
                    rsp.setOffers(offers);
                    log.info("{}: Polling context is {}", logId, rsp.getContext());

                    return rsp;
                });
    }


    private String createLogId() {
        return "favorites-" + UUID.randomUUID().toString().replace("-", "");
    }

    private CompletableFuture<Integer> getHotelGeoId(CommonHttpHeaders commonHttpHeaders,
                                                     String domain,
                                                     Permalink permalink) {
        var origin = HotelsPortalUtils.selectGeoOriginByDevice(commonHttpHeaders,
                GeoOriginEnum.FAVORITES_PAGE_DESKTOP, GeoOriginEnum.FAVORITES_PAGE_TOUCH);

        return geoSearch.querySingleHotel(GeoSearchReq.byPermalink(origin, permalink, commonHttpHeaders))
                .thenApply(geoRsp -> HotelsPortalUtils.determineHotelGeoId(geoBase, geoRsp.getHotel(), domain,
                        GeoBaseHelpers.WORLD_REGION));
    }

    private GetFavoriteHotelsRspV1.FavoriteCategory geoIdToCategory(Integer geoId, int hotelCount) {
        return new GetFavoriteHotelsRspV1.FavoriteCategory(
                geoId.toString(),
                geoBase.getLinguistics(geoId, LANG).getNominativeCase(),
                hotelCount
        );
    }

    private String getSelectedCategoryId(String categoryIdFromReq) {
        if (categoryIdFromReq == null || categoryIdFromReq.equals(ALL_CATEGORY_ID)) {
            return ALL_CATEGORY_ID;
        }
        try {
            return Integer.toString(Integer.parseInt(categoryIdFromReq));
        } catch (NumberFormatException e) {
            throw new TravelApiBadRequestException(String.format("Invalid category id %s", categoryIdFromReq));
        }
    }

    private CompletableFuture<List<GetFavoriteHotelsRspV1.FavoriteCategory>> getFavoriteCategories(String logId, UserCredentials userCredentials,
                                                                                                   String selectedCategoryId) {
        return tugcService.getFavoriteGeoIds(logId, userCredentials)
                .thenApply(rsp -> {
                    var categoryToCount = new HashMap<String, GetFavoriteHotelsRspV1.FavoriteCategory>();

                    var totalCount = rsp.getGeoIds().stream().collect(Collectors.summarizingInt(FavoriteGeoIdsRsp.GeoIdInfo::getPermalinkCount)).getSum();
                    categoryToCount.put(ALL_CATEGORY_ID, new GetFavoriteHotelsRspV1.FavoriteCategory(ALL_CATEGORY_ID, ALL_CATEGORY_NAME, totalCount));

                    for (var geoIdInfo: rsp.getGeoIds()) {
                        var category = geoIdToCategory(geoIdInfo.getGeoId(), geoIdInfo.getPermalinkCount());
                        categoryToCount.putIfAbsent(category.getId(), category);
                    }

                    if (!selectedCategoryId.equals(ALL_CATEGORY_ID)) {
                        categoryToCount.putIfAbsent(selectedCategoryId, geoIdToCategory(Integer.parseInt(selectedCategoryId), 0));
                    }

                    return categoryToCount.entrySet().stream()
                            .sorted((lhs, rhs) -> {
                                if (lhs.getKey().equals(rhs.getKey())) {
                                    return 0;
                                }
                                if (lhs.getKey().equals(ALL_CATEGORY_ID)) {
                                    return -1;
                                }
                                if (rhs.getKey().equals(ALL_CATEGORY_ID)) {
                                    return 1;
                                }
                                if (!lhs.getValue().getName().equals(rhs.getValue().getName())) {
                                    return lhs.getValue().getName().compareTo(rhs.getValue().getName());
                                }
                                return lhs.getKey().compareTo(rhs.getKey());
                            })
                            .map(Map.Entry::getValue)
                            .collect(Collectors.toUnmodifiableList());
                });
    }

    private List<Permalink> applyPaging(List<Permalink> permalinks, int offset, int limit) {
        TravelPreconditions.checkRequestArgument(limit <= MAX_LIMIT_PER_PAGE, "Limit should not exceed {}", MAX_LIMIT_PER_PAGE);
        return permalinks.stream().skip(offset).limit(limit).collect(Collectors.toUnmodifiableList());
    }

    private List<HotelWithOffers> processGeoSearchHotels(GeoSearchRsp geoRsp, OfferCacheMetadata ocMeta,
                                                         ImageParamsProvider req, Experiments exps,
                                                         UaasSearchExperiments uaasSearchExperiments, Map<Permalink, Boolean> favoriteHotelsInfo) {
        return geoRsp.getHotels()
                .stream()
                .map(geoHotel -> {
                    boolean isFavorite = favoriteHotelsInfo == null || favoriteHotelsInfo.get(geoHotel.getPermalink());

                    return HotelsPortalUtils.extractHotelWithOffers(amenityService, hotelSlugService,
                            hotelImagesService, ocMeta,
                            geoHotel, req, true, config.getHotelBadgePriorities(), config.getOfferBadgePriorities(),
                            null, exps, config.getHotelCategoriesToShow(),
                            false, false, isFavorite,
                            uaasSearchExperiments, imageWhitelistDataProvider, config.isShowNearestStationOnSnippet());
                })
                .collect(Collectors.toUnmodifiableList());
    }

    private CompletableFuture<GeoSearchRsp> getHotelsFromGeoSearch(String logId,
                                                                   OfferSearchParamsProvider offerSearchParams,
                                                                   RequestAttributionProvider requestAttributionProvider,
                                                                   DebugOfferSearchParamsProvider debugOfferSearchParamsProvider,
                                                                   CommonHttpHeaders commonHttpHeaders,
                                                                   UserCredentials userCredentials,
                                                                   String domain,
                                                                   boolean allowPastDates,
                                                                   List<Permalink> hotels,
                                                                   String parentReqId,
                                                                   UaasSearchExperiments uaasSearchExperiments) {
        return retryHelper.withRetry("FavoritesService::GetHotelsFromGeoSearch/" + logId,
                geoReq -> geoSearch.query(geoReq).thenApply(geoRsp -> {
                    String reqId = geoRsp.getResponseInfo() != null ? geoRsp.getResponseInfo().getReqid() : "---";
                    log.info("{}: GeoSearch request finished in {} + {} ms, ReqId: {}", logId,
                            geoRsp.getResponseTime().toMillis(),
                            geoRsp.getParseTime().toMillis(), reqId);
                    return geoRsp;
                }),
                buildGeoSearchReq(logId, offerSearchParams, requestAttributionProvider,
                        debugOfferSearchParamsProvider, commonHttpHeaders, userCredentials, domain, allowPastDates,
                        hotels, parentReqId, uaasSearchExperiments),
                new RetryStrategy<>() {
                    @Override
                    public boolean shouldRetryOnException(Exception ex) {
                        return false;
                    }

                    @Override
                    public void validateResult(GeoSearchRsp result) {}

                    @Override
                    public Duration getWaitDuration(int iteration, Exception ex, GeoSearchRsp result) {
                        return Duration.ZERO;
                    }

                    @Override
                    public int getNumRetries() {
                        return MAX_GEOSEARCH_RETRIES;
                    }
                });
    }

    private OfferCacheRequestParams generateOfferCacheRequestParams(RequestAttributionProvider requestAttribution,
                                                                    CommonHttpHeaders commonHttpHeaders,
                                                                    UserCredentials userCredentials,
                                                                    OfferSearchParamsProvider offerSearchParams,
                                                                    DebugOfferSearchParamsProvider debugOfferSearchParams,
                                                                    boolean allowPastDates,
                                                                    String logId,
                                                                    String domain,
                                                                    UaasSearchExperiments uaasSearchExperiments) {
        var exps = new Experiments(requestAttribution, config);
        var userRegion = HotelsPortalUtils.determineUserRegion(geoBase, commonHttpHeaders, requestAttribution);
        TReadReq.Builder ocReqBuilder = HotelsPortalUtils.prepareOfferCacheRequestParams(
                requestAttribution, commonHttpHeaders, userCredentials, userRegion, offerSearchParams, debugOfferSearchParams, null,
                null, true, exps, uaasSearchExperiments, "get_favorite_hotels/" + logId, true, config.isSortOffersUsingPlus());
        ocReqBuilder.setAllowPastDates(allowPastDates);
        ocReqBuilder.setRespMode(ERespMode.RM_MultiHotel);
        ocReqBuilder.setShowAllPansions(true);
        return OfferCacheRequestParams.build(ocReqBuilder);
    }

    private GeoSearchReq buildGeoSearchReq(String logId,
                                           OfferSearchParamsProvider offerSearchParams,
                                           RequestAttributionProvider requestAttribution,
                                           DebugOfferSearchParamsProvider debugOfferSearchParams,
                                           CommonHttpHeaders commonHttpHeaders,
                                           UserCredentials userCredentials,
                                           String domain,
                                           boolean allowPastDates,
                                           List<Permalink> hotels,
                                           String parentReqId,
                                           UaasSearchExperiments uaasSearchExperiments) {
        var origin = HotelsPortalUtils.selectGeoOriginByDevice(commonHttpHeaders,
                GeoOriginEnum.FAVORITES_PAGE_DESKTOP, GeoOriginEnum.FAVORITES_PAGE_TOUCH);

        var geoReq = GeoSearchReq.byPermalinks(origin, hotels, commonHttpHeaders);
        HotelsPortalUtils.setAdditionalGeoSearchParams(geoReq, config);
        geoReq.setLimit(hotels.size());
        geoReq.setOfferCacheRequestParams(generateOfferCacheRequestParams(requestAttribution, commonHttpHeaders,
                userCredentials, offerSearchParams, debugOfferSearchParams, allowPastDates, logId, domain,
                uaasSearchExperiments));
        geoReq.setUseProdOfferCache(debugOfferSearchParams.isDebugUseProdOffers() && config.isEnableDebugParams());
        geoReq.setIncludeOfferCache(true);
        geoReq.setIncludeSpravPhotos(true);
        geoReq.setIncludePhotos(true);
        geoReq.setIncludeRating(true);
        geoReq.setIncludeNearByStops(true);
        geoReq.setRequestLogId(logId);
        if (parentReqId != null) {
            geoReq.setParentReqId(parentReqId);
        }
        return geoReq;
    }

    private OfferCacheMetadata extractOcMeta(GeoSearchRsp geoRsp, DebugOfferSearchParamsProvider debugOfferSearchParamsProvider) {
        return OfferCacheMetadata.extractFromProto(geoRsp.getOfferCacheResponseMetadata(),
                debugOfferSearchParamsProvider.isDebugUseProdOffers() && config.isEnableDebugParams());
    }

    private String buildPollingContext(String reqId) {
        var pollingId = UUID.randomUUID().toString().replace("-", "");
        return String.format("%s~%s", pollingId, reqId);
    }

    private String getParentReqIdFromContext(String context) {
        int delimiterPos = context.indexOf('~');
        TravelPreconditions.checkRequestArgument(delimiterPos >= 0, "Invalid context");
        return context.substring(delimiterPos + 1);
    }

    private int getCategoryAsGeoId(String categoryId) {
        try {
            return Integer.parseInt(categoryId);
        } catch (NumberFormatException e) {
            throw new TravelApiBadRequestException(String.format("Invalid categoryId: %s", categoryId));
        }
    }

    private CompletableFuture<FavoriteHotelsRsp> getFavoriteHotelsByCategory(String logId, UserCredentials userCredentials, String categoryId) {
        if (categoryId.equals(ALL_CATEGORY_ID)) {
            return tugcService.getFavoriteHotels(logId, userCredentials);
        }
        return tugcService.getFavoriteHotels(logId, userCredentials, getCategoryAsGeoId(categoryId));
    }
}
