package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.micrometer.core.instrument.Tags;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.geobase6.LookupException;
import ru.yandex.geobase6.RegionHash;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.Linguistics;
import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.GeoSearchPollingResult;
import ru.yandex.travel.api.models.hotels.HotelFilterGroup;
import ru.yandex.travel.api.models.hotels.HotelWithOffers;
import ru.yandex.travel.api.models.hotels.OfferCacheMetadata;
import ru.yandex.travel.api.models.hotels.OfferSearchParamsWithBbox;
import ru.yandex.travel.api.models.hotels.SearchContext;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.proto.hotels_portal.TNavigationToken;
import ru.yandex.travel.api.proto.hotels_portal.TSearchContext;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.promo.CachedActivePromosService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.tugc.TugcService;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.PartnerConfigService;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.geosearch.model.OfferCacheRequestParams;
import ru.yandex.travel.hotels.offercache.api.ERespMode;
import ru.yandex.travel.hotels.offercache.api.TReadReq;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TPartner;

@Slf4j
public class GeoSearchPoller {
    private final static int MAX_GEOPAGES_TO_SCAN = 5;
    private final static int MAX_RETRIES = 3;
    private final static int MAX_EMPTY_ITERATIONS_FOR_SORT = 4;
    private final static int DEFAULT_SORT_STABILITY_MARGIN = 20;
    private final static String GEOSEARCH_HOST_EXP = "geosearch";
    private final static String GEOWHERE_KINDS_EXP = "relev_filter_gwkinds";
    private final static String GEOWHERE_EXP = "geowhere";
    private final static String REARR_EXP = "rearr";

    private final SearchHotelsReqV1 req;
    private final CommonHttpHeaders headers;
    private final HotelsPortalProperties config;
    private final GeoSearchService geoSearchService;
    private final PartnerConfigService partnerConfigService;
    private final HotelsFilteringService hotelsFilteringService;
    private final GeoBase geoBase;
    private final AmenityService amenityService;
    private final HotelSlugService hotelSlugService;
    private final Experiments experiments;
    private final GeoSearchPollingResult result;
    private final SortTypeRegistry sortTypeRegistry;
    private final Instant userRequestStart;
    private final UaasSearchExperiments uaasSearchExperiments;
    private final TugcService tugcService;
    private final UserCredentials userCredentials;
    private final HotelImagesService hotelImagesService;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;
    private String logId;
    private String addedTextPrefix = null;
    private SearchContext searchContext;
    private TSearchContext.Builder ctx; // shortcut to searchContext.proto
    private Instant pollingStartTime;
    private Instant pollingIterationStartTime;
    private Integer userRegion = null; // Местоположение пользователя
    private Integer effectiveSearchRegion = null; // Где будем искать
    private RegionHash effectiveSearchRegionHash = null;
    private Linguistics effectiveSearchRegionLinguistics = null;
    private SortTypeRegistry.SortType selectedSortType = null;
    private int sortStabilityMargin = DEFAULT_SORT_STABILITY_MARGIN;
    private Coordinates sortByDistanceOrigin = null;
    private boolean mirFilterEnabled = false;
    private final boolean isHotelsNearbyEnabled;
    private final CachedActivePromosService cachedActivePromosService;

    public GeoSearchPoller(SearchHotelsReqV1 req,
                           CommonHttpHeaders headers,
                           HotelsPortalProperties config,
                           GeoSearchService geoSearchService,
                           PartnerConfigService partnerConfigService,
                           HotelsFilteringService hotelsFilteringService,
                           GeoBase geoBase,
                           AmenityService amenityService,
                           HotelSlugService hotelSlugService,
                           HotelImagesService hotelImagesService,
                           Experiments experiments,
                           SortTypeRegistry sortTypeRegistry,
                           Instant userRequestStart,
                           UaasSearchExperiments uaasSearchExperiments,
                           TugcService tugcService,
                           UserCredentials userCredentials,
                           ImageWhitelistDataProvider imageWhitelistDataProvider,
                           CachedActivePromosService cachedActivePromosService) {
        this.req = req;
        this.headers = headers;
        this.config = config;
        this.geoSearchService = geoSearchService;
        this.partnerConfigService = partnerConfigService;
        this.hotelsFilteringService = hotelsFilteringService;
        this.geoBase = geoBase;
        this.amenityService = amenityService;
        this.hotelSlugService = hotelSlugService;
        this.hotelImagesService = hotelImagesService;
        this.experiments = experiments;
        this.sortTypeRegistry = sortTypeRegistry;
        this.userRequestStart = userRequestStart;
        this.uaasSearchExperiments = uaasSearchExperiments;
        this.tugcService = tugcService;
        this.userCredentials = userCredentials;
        this.imageWhitelistDataProvider = imageWhitelistDataProvider;
        this.cachedActivePromosService = cachedActivePromosService;

        this.result = new GeoSearchPollingResult();
        this.isHotelsNearbyEnabled = HotelsPortalUtils.isHotelsNearbyEnabled(headers);
    }

    public String getLogId() {
        return logId;
    }

    private void updateLogId() {
        String oldLogId = logId;
        if (searchContext != null && searchContext.getPollingId() != null) {
            logId = String.format("SearchHotels(%s)", searchContext.getPollingId());
        } else {
            logId = "SearchHotels";
        }
        if (oldLogId != null && !oldLogId.equals(logId)) {
            log.info("{}: Old log id was {}", logId, oldLogId);
        }
    }

    public void prepare() {
        pollingIterationStartTime = Instant.now();
        try {
            searchContext = SearchContext.parse(req.getContext());
            ctx = searchContext.protoBuilder();
        } catch (Exception exc) {
            throw new TravelApiBadRequestException("Failed to parse context: " + exc.getMessage(), exc);
        }
        updateLogId();

        // 0 < PageHotelCount < PricedHotelLimit <= TotalHotelLimit
        if (req.getPageHotelCount() <= 0) {
            throw new TravelApiBadRequestException("pageHotelCount should be > 0");
        }
        if (req.getPageHotelCount() >= req.getPricedHotelLimit()) {
            // иначе сломается корректное определение следующей страницы
            throw new TravelApiBadRequestException("pageHotelCount should be < pricedHotelLimit");
        }
        if (req.getPricedHotelLimit() > req.getTotalHotelLimit()) {
            throw new TravelApiBadRequestException("pricedHotelLimit should be <= totalHotelLimit");
        }
        TNavigationToken navigationToken = null;
        if (!Strings.isNullOrEmpty(req.getNavigationToken())) {
            try {
                navigationToken =
                        TNavigationToken.parseFrom(HotelsPortalUtils.decodeBytesFromString(req.getNavigationToken()));
            } catch (Exception exc) {
                throw new TravelApiBadRequestException("Failed to parse Navigation Token: " + exc.getMessage(), exc);
            }
        }
        var allowTopHotel = true;
        var allowSortOriginRedetection = false;
        // Check request parameters
        if (req.getPollIteration() == 0) {
            // Поллинг только что (ре)стартовал
            ctx.getPollingStateBuilder().clear();
            ctx.getPollingStateBuilder().setPollEpoch(req.getPollEpoch());
            ctx.getPollingStateBuilder().setStartedAtMillis(pollingIterationStartTime.toEpochMilli());
            ctx.getPollingStateBuilder().setGeoPageFirstAccess(true);
            ctx.getPollingStateBuilder().setHasBoyOffers(false);
            HotelSearchUtils.updateLegacyFilters(req);
            if (req.getStartSearchReason() == SearchHotelsReqV1.EStartSearchReasonType.queryByLocation) {
                HotelSearchUtils.clearNonPersistentFilters(req, hotelsFilteringService, userRequestStart, headers, cachedActivePromosService);
            }
            if (req.getStartSearchReason() == SearchHotelsReqV1.EStartSearchReasonType.mount) {
                HotelSearchUtils.clearInvalidFilters(req, hotelsFilteringService, userRequestStart, headers, cachedActivePromosService);
            }
            if (!List.of(
                    SearchHotelsReqV1.EStartSearchReasonType.mount,
                    SearchHotelsReqV1.EStartSearchReasonType.mapBounds,
                    SearchHotelsReqV1.EStartSearchReasonType.queryByLocation,
                    SearchHotelsReqV1.EStartSearchReasonType.queryWithSameGeo,
                    SearchHotelsReqV1.EStartSearchReasonType.navigationToken).contains(req.getStartSearchReason())) {
                allowTopHotel = false;
            }
            if (navigationToken == null) {
                // Начинаем с начала
                preparePollStartWithoutNavigation();
                allowSortOriginRedetection = true;
            } else {
                // Хотим определенную страницу
                if (req.getContext() == null) {
                    // без контекста => HOTELS-4861
                    preparePollStartWithNavigationWithoutContext(navigationToken);
                } else {
                    preparePollStartWithNavigation(navigationToken);
                }
            }
        } else {
            preparePollContinuation(navigationToken);
        }
        pollingStartTime = Instant.ofEpochMilli(ctx.getPollingState().getStartedAtMillis());
        result.setTopHotelSelected(allowTopHotel && req.getTopHotelSlug() != null);
        result.setOffset(ctx.getPollingState().getOffset());
        result.setBbox(req.getBbox());

        detectSelectedSort(allowSortOriginRedetection);

        userRegion = HotelsPortalUtils.determineUserRegion(geoBase, headers, req);

        if (req.getFilterAtoms() != null) {
            var mirAtoms = hotelsFilteringService.getFilterAtoms(HotelsPortalUtils.MIR_OFFERS_FILTER_ID,
                    userRequestStart, HotelSearchUtils.getForceFilterLayout(req), req.getGeoId(), headers);
            if (req.getFilterAtoms().stream().anyMatch(mirAtoms::contains)) {
                mirFilterEnabled = true;
            }
        }

        log.info("{}: SearchRegion {}, UserRegion {}, " +
                        "CheckIn {}, CheckOut {}, Adults {}, Children {}, " +
                        "GeoPage {}, Offset {}, FirstPermalink {}, " +
                        "StartedAt {}, Poll: {}/{}, Experiments: {}", logId,
                req.getGeoId(), userRegion,
                req.getCheckinDate(), req.getCheckoutDate(),
                req.getAdults(), req.getChildrenAges(),
                ctx.getPollingState().getGeoPage(),
                ctx.getPollingState().getOffset(),
                ctx.getPollingState().getFirstPermalink(),
                ctx.getPollingState().getStartedAtMillis(),
                req.getPollEpoch(), req.getPollIteration(),
                experiments
        );
    }

    public void determineEffectiveSearchRegion(Integer topHotelRegion) {
        if (req.getGeoId() != null) {
            effectiveSearchRegion = req.getGeoId();
        } else if (topHotelRegion != null) {
            effectiveSearchRegion = topHotelRegion;
        } else if (isHotelsNearbyEnabled && sortByDistanceOrigin != null && selectedSortType.isByDistance()) {
            var sortOriginGeoId = GeoBaseHelpers.getRegionIdByLocationOrNull(geoBase, sortByDistanceOrigin.getLat(), sortByDistanceOrigin.getLon());
            if (sortOriginGeoId != null) {
                var rounded = GeoBaseHelpers.getRegionRoundTo(geoBase, sortOriginGeoId, GeoBaseHelpers.CITY_REGION_TYPE, req.getDomain());
                effectiveSearchRegion = Objects.requireNonNullElse(rounded, sortOriginGeoId);
                log.info("{}: Have no geoId in request, sort by distance enabled, will use user location by coordinates: {}", logId, effectiveSearchRegion);
            } else {
                effectiveSearchRegion = userRegion;
                log.info("{}: Have no geoId in request, sort by distance enabled, will use userRegion: {}", logId, userRegion);
            }
        } else if (userRegion != null) {
            effectiveSearchRegion = userRegion;
            log.info("{}: Have no geoId in request, will use userRegion: {}", logId, userRegion);
        }

        if (effectiveSearchRegion != null) {
            try {
                calcEffectiveSearchRegionInfos();
                var regionType = effectiveSearchRegionHash.getAttr("type").getInteger();
                if (regionType == GeoBaseHelpers.HIDDEN_REGION_TYPE || regionType == GeoBaseHelpers.OTHER_REGION_TYPE) {
                    log.info("{}: Banned region (geoId={}, domain={})", logId, effectiveSearchRegion, req.getDomain());
                    effectiveSearchRegion = GeoBaseHelpers.MOSCOW_REGION;
                    calcEffectiveSearchRegionInfos();
                }
            } catch (LookupException e) {
                log.info("{}: Invalid region (geoId={}, domain={})", logId, effectiveSearchRegion, req.getDomain());
                effectiveSearchRegion = GeoBaseHelpers.MOSCOW_REGION;
                calcEffectiveSearchRegionInfos();
            }
        }

        log.info("{}: Got EffectiveSearchRegion: {}", logId, effectiveSearchRegion);
    }

    public void calcEffectiveSearchRegionInfos() {
        effectiveSearchRegionHash = geoBase.getRegionById(effectiveSearchRegion, req.getDomain());
        effectiveSearchRegionLinguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, effectiveSearchRegion, "ru");
    }

    @Data
    @AllArgsConstructor
    public static class PollingResults {
        private CompletableFuture<GeoSearchPollingResult> pollingResultFuture;
        private CompletableFuture<OfferSearchParamsWithBbox> offerSearchParamsWithBboxFuture;
    }

    public PollingResults run() {
        // Первый шаг в обработке запроса - определение инфы про текст
        var determineEffectiveSearchRegionFuture = determineTopHotelRegion().thenAccept(this::determineEffectiveSearchRegion);

        var determineBboxFuture = determineEffectiveSearchRegionFuture
                .thenCompose(ignored -> determineBbox());
        var knownOfferSearchParams = getKnownOfferSearchParams();

        var firstPollingReqFuture = determineBboxFuture
                .thenCompose((v) -> prepareAndSendGeoSearchRequest(true))
                .thenApply(geoReqAndRsp -> processGeoSearchResponse(geoReqAndRsp.req, geoReqAndRsp.rsp));

        var determineOfferSearchParamsFuture = determineOfferSearchParams(knownOfferSearchParams, firstPollingReqFuture);

        // TODO: Looks like a bug: when building next geoReqs we pass search params from initial req instead of detected search params

        var resultsFuture = firstPollingReqFuture
                .thenCompose(this::doGeoSearchRequestsTillMadeResponse)
                .thenCompose(voidResult -> enrichHotelsWithLikes())
                .thenApply(voidResult -> CompletableFuture.completedFuture(reply()))
                .exceptionally(t -> {
                    log.error("{}: Unable to complete search", logId, t);
                    return CompletableFuture.failedFuture(t);
                })
                .thenCompose(x -> x);

        var offerSearchParamsWithBboxFuture = CompletableFuture.allOf(determineOfferSearchParamsFuture, determineBboxFuture)
                .thenApply(x -> new OfferSearchParamsWithBbox(determineOfferSearchParamsFuture.join(), determineBboxFuture.join()));

        return new PollingResults(resultsFuture, offerSearchParamsWithBboxFuture);
    }

    public CompletableFuture<Void> enrichHotelsWithLikes() {
        if (!HotelsPortalUtils.needLikes(headers, userCredentials)) {
            return CompletableFuture.completedFuture(null);
        }

        var permalinks = Stream
                .concat(result.getHeadHotels().stream(), result.getTailHotels().stream())
                .map(x -> x.getHotel().getPermalink())
                .distinct()
                .collect(Collectors.toUnmodifiableList());
        return tugcService.getHotelFavoriteInfos(logId, userCredentials, permalinks)
                .thenAccept(hotelFavoriteInfosRsp -> {
                    for (var hotelsList: List.of(result.getHeadHotels(), result.getTailHotels())) {
                        for (var hotel : hotelsList) {
                            hotel.getHotel().setIsFavorite(hotelFavoriteInfosRsp.getIsFavorite()
                                    .getOrDefault(hotel.getHotel().getPermalink(), false));
                        }
                    }
                })
                .exceptionally(exc -> {
                    log.error("TUGC request failed -> no likes", exc);
                    return null;
                });
    }

    public CompletableFuture<Integer> determineTopHotelRegion() {
        if (result.isTopHotelSelected() && req.getGeoId() == null) {
            GeoOriginEnum origin = HotelsPortalUtils.selectGeoOriginByDevice(headers,
                    GeoOriginEnum.SEARCH_HOTELS_PAGE_DESKTOP, GeoOriginEnum.SEARCH_HOTELS_PAGE_TOUCH);
            GeoSearchReq geoSearchReq = GeoSearchReq.byPermalink(origin, getTopHotelPermalink(), headers);

            return geoSearchService.querySingleHotel(geoSearchReq).thenApply(hotelRsp -> {
                var result = HotelsPortalUtils.determineHotelGeoId(geoBase, hotelRsp.getHotel(), req.getDomain(), GeoBaseHelpers.RUSSIA_REGION);
                log.info("{}: Detected search region for top hotel: {}", logId, result);
                return result;
            });
        }
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<BoundingBox> determineBbox() {
        if (result.getBbox() != null) {
            return CompletableFuture.completedFuture(result.getBbox());
        }
        if (req.getBbox() != null) {
            return CompletableFuture.completedFuture(req.getBbox());
        }
        Preconditions.checkState(ctx.getPollingState().getGeoPageFirstAccess(), "No Bbox on non-first geo page access");
        return prepareAndSendGeoSearchRequest(false)
                .thenApply(geoReqAndRsp -> {
                    result.setBbox(determineBbox(geoReqAndRsp.rsp));
                    log.info("{}: Got bbox: {}, re-request with it", logId, result.getBbox());
                    return result.getBbox();
                });
    }

    public OfferSearchParamsProvider getKnownOfferSearchParams() {
        if (req.getCheckinDate() != null && req.getCheckoutDate() != null && req.getAdults() != null) {
            return req;
        }
        Preconditions.checkState(ctx.getPollingState().getGeoPageFirstAccess(), "No OfferSearchParams on non-first geo page access");
        return null;
    }

    public CompletableFuture<OfferSearchParamsProvider> determineOfferSearchParams(OfferSearchParamsProvider knownOfferSearchParams, CompletableFuture<Boolean> firstPollingReqFuture) {
        if (knownOfferSearchParams != null) {
            return CompletableFuture.completedFuture(knownOfferSearchParams);
        }
        return firstPollingReqFuture.thenApply(ignored -> result.getOfferCacheMeta().getOfferSearchParams());
    }

    private void preparePollStartWithoutNavigation() {
        ctx.clearPages();
        ctx.getPollingStateBuilder().setOffset(0);
        ctx.getPollingStateBuilder().setGeoPage(0);
        ctx.getPollingStateBuilder().setLastGeoSearchPage(MAX_GEOPAGES_TO_SCAN);
        if (checkSearchParamsHash()) {
            searchContext.incrementPollingIdOnBboxChange();
        } else {
            searchContext.generateNewPollingId();
        }
        ctx.clearGeoSearchCtxStr();// грязный хак, иначе случается фигня, see GEOSEARCH-5871
        updateLogId();
        log.info("{}: Situation: Polling start without navigation", logId);
        // Gen увеличиваем для инвалидации всех старых токенов навигации
        ctx.setGen(ctx.getGen() + 1);
        // чтобы перевычислились в конце
        ctx.clearSearchParamsHash();
        ctx.clearActualGeoId();
        ctx.clearLastGeoSearchPage();
        ctx.clearTotalPricedHotelsCountAmongAllPages();
        searchContext.clearOriginalBbox();
    }

    private void preparePollStartWithNavigation(TNavigationToken navigationToken) {
        validateNotInitialRequest();
        validateNavigTokenHash(navigationToken);
        if (navigationToken.getGen() != ctx.getGen()) {
            log.info("{}: Gen in navig token {} is different from context gen {} so ignore navig token", logId,
                    navigationToken.getGen(), ctx.getGen());
            // Simply ignore navig token
            preparePollStartWithoutNavigation();
            return;
        }
        if (checkOriginalBboxDiffer()) {
            // Simply ignore navig token
            preparePollStartWithoutNavigation();
            return;
        }
        searchContext.incrementPollingIdOnNavigation();
        updateLogId();
        log.info("{}: Situation: Polling start with navigation and ctx to offset {}", logId,
                navigationToken.getOffset());
        // Начинаем поллинг какой-то страницы
        List<TSearchContext.TPageInfo> pagesToPreserve = new ArrayList<>();
        TSearchContext.TPageInfo currentPage = null;
        for (TSearchContext.TPageInfo pageInfo : ctx.getPagesList()) {
            if (pageInfo.getOffset() < navigationToken.getOffset()) {
                pagesToPreserve.add(pageInfo);
            } else if (pageInfo.getOffset() == navigationToken.getOffset()) {
                currentPage = pageInfo;
            }
        }
        if (currentPage == null) {
            throw new TravelApiBadRequestException(String.format("Navigation token points to unknown offset %d",
                    navigationToken.getOffset()));
        }
        // Удаляем все страницы правее текущей, и текущую
        ctx.clearPages();
        ctx.addAllPages(pagesToPreserve);
        // И навигируемся на текущую
        ctx.getPollingStateBuilder().setOffset(currentPage.getOffset());
        ctx.getPollingStateBuilder().setGeoPage(currentPage.getFirstGeoPage());
        ctx.getPollingStateBuilder().setLastGeoSearchPage(currentPage.getFirstGeoPage() + MAX_GEOPAGES_TO_SCAN);
        ctx.getPollingStateBuilder().setFirstPermalink(currentPage.getFirstPermalink());
    }

    private void preparePollStartWithNavigationWithoutContext(TNavigationToken navigationToken) {
        ctx.getPollingStateBuilder().setOffset(navigationToken.getOffset());
        ctx.getPollingStateBuilder().setHotelsToSkip(navigationToken.getOffset());
        ctx.getPollingStateBuilder().setGeoPage(0);
        ctx.getPollingStateBuilder().setLastGeoSearchPage(MAX_GEOPAGES_TO_SCAN * (1 + navigationToken.getOffset() / req.getPageHotelCount())); // Должен же быть лимит
        searchContext.generateNewPollingId();
        updateLogId();
        log.info("{}: Situation: Polling start with navigation, but w/o ctx to offset {}", logId,
                navigationToken.getOffset());
    }

    private void preparePollContinuation(TNavigationToken navigationToken) {
        log.info("{}: Situation: Polling continuation, iteration {}", logId, req.getPollIteration());
        validateNotInitialRequest();
        if (checkOriginalBboxDiffer()) {
            throw new TravelApiBadRequestException("Bbox changed too much");
        }
        if (searchContext.getPollingId() == null) {
            throw new TravelApiBadRequestException("PollIteration > 0, but context is not passed");
        }
        if (!ctx.hasPollingState()) {
            throw new TravelApiBadRequestException("PollIteration > 0, but context doesn't contain polling state - most " +
                    "likely, polling is finished");
        }
        if (navigationToken != null) {
            validateNavigTokenHash(navigationToken);
        }
        if (req.getPollEpoch() != ctx.getPollingState().getPollEpoch()) {
            throw new TravelApiBadRequestException(String.format("Poll epoch doesn't match: expected %d, got %d",
                    ctx.getPollingState().getPollEpoch(), req.getPollEpoch()));
        }
    }

    private boolean checkOriginalBboxDiffer() {
        if (req.getBbox() == null) {
            throw new TravelApiBadRequestException("No bbox in request");
        }
        BoundingBox originalBbox = searchContext.getOriginalBbox();
        if (originalBbox == null) {
            throw new TravelApiBadRequestException("No originalBbox in context");
        }
        double origSquare = originalBbox.square();
        double curSquare = req.getBbox().square();
        if (curSquare < 1e-16) {
            return origSquare > 1e-16;
        }
        double squareRatio = origSquare / curSquare;
        if (squareRatio < config.getBboxMoveMinSquareRatio() || squareRatio > config.getBboxMoveMaxSquareRatio()) {
            log.debug("{}: OrigBbox: {}, ReqBbox {}, SquareRatio too big: {}", logId, originalBbox, req.getBbox(),
                    squareRatio);
            return true;
        }
        double spanLon = Math.abs(req.getBbox().getLeftDown().getLon() - req.getBbox().getUpRight().getLon());
        double spanLat = Math.abs(req.getBbox().getLeftDown().getLat() - req.getBbox().getUpRight().getLat());
        if (spanLon < 1e-16 || spanLat < 1e-16) {
            log.debug("{}: ReqBbox {}, too small span: lon {}, lat {}", logId, req.getBbox(), spanLon, spanLat);
            return true;
        }
        Coordinates origCenter = originalBbox.center();
        Coordinates reqCenter = req.getBbox().center();
        double distanceLon = Math.abs(reqCenter.getLon() - origCenter.getLon());
        double distanceLat = Math.abs(reqCenter.getLat() - origCenter.getLat());
        double relDistanceLon = distanceLon / spanLon;
        double relDistanceLat = distanceLat / spanLat;
        log.debug("{}: OrigBbox: {}, ReqBbox {}, relDistLon: {}, relDistLat: {}", logId, originalBbox, req.getBbox(),
                relDistanceLon, relDistanceLat);
        return relDistanceLon > config.getBboxMoveMaxRelativeDistance() ||
                relDistanceLat > config.getBboxMoveMaxRelativeDistance();
    }

    private void validateNavigTokenHash(TNavigationToken navigationToken) {
        for (Integer expectedHash : ctx.getSearchParamsHashList()) {
            if (navigationToken.getSearchParamsHash() == expectedHash) {
                return;
            }
        }
        throw new TravelApiBadRequestException("Navigation token has wrong searchparams hash");
    }

    private void validateNotInitialRequest() {
        if (req.getCheckinDate() == null || req.getCheckoutDate() == null || req.getAdults() == null) {
            throw new TravelApiBadRequestException("OfferSearchParams (checkinDate, checkoutDate, adults) should be given" +
                    " for not initial request");
        }
        if (!checkSearchParamsHash()) {
            throw new TravelApiBadRequestException("Search params hash changed");
        }
    }

    private boolean checkSearchParamsHash() {
        int currentHash = HotelsPortalUtils.calcSearchParamsHash(req, req, null);
        for (Integer expectedHash : ctx.getSearchParamsHashList()) {
            if (currentHash == expectedHash) {
                return true;
            }
        }
        return false;
    }

    private void determineAddedTextPrefix() {
        addedTextPrefix = config.getSearchPrefix();
        boolean addGeo = effectiveSearchRegionLinguistics != null;
        if (addGeo && result.getBbox() != null && effectiveSearchRegionHash != null) {
            // Регион где ищем
            BoundingBox searchRegionBbox = HotelsPortalUtils.getBboxByRegion(effectiveSearchRegionHash);
            // Его видимая часть
            BoundingBox intersectionBbox = BoundingBox.intersection(searchRegionBbox, result.getBbox());
            double searchRegionSquare = searchRegionBbox.square();
            double intersectionSquare = intersectionBbox.square();
            if (intersectionSquare < 0.5d * searchRegionSquare) {
                // Если показываем слишком малую часть региона поиска - то не добавляем гео
                addGeo = false;
            }
        }
        if (addGeo) {
            String where = effectiveSearchRegionLinguistics.getPrepositionalCase();
            if (Strings.isNullOrEmpty(where)) {
                where = effectiveSearchRegionLinguistics.getNominativeCase();
            }
            var countryGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, effectiveSearchRegion, GeoBaseHelpers.COUNTRY_REGION_TYPE, req.getDomain());
            if (countryGeoId != null) {
                var countryLinguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, countryGeoId, "ru");
                where += " " + countryLinguistics.getNominativeCase();
            }
            addedTextPrefix += String.format("%s %s ", effectiveSearchRegionLinguistics.getPreposition(), where);
            log.info("{}: Will add where to request text, final text prefix is '{}'", logId, addedTextPrefix);
        }
    }

    @Data
    @AllArgsConstructor
    static class GeoReqAndRsp {
        GeoSearchReq req;
        GeoSearchRsp rsp;
    }

    private CompletableFuture<Void> doGeoSearchRequestsTillMadeResponse(boolean needMoreRequests) {
        if (!needMoreRequests) {
            return CompletableFuture.completedFuture(null);
        }
        return prepareAndSendGeoSearchRequest(true)
                .thenApply(geoReqAndRsp -> processGeoSearchResponse(geoReqAndRsp.req, geoReqAndRsp.rsp))
                .thenCompose(this::doGeoSearchRequestsTillMadeResponse);
    }

    private CompletableFuture<GeoReqAndRsp> prepareAndSendGeoSearchRequest(boolean usePreprocess) {
        GeoOriginEnum origin = HotelsPortalUtils.selectGeoOriginByDevice(headers,
                GeoOriginEnum.SEARCH_HOTELS_PAGE_DESKTOP, GeoOriginEnum.SEARCH_HOTELS_PAGE_TOUCH);
        GeoSearchReq geoReq;
        if (ctx.getPollingState().getGeoPageFirstAccess()) {
            geoReq = prepareFirstGeoPageAccessReq(origin);
        } else {
            geoReq = prepareSubsequentGeoPageAccessReq(origin);
        }
        prepareCommonGeoReqFields(geoReq);

        log.info("{}: Do geoSearch request Offset {}, Limit {}", logId, geoReq.getOffset(), geoReq.getLimit());

        if (usePreprocess) {
            return doGeoSearchRequestWithRetries(geoReq, MAX_RETRIES)
                    .thenApply(geoRsp -> new GeoReqAndRsp(geoReq, geoRsp));
        } else {
            return doGeoSearchRequestWithoutPreprocess(geoReq)
                    .thenApply(geoRsp -> new GeoReqAndRsp(geoReq, geoRsp));
        }
    }

    private CompletableFuture<GeoSearchRsp> doGeoSearchRequestWithRetries(GeoSearchReq geoReq, int retriesLeft) {
        return doGeoSearchRequestWithoutPreprocess(geoReq)
                .thenCompose(geoRsp -> {
                    if (preProcessResponse(geoRsp, retriesLeft > 0) || retriesLeft == 0) {
                        return CompletableFuture.completedFuture(geoRsp);
                    } else {
                        result.setTotalRetries(result.getTotalRetries() + 1);
                        return doGeoSearchRequestWithRetries(geoReq, retriesLeft - 1);
                    }
                });
    }

    private CompletableFuture<GeoSearchRsp> doGeoSearchRequestWithoutPreprocess(GeoSearchReq geoReq) {
        return geoSearchService.query(geoReq, experiments.getValue(GEOSEARCH_HOST_EXP))
                .thenApply(geoRsp -> {
                    log.info("{}: Current context is {}", logId, searchContext);
                    String reqId = geoRsp.getResponseInfo() != null ? geoRsp.getResponseInfo().getReqid() : "---";
                    log.info("{}: GeoSearch request finished in {} + {} ms, ReqId: {}", logId, geoRsp.getResponseTime().toMillis(),
                            geoRsp.getParseTime().toMillis(), reqId);
                    return geoRsp;
                });
    }

    private GeoSearchReq prepareFirstGeoPageAccessReq(GeoOriginEnum origin) {
        // при первом обращении к странице ГП делаем полноценный запрос
        determineAddedTextPrefix();
        var geoReq = GeoSearchReq.byText(origin, addedTextPrefix, headers);
        HotelsPortalUtils.setAdditionalGeoSearchParams(geoReq, config, req.getGeosearchParams());
        geoReq.setFilterCategories(config.getHotelCategoriesToSearch());
        if (effectiveSearchRegion != null) {
            geoReq.setLr(effectiveSearchRegion);
            if (experiments.isExp("4846") && effectiveSearchRegionLinguistics != null) {
                geoReq.setGeowhere(effectiveSearchRegionLinguistics.getNominativeCase());
            }
        }
        addGeowhereKindsToRequest(geoReq);
        maybeAddGeowhereToRequest(geoReq);
        maybeAddRearrToRequest(geoReq);
        maybeEnableSort(geoReq);
        addFixedTop(geoReq);

        if (result.getBbox() != null) {
            // Чтобы bbox был конкретный даже без контекста
            if (experiments.isExp("158")) {
                geoReq.setAutoscale(false);
            } else {
                geoReq.setRspn(true);
            }
            geoReq.setBoundingBox(result.getBbox().toString());
        }
        geoReq.setSupressTextCorrection(true);
        if (ctx.hasGeoSearchCtxStr()) {
            geoReq.setContext(ctx.getGeoSearchCtxStr());
        }
        geoReq.setOffset(ctx.getPollingState().getGeoPage() * req.getTotalHotelLimit());
        geoReq.setLimit(req.getTotalHotelLimit());

        return geoReq;
    }

    private GeoSearchReq prepareSubsequentGeoPageAccessReq(GeoOriginEnum origin) {
        // при последующих обращениях - опрашиваем только нужные пермалинки
        Preconditions.checkArgument(ctx.getPollingState().getPermalinksToPollCount() > 0, "No in progress " +
                "permalink");
        List<Permalink> permalinks = new ArrayList<>();
        for (Long permalink : ctx.getPollingState().getPermalinksToPollList()) {
            permalinks.add(Permalink.of(permalink));
        }
        var geoReq = GeoSearchReq.byPermalinks(origin, permalinks, headers);
        HotelsPortalUtils.setAdditionalGeoSearchParams(geoReq, config, req.getGeosearchParams());
        geoReq.setLimit(permalinks.size());
        return geoReq;
    }

    private void prepareCommonGeoReqFields(GeoSearchReq geoReq) {
        TReadReq.Builder ocReqBuilder = HotelsPortalUtils.prepareOfferCacheRequestParams(
                req, headers, userCredentials, userRegion, req, req, null,
                null,
                false, /* Dates are passed via GeoSearch filter */
                experiments,
                uaasSearchExperiments,
                "search_hotels/" + logId,
                true,
                config.isSortOffersUsingPlus());
        ocReqBuilder.setRespMode(ERespMode.RM_MultiHotel);
        ocReqBuilder.setShowAllPansions(true);
        ocReqBuilder.setCompactResponseForSearch(true);
        ocReqBuilder.setAdjustDefaultSubKeyForMirPromoAlways(mirFilterEnabled);
        addFiltersToGeoReq(geoReq, ocReqBuilder);

        geoReq.setOfferCacheRequestParams(OfferCacheRequestParams.build(ocReqBuilder));
        geoReq.setUseProdOfferCache(req.isDebugUseProdOffers() && config.isEnableDebugParams());
        geoReq.setIncludeOfferCache(true);
        geoReq.setIncludeSpravPhotos(true);
        geoReq.setIncludePhotos(true);
        geoReq.setIncludeRating(true);
        geoReq.setIncludeNearByStops(true);
        geoReq.setRequestLogId(logId);
        if (ctx.hasReqId()) {
            geoReq.setParentReqId(ctx.getReqId());
        }
        if (selectedSortType.isByDistance()) {
            geoReq.setUll(sortByDistanceOrigin.toString());
        }
    }

    private void addFiltersToGeoReq(GeoSearchReq geoReq, TReadReq.Builder ocReqBuilder) {
        List<String> enabledPartnerCodes = partnerConfigService.getAll().values().stream().map(TPartner::getCode).collect(Collectors.toUnmodifiableList());
        // Геопоиску все эти фильтры нужны только при первом запросе (byText).
        // Но часть из них нужна офферкешу: Price[From|To], Check[In|Out]Date, некоторые из атомов (например
        // hotel_breakfast_included, hotel_free_cancellation).
        // Поэтому для единообразия передаем их все каждый раз.
        geoReq.setFilterPriceFrom(req.getFilterPriceFrom());
        geoReq.setFilterPriceTo(req.getFilterPriceTo());
        geoReq.setFilterDateFrom(req.getCheckinDate());
        geoReq.setFilterDateTo(req.getCheckoutDate());
        List<HotelFilterGroup> filtersGroups = hotelsFilteringService.prepareFilters(req.getFilterAtoms(),
                userRequestStart,
                HotelSearchUtils.getForceFilterLayout(req), req.getGeoId(), headers);
        HotelsPortalUtils.addGeoSearchFilters(geoReq, filtersGroups
                .stream()
                .filter(group -> !group.getUniqueId().startsWith(HotelsFilteringService.FAKE_BUSINESS_ID_PREFIX))
                .filter(group -> !group.getUniqueId().equals(HotelsFilteringService.CATEGORY_FILTER_GROUP_ID))
                .collect(Collectors.toUnmodifiableList()));

        var partnerFilterGroup = filtersGroups.stream()
                .filter(group -> group.getUniqueId().equals(HotelsFilteringService.PARTNER_FILTER_FEATURE_ID))
                .findFirst();
        if (partnerFilterGroup.isPresent()) {
            if (partnerFilterGroup.get().getFilters().stream().anyMatch(x -> x.getListValue() == null)) {
                throw new IllegalStateException(String.format("Partner filter has non-list value (groupId=%s)",
                        partnerFilterGroup.get().getUniqueId()));
            }
            var partnerIds = partnerFilterGroup.get().getFilters().stream()
                    .flatMap(x -> x.getListValue().getValues().stream())
                    .map(x -> {
                        try {
                            return EPartnerId.valueOf(x);
                        } catch (IllegalArgumentException e) {
                            throw new TravelApiBadRequestException(String.format("Unknown partner in filter: %s", x));
                        }
                    })
                    .collect(Collectors.toUnmodifiableList());
            enabledPartnerCodes = partnerIds.stream()
                    .map(partnerConfigService::getByKey)
                    .map(TPartner::getCode)
                    .filter(enabledPartnerCodes::contains)
                    .collect(Collectors.toUnmodifiableList());
            ocReqBuilder.addAllFilterPartnerIdAfterBoYSelection(partnerIds.stream().map(EPartnerId::getNumber).collect(Collectors.toUnmodifiableList()));
        }
        var yandexOfferFilterGroup = filtersGroups.stream()
                .filter(group -> group.getUniqueId().equals(HotelsFilteringService.YANDEX_OFFERS_FILTER_BUSINESS_ID))
                .findFirst();
        if (yandexOfferFilterGroup.isPresent()) {
            enabledPartnerCodes = partnerConfigService.getAll().values().stream()
                    .filter(TPartner::getIsBoY)
                    .map(TPartner::getCode)
                    .filter(enabledPartnerCodes::contains)
                    .collect(Collectors.toUnmodifiableList());
            ocReqBuilder.setBumpMirOffers(true); // TRAVELBACK-1308
            ocReqBuilder.setBumpBoYOffers(true);
            ocReqBuilder.setFilterRequireBoYOffer(true);
        }

        var mirOfferFilterGroup = filtersGroups.stream()
                .filter(group -> group.getUniqueId().equals(HotelsFilteringService.MIR_OFFERS_FILTER_BUSINESS_ID))
                .findFirst();
        if (mirOfferFilterGroup.isPresent()) { // TRAVELBACK-1308
            enabledPartnerCodes = partnerConfigService.getAll().values().stream()
                    .filter(TPartner::getIsBoY)
                    .map(TPartner::getCode)
                    .filter(enabledPartnerCodes::contains)
                    .collect(Collectors.toUnmodifiableList()); // only boy has cashback
            ocReqBuilder.setBumpMirOffers(true);
            ocReqBuilder.setFilterRequireMirOffer(true);
            result.setHasMirFilter(true);
        }

        var categoryFilterGroup = filtersGroups.stream()
                .filter(group -> group.getUniqueId().equals(HotelsFilteringService.CATEGORY_FILTER_GROUP_ID))
                .findFirst();
        if (categoryFilterGroup.isPresent()) {
            var allowedCategories = categoryFilterGroup.get().getFilters()
                    .stream()
                    .flatMap(x -> x.getListValue().getValues().stream())
                    .collect(Collectors.toUnmodifiableSet());
            if (geoReq.getFilterCategories() != null) {
                geoReq.setFilterCategories(geoReq.getFilterCategories()
                        .stream()
                        .filter(allowedCategories::contains)
                        .collect(Collectors.toUnmodifiableList()));
            } else {
                geoReq.setFilterCategories(allowedCategories.stream().collect(Collectors.toUnmodifiableList()));
            }
        }

        geoReq.setFilterHotelProviders(enabledPartnerCodes);
    }

    private boolean processGeoSearchResponse(GeoSearchReq geoReq, GeoSearchRsp geoRsp) {
        if (ctx.getPollingState().getGeoPageFirstAccess() && selectedSortType.isByPrice() && !isSortedResponseReadyToShow(geoRsp)) {
            if (req.getPollIteration() < MAX_EMPTY_ITERATIONS_FOR_SORT) {
                log.info("{}: Waiting for sorted results at same geopage {}", logId,
                        ctx.getPollingState().getGeoPage());
                return false;
            } else {
                log.warn("{}: Proceeding with not ready sorted response due to exceeded empty iterations count", logId);
            }
        }

        processGeoHotels(findGeoHotels(geoRsp), geoReq.getFilterCategories());

        boolean doNextGeoReq = false;
        if (!result.isSeenNotFinishedHotels()) {
            boolean setTotalPricedHotels = false;
            if (geoRsp.isNotImplementedError()) {
                result.setPollingFinished(true);
                result.setPollingFinishReason(HotelSearchStats.PollingFinishReason.GEOSEARCH_REPLIED_NOT_IMPLEMENTED);
                setTotalPricedHotels = true;
                log.warn("{}: Polling finished because geosearch replied 'NotImplemented'", logId);
            } else if (result.getOfferCacheMeta().isFake()) {
                result.setPollingFinished(true);
                result.setPollingFinishReason(HotelSearchStats.PollingFinishReason.NO_OFFERCACHE_RESPONSE);
                setTotalPricedHotels = true;
                log.warn("{}: Polling finished because no offerCache response", logId);
            } else if (ctx.getPollingState().getSentPricedHotelCountTotal() >= req.getPricedHotelLimit()) {
                // Совсем конец
                result.setPollingFinished(true);
                result.setPollingFinishReason(HotelSearchStats.PollingFinishReason.ENOUGH_HOTELS_COLLECTED);
                setTotalPricedHotels = false;
                log.info("{}: Polling finished because enough hotels collected", logId);
            } else if (ctx.getPollingState().hasLastGeoSearchPage() &&
                    (ctx.getPollingState().getGeoPage() >= ctx.getPollingState().getLastGeoSearchPage())) {
                // далековато ушли
                result.setPollingFinished(true);
                result.setPollingFinishReason(HotelSearchStats.PollingFinishReason.TOO_MANY_GEOPAGES_SCANNED);
                setTotalPricedHotels = true;
                log.warn("{}: Polling finished because too much geo pages scanned", logId);
            } else if (ctx.hasLastGeoSearchPage() && ctx.getPollingState().getGeoPage() >= ctx.getLastGeoSearchPage()) {
                // отели у геопоиска кончились
                result.setPollingFinished(true);
                result.setPollingFinishReason(HotelSearchStats.PollingFinishReason.LAST_GEOPAGE_REACHED);
                setTotalPricedHotels = true;
                log.info("{}: Polling finished because the last geosearch page was reached", logId);
            }

            if (setTotalPricedHotels) {
                ctx.setTotalPricedHotelsCountAmongAllPages(ctx.getPollingState().getOffset() + ctx.getPollingState().getSentPricedHotelCountTotal());
                result.setTotalPricedHotelsCountAmongAllPages(ctx.getTotalPricedHotelsCountAmongAllPages());
            }

            if (result.isPollingFinished()) {
                result.setFinalGeoPage(ctx.getPollingState().getGeoPage());
                result.setHasBoyOffers(ctx.getPollingState().getHasBoyOffers());
                ctx.clearPollingState();
            } else {
                // проследуем на следующую страницу ГП
                ctx.getPollingStateBuilder().setGeoPage(ctx.getPollingStateBuilder().getGeoPage() + 1);
                ctx.getPollingStateBuilder().setGeoPageFirstAccess(true);
                ctx.getPollingStateBuilder().clearPermalinksToPoll();
                ctx.getPollingStateBuilder().clearInprogressMarks();
                ctx.getPollingStateBuilder().clearFirstPermalink();
                doNextGeoReq = true;
                log.info("{}: Polling continues to next geopage {}", logId, ctx.getPollingState().getGeoPage());
            }
        } else {
            // Продолжаем копать текущую страницу ГП
            ctx.getPollingStateBuilder().setGeoPageFirstAccess(false);
            log.info("{}: Polling continues at same geopage {}", logId, ctx.getPollingState().getGeoPage());
        }

        if (ctx.hasTotalPricedHotelsCountAmongAllPages()) {
            result.setTotalPricedHotelsCountAmongAllPages(ctx.getTotalPricedHotelsCountAmongAllPages());
        }

        return doNextGeoReq;
    }

    private GeoSearchPollingResult reply() {
        if (!searchContext.hasOriginalBbox()) {
            searchContext.setOriginalBbox(result.getBbox());
        }

        if (!ctx.hasActualGeoId()) {
            ctx.setActualGeoId(GeoBaseHelpers.MOSCOW_REGION);
        }
        if (req.getGeoId() == null) {
            result.setSearchGeoId(ctx.getActualGeoId());
        } else {
            result.setSearchGeoId(req.getGeoId());
        }
        result.setActualGeoId(ctx.getActualGeoId());

        if (ctx.getSearchParamsHashCount() == 0) {
            // оба варианта допустимы: поллинг с поправленным текстом и с непоправленным
            ctx.addSearchParamsHash(HotelsPortalUtils.calcSearchParamsHash(req,
                    result.getOfferCacheMeta().getOfferSearchParams(), result.getSearchGeoId()));
        }

        result.setPrevNavigToken(maybeGenerateNavigationToken(result.getOffset() - req.getPageHotelCount()));
        result.setCurrentNavigToken(maybeGenerateNavigationToken(result.getOffset()));
        result.setNextNavigToken(maybeGenerateNavigationToken(result.getOffset() + req.getPageHotelCount()));
        result.setContext(searchContext.serialize());
        result.setPollingId(searchContext.getPollingId());
        result.setTotalPollingTime(Duration.between(pollingStartTime, Instant.now()));
        result.setIterationTime(Duration.between(pollingIterationStartTime, Instant.now()));
        log.info("{}: Final context is {}", logId, searchContext);
        return result;
    }

    private boolean preProcessResponse(GeoSearchRsp geoRsp, boolean retryAvailable) {
        if (geoRsp.isNotImplementedError()) {
            log.info("{}: GeoSearch returned 'NotImplemented'", logId);
            result.setOfferCacheMeta(OfferCacheMetadata.createFake(req));
            return true;
        }
        if (geoRsp.getResponseMetadata() == null) {
            log.error("{}: failed, No ResponseMetadata in GeoSearch response", logId);
            throw new RuntimeException("No ResponseMetadata in GeoSearch response");
        }
        if (geoRsp.getResponseInfo() == null) {
            log.error("{}: failed, No ResponseInfo in GeoSearch response", logId);
            throw new RuntimeException("No ResponseInfo in GeoSearch response");
        }
        ctx.setReqId(geoRsp.getResponseInfo().getReqid());
        if (ctx.getPollingState().getGeoPageFirstAccess()) {
            var geoContext = geoRsp.getResponseInfo().getContext();
            HotelsPortalUtils.reportDistribution("hotels.search.contextSize", geoContext.length(), Tags.empty(),
                    50, 200, 400, 600, 800, 1000, 1200, 1400, 1600, 1800, 2000, 2500, 3000);
            if (geoContext.length() < 1024) {
                ctx.setGeoSearchCtxStr(geoContext);
            } else {
                HotelsPortalUtils.incrementCounter("hotels.search.contextDropped");
                log.warn("{}: dropping geosearch context, because it's too big ({} bytes). It was: {}", logId,
                        geoContext.length(), geoContext);
                ctx.clearGeoSearchCtxStr();
            }
            if (geoRsp.getResponseMetadata().hasCorrected() && geoRsp.getBusinessMetadata() != null) {
                String text = geoRsp.getBusinessMetadata().getRequest().getText();
                if (addedTextPrefix != null) {
                    if (text.startsWith(addedTextPrefix)) {
                        text = text.substring(addedTextPrefix.length());
                    }
                }
                ctx.getPollingStateBuilder().setCorrectedRequestText(text);
            } else {
                ctx.getPollingStateBuilder().clearCorrectedRequestText();
            }
            if (geoRsp.getHotels().size() < req.getTotalHotelLimit()) {
                result.setTotalPricedHotelsCountAmongAllPages(ctx.getPollingState().getGeoPage() * req.getTotalHotelLimit() + geoRsp.getHotels().size());
                ctx.setLastGeoSearchPage(ctx.getPollingState().getGeoPage());
            }
            if (isHotelsNearbyEnabled && selectedSortType.isByDistance()) {
                geoRsp.setHotels(geoRsp.getHotels()
                        .stream()
                        .sorted(Comparator.comparing(x ->
                                !x.getGeoObjectMetadata().hasDistance()
                                        ? Double.MAX_VALUE
                                        : x.getGeoObjectMetadata().getDistance().getValue()))
                        .collect(Collectors.toUnmodifiableList()));
            }
        }
        if (!ctx.hasActualGeoId()) {
            ctx.setActualGeoId(determineActualGeoId(geoRsp));
        }
        result.setGeoRespFoundHotelsCount(geoRsp.getResponseMetadata().getFound());
        if (geoRsp.getOfferCacheResponseMetadata() == null) {
            if (retryAvailable) {
                HotelsPortalUtils.incrementCounter("hotels.search.noOfferCacheMeta");
                log.info("{}: No OfferCache metadata, retry", logId);
                return false;
            }

            // Most likely, there are no hotels in response, that is okay
            log.info("{}: No OfferCache metadata, create fake one", logId);
            HotelsPortalUtils.incrementCounter("hotels.search.noOfferCacheMetaAfterRetries");
            result.setOfferCacheMeta(OfferCacheMetadata.createFake(req));
        } else {
            result.setOfferCacheMeta(OfferCacheMetadata.extractFromProto(geoRsp.getOfferCacheResponseMetadata(),
                    req.isDebugUseProdOffers() && config.isEnableDebugParams()));
        }
        return true;
    }

    private boolean isSortedResponseReadyToShow(GeoSearchRsp geoRsp) {
        Preconditions.checkState(selectedSortType.isByPrice());
        var stats = geoRsp.getTravelSortStats();
        if (stats == null) {
            log.error("{}: No travel sort stats for request in sort mode", logId);
            return false;
        }

        try {
            log.info("{}: TravelSortStats: {}", logId, new ObjectMapper().writeValueAsString(stats));
        } catch (JsonProcessingException e) {
            log.warn("{}: Failed to log TravelSortStats", logId, e);
        }

        if (stats.getBySnippets() == null || !stats.getBySnippets().isEnabled()) {
            log.info("{}: Using price from snippets for sort is disabled, so assuming response is ready", logId);
            return true;
        }

        var finishedPrefix = stats.getBySnippets().getRealtimeDataCompleteness().getPrefixFinishedHotelsSize();
        var importantPrefixLength = (ctx.getPagesCount() + 1) * req.getPageHotelCount();

        var readyToShow = finishedPrefix == stats.getBySnippets().getDocsCount()
                || finishedPrefix >= importantPrefixLength + sortStabilityMargin;

        log.info("{}: Prefix of finished hotels is {}, important prefix is {}, margin is {} (readyToShow={})", logId,
                finishedPrefix, importantPrefixLength, sortStabilityMargin, readyToShow);

        return readyToShow;
    }

    private List<GeoHotel> findGeoHotels(GeoSearchRsp geoRsp) {
        List<String> hotelsWithoutOfferCacheData = new ArrayList<>();
        List<GeoHotel> hotels;
        if (ctx.getPollingState().getGeoPageFirstAccess()) {
            List<GeoHotel> geoHotelsAll = new ArrayList<>();
            List<GeoHotel> geoHotelsAllSinceFirstPermalink = new ArrayList<>();
            for (GeoHotel geoHotel : geoRsp.getHotels()) {
                if (geoHotel.getPermalink() == null) {
                    continue;
                }
                if (geoHotel.getOfferCacheResponse() == null) {
                    hotelsWithoutOfferCacheData.add(geoHotel.getGeoObjectMetadata().getId());
                    // в случае первого обращения нас не интересуют такие отели
                    continue;
                }
                geoHotelsAll.add(geoHotel);
                if (ctx.getPollingState().hasFirstPermalink() && (!geoHotelsAllSinceFirstPermalink.isEmpty() ||
                        (geoHotel.getPermalink().asLong() == ctx.getPollingState().getFirstPermalink()))) {
                    geoHotelsAllSinceFirstPermalink.add(geoHotel);
                }
            }
            if (geoHotelsAllSinceFirstPermalink.isEmpty()) {
                if (ctx.getPollingState().hasFirstPermalink()) {
                    HotelsPortalUtils.incrementCounter("firstPermalinkNotFoundInGeoResp");
                    log.warn("{}: FirstPermalink {} not found", logId, ctx.getPollingState().getFirstPermalink());
                }
                hotels = geoHotelsAll;
            } else {
                hotels = geoHotelsAllSinceFirstPermalink;
            }
        } else {
            List<Long> notFoundHotels = new ArrayList<>();
            Map<Long, GeoHotel> geoHotelMap = new HashMap<>();
            for (GeoHotel geoHotel : geoRsp.getHotels()) {
                if (geoHotel.getPermalink() != null) {
                    geoHotelMap.putIfAbsent(geoHotel.getPermalink().asLong(), geoHotel);
                }
            }
            // preserve initial hotel order
            List<GeoHotel> geoHotels = new ArrayList<>();
            for (Long permalink : ctx.getPollingState().getPermalinksToPollList()) {
                GeoHotel geoHotel = geoHotelMap.get(permalink);
                if (geoHotel == null) {
                    notFoundHotels.add(permalink);
                } else {
                    if (geoHotel.getOfferCacheResponse() == null) {
                        hotelsWithoutOfferCacheData.add(geoHotel.getGeoObjectMetadata().getId());
                    }
                    geoHotels.add(geoHotel);
                }
            }
            hotels = geoHotels;
            if (!notFoundHotels.isEmpty()) {
                HotelsPortalUtils.incrementCounter("missingHotelsInGeoResp");
                log.warn("{}: Hotels are missing in geoRsp: {}", logId, notFoundHotels);
            }
        }
        if (!hotelsWithoutOfferCacheData.isEmpty()) {
            HotelsPortalUtils.incrementCounter("hotelsWithNoOcDataInGeoResp");
            log.error("{}: Failed to find OfferCache data for hotel(s): {}", logId, hotelsWithoutOfferCacheData);
        }
        return hotels;
    }

    private void processGeoHotels(List<GeoHotel> geoHotels, List<String> categoryIdsFromFilters) {
        Set<Permalink> inProgressPermalinks = new HashSet<>();
        for (int idx = 0; idx < ctx.getPollingState().getPermalinksToPollCount() && idx < ctx.getPollingState().getInprogressMarksCount(); ++idx) {
            if (ctx.getPollingState().getInprogressMarks(idx)) {
                inProgressPermalinks.add(Permalink.of(ctx.getPollingState().getPermalinksToPoll(idx)));
            }
        }
        ctx.getPollingStateBuilder().clearPermalinksToPoll();
        ctx.getPollingStateBuilder().clearInprogressMarks();
        boolean hasBoyOffers = ctx.getPollingState().getHasBoyOffers();
        for (GeoHotel geoHotel : geoHotels) {
            if (geoHotel.getGeoObjectMetadata() == null) {
                log.error("{}: Failed to find GeoObjectMetadata for hotel {}", logId,
                        geoHotel.getGeoObject().getName());
                continue;
            }
            boolean isTopHotel = result.isTopHotelSelected() && geoHotel.getPermalink().equals(getTopHotelPermalink());
            boolean isFinished = true;
            boolean hasPrices = false;
            if (geoHotel.getOfferCacheResponse() != null) {
                isFinished = geoHotel.getOfferCacheResponse().getIsFinished();
                hasPrices = geoHotel.getOfferCacheResponse().getPricesCount() > 0;
                hasBoyOffers = hasBoyOffers || geoHotel.getOfferCacheResponse().getHasBoyOffers();
            }
            Permalink permalink = geoHotel.getPermalink();

            // На следующей итерации нам понадобятся:
            if (isFinished) {
                if (result.isSeenNotFinishedHotels() && hasPrices) {
                    // Плюсики после первого вопросика
                    ctx.getPollingStateBuilder().addPermalinksToPoll(permalink.asLong());
                    ctx.getPollingStateBuilder().addInprogressMarks(false);
                }
            } else {
                // вопросики
                result.setSeenNotFinishedHotels(true);
                ctx.getPollingStateBuilder().addPermalinksToPoll(permalink.asLong());
                ctx.getPollingStateBuilder().addInprogressMarks(true);
            }

            boolean sendInHead = false;
            boolean sendInTail = false;
            if (!result.isSeenNotFinishedHotels() && (hasPrices || isTopHotel)) {
                // если это плюсик до первого вопросика - посылаем его в голове
                // top_hotel тоже считаем плюсиком, даже если цен нет
                sendInHead = true;
            } else {
                if (ctx.getPollingState().getGeoPageFirstAccess()) {
                    // при первом посещении всё остальное посылаем в хвосте
                    sendInTail = true;
                } else {
                    // при повторном - только законченные отели, которые были вопросиками
                    if (isFinished && inProgressPermalinks.contains(permalink)) {
                        sendInTail = true;
                    }
                }
            }

            if (!sendInHead && !sendInTail) {
                continue;
            }
            // На странице поиска отелей показываем только главные (сниппетные) amenities
            HotelWithOffers rspHotel = HotelsPortalUtils.extractHotelWithOffers(
                    amenityService, hotelSlugService, hotelImagesService,
                    result.getOfferCacheMeta(), geoHotel, req,
                    true, config.getHotelBadgePriorities(), config.getOfferBadgePriorities(),
                    categoryIdsFromFilters, experiments, config.getHotelCategoriesToShow(),
                    isHotelsNearbyEnabled && selectedSortType.isByDistance(),
                    isTopHotel, false, uaasSearchExperiments, imageWhitelistDataProvider,
                    config.isShowNearestStationOnSnippet());
            if (sendInHead) {
                int hotelAbsolutePos =
                        ctx.getPollingState().getOffset() - ctx.getPollingState().getHotelsToSkip() + result.getHeadHotels().size();
                if (ctx.getPollingState().getHotelsToSkip() > 0) {
                    ctx.getPollingStateBuilder().setHotelsToSkip(ctx.getPollingState().getHotelsToSkip() - 1);
                    result.getTailHotels().add(rspHotel);
                } else {
                    ctx.getPollingStateBuilder().setSentPricedHotelCountTotal(ctx.getPollingState().getSentPricedHotelCountTotal() + 1);
                    result.getHeadHotels().add(rspHotel);
                }
                if (hotelAbsolutePos % req.getPageHotelCount() == 0) {
                    // Формируем pages по первым отелям на каждой странице
                    TSearchContext.TPageInfo.Builder pageBuilder = TSearchContext.TPageInfo.newBuilder();
                    pageBuilder.setOffset(hotelAbsolutePos);
                    pageBuilder.setFirstGeoPage(ctx.getPollingState().getGeoPage());
                    pageBuilder.setFirstPermalink(rspHotel.getHotel().getPermalink().asLong());
                    ctx.addPages(pageBuilder);
                }
            } else {
                result.getTailHotels().add(rspHotel);
            }
        }
        ctx.getPollingStateBuilder().setHasBoyOffers(hasBoyOffers);
    }

    private BoundingBox determineBbox(GeoSearchRsp geoRsp) {
        if (req.getBbox() != null) {
            return req.getBbox();
        }
        if (config.isUseGeoBbox() && geoRsp.getResponseMetadata() != null && geoRsp.getResponseMetadata().hasBoundedBy()) {
            Coordinates leftDown = new Coordinates();
            leftDown.setLon(geoRsp.getResponseMetadata().getBoundedBy().getLowerCorner().getLon());
            leftDown.setLat(geoRsp.getResponseMetadata().getBoundedBy().getLowerCorner().getLat());
            Coordinates upRight = new Coordinates();
            upRight.setLon(geoRsp.getResponseMetadata().getBoundedBy().getUpperCorner().getLon());
            upRight.setLat(geoRsp.getResponseMetadata().getBoundedBy().getUpperCorner().getLat());
            return BoundingBox.of(leftDown, upRight);
        }
        BoundingBox bbox = null;
        if (isHotelsNearbyEnabled && selectedSortType.isByDistance()) {
            bbox = HotelsPortalUtils.extendOrCreateBbox(bbox, sortByDistanceOrigin);
        }
        for (GeoHotel geoHotel : geoRsp.getHotels()) {
            if (result.isTopHotelSelected() && geoHotel.getPermalink().equals(getTopHotelPermalink())) {
                continue;
            }
            Coordinates c = HotelsPortalUtils.extractHotelCoordinates(geoHotel);
            if (c == null) {
                continue;
            }
            bbox = HotelsPortalUtils.extendOrCreateBbox(bbox, c);
        }
        if (result.isTopHotelSelected()) {
            var topHotel = geoRsp.getHotels().stream().filter(x -> x.getPermalink().equals(getTopHotelPermalink())).findFirst();
            if (topHotel.isPresent()) {
                Coordinates c = HotelsPortalUtils.extractHotelCoordinates(topHotel.get());
                if (c != null && (bbox == null || bbox.extendByRelativeValue(0.5).contains(c))) { // 50% to each side
                    bbox = HotelsPortalUtils.extendOrCreateBbox(bbox, c);
                }
            }
        }
        if (bbox == null) {
            int geoId = Objects.requireNonNullElse(effectiveSearchRegion, GeoBaseHelpers.MOSCOW_REGION);
            bbox = HotelsPortalUtils.getBboxByGeoId(geoBase, geoId, req.getDomain());
        }
        // Bbox needs to be extended a bit
        return HotelsPortalUtils.extendBboxABit(bbox);
    }

    private int determineActualGeoId(GeoSearchRsp geoRsp) {
        if (req.getBbox() == null && effectiveSearchRegionHash != null && result.getBbox() != null) {
            BoundingBox effectiveSearchRegionBBox = HotelsPortalUtils.getBboxByRegion(effectiveSearchRegionHash);
            if (BoundingBox.intersection(effectiveSearchRegionBBox, result.getBbox()).square() > 0.01) {
                return effectiveSearchRegion;
            }
        }
        class RegionInfo {
            private int votes;
            private int level;
        }
        Map<Integer/*geoId*/, RegionInfo> regions = new HashMap<>();
        int cnt = 0;
        for (GeoHotel geoHotel : geoRsp.getHotels()) {
            Coordinates c = HotelsPortalUtils.extractHotelCoordinates(geoHotel);
            if (c == null) {
                continue;
            }
            ++cnt;
            int geoId = geoBase.getRegionIdByLocation(c.getLat(), c.getLon());
            int[] parents = geoBase.getParentsIds(geoId);
            for (int pos = 0; pos < parents.length; ++pos) {
                int parentGeoId = parents[pos];
                RegionInfo ri = regions.get(parentGeoId);
                if (ri == null) {
                    ri = new RegionInfo();
                    ri.level = parents.length - pos - 1;
                    regions.put(parentGeoId, ri);
                }
                ++ri.votes;
            }
        }
        Integer bestGeoId = null;
        RegionInfo bestRegion = null;
        // Find deepest region with given votes percentage
        for (Map.Entry<Integer, RegionInfo> entry : regions.entrySet()) {
            if (((entry.getValue().votes * 100 / cnt) > 60) &&
                    (bestGeoId == null || (entry.getValue().level > bestRegion.level))) {
                bestGeoId = entry.getKey();
                bestRegion = entry.getValue();
            }
        }
        if (bestGeoId != null) {
            // TODO counter for level
            return bestGeoId;
        }
        if (result.getBbox() != null) {
            Coordinates center = result.getBbox().center();
            int geoId = geoBase.getRegionIdByLocation(center.getLat(), center.getLon());
            if (geoId != GeoBaseHelpers.DELETED_REGION) {
                // Возьмем самый большой регион, входящий в bbox
                for (int parentGeoId : geoBase.getParentsIds(geoId)) {
                    BoundingBox bbox = HotelsPortalUtils.getBboxByGeoId(geoBase, parentGeoId, req.getDomain());
                    if (result.getBbox().contains(bbox)) {
                        geoId = parentGeoId;
                    } else {
                        break;
                    }
                }
                return geoId;
            }
        }
        return GeoBaseHelpers.WORLD_REGION; // insane default
    }

    private String maybeGenerateNavigationToken(int offset) {
        if (offset < 0 || ctx.getSearchParamsHashCount() == 0) {
            return null;
        }
        for (TSearchContext.TPageInfo pageInfo : ctx.getPagesList()) {
            if (pageInfo.getOffset() == offset) {
                TNavigationToken.Builder navigToken = TNavigationToken.newBuilder();
                navigToken.setOffset(offset);
                navigToken.setSearchParamsHash(ctx.getSearchParamsHash(0));
                navigToken.setGen(ctx.getGen());
                return HotelsPortalUtils.encodeProtoToString(navigToken.build());
            }
        }
        return null;
    }

    private void addGeowhereKindsToRequest(GeoSearchReq geoReq) {
        var geowhereKindsExpValue = experiments.getValue(GEOWHERE_KINDS_EXP);
        if (geowhereKindsExpValue != null) {
            if (geowhereKindsExpValue.equals("all")) {
                geoReq.setGeowhereKinds(Arrays.stream(GeoSearchReq.GeowhereKind.values()).collect(Collectors.toList()));
            } else {
                var geowhereKinds = Arrays.stream(geowhereKindsExpValue.split(";"))
                        .map(GeoSearchReq.GeowhereKind::getByNameOrValue)
                        .filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
                geoReq.setGeowhereKinds(geowhereKinds);
            }
        }
    }

    private void maybeAddGeowhereToRequest(GeoSearchReq geoReq) {
        var geowhere = experiments.getValue(GEOWHERE_EXP);
        if (geowhere != null) {
            geoReq.setGeowhere(geowhere);
        }
    }

    private void maybeAddRearrToRequest(GeoSearchReq geoReq) {
        var rearr = experiments.getValue(REARR_EXP);
        if (rearr != null) {
            geoReq.setRearr(Arrays.stream(rearr.split(";")).collect(Collectors.toList()));
        }
    }

    private void detectSelectedSort(boolean allowSortOriginRedetection) {
        var isSortByDistance = req.getSelectedSortId() != null && req.getSelectedSortId().equals(SortTypeRegistry.SORT_BY_DISTANCE_ID);
        if (req.getSelectedSortId() != null && (!isSortByDistance || isHotelsNearbyEnabled)) {
            if (isSortByDistance) {
                if (allowSortOriginRedetection) {
                    TravelPreconditions.checkRequestArgument(req.getSortOrigin() != null || req.getUserCoordinates() != null, "sortOrigin or userCoordinates required for selected_sort_id=%s on new search", req.getSelectedSortId());
                    sortByDistanceOrigin = Objects.requireNonNullElse(req.getUserCoordinates(), req.getSortOrigin());
                } else {
                    TravelPreconditions.checkRequestArgument(req.getSortOrigin() != null, "sortOrigin required for selected_sort_id=%s on search continuation", req.getSelectedSortId());
                    sortByDistanceOrigin = req.getSortOrigin();
                }
            }
            selectedSortType = sortTypeRegistry.getSortType(req.getSelectedSortId());
        } else {
            selectedSortType = sortTypeRegistry.getDefaultSortType();
        }
        Preconditions.checkState(selectedSortType != null);

        result.setSelectedSortId(selectedSortType.getId());
        result.setSortOrigin(sortByDistanceOrigin);
    }

    private void maybeEnableSort(GeoSearchReq geoReq) {
        if (selectedSortType.getGeoSortType() != null) {
            geoReq.setSortType(selectedSortType.getGeoSortType());
        }

        if (selectedSortType.isByPrice()) {
            geoReq.setIncludeSortStats(true);
            if (experiments.isExp("sort-ignore-factors")) {
                geoReq.setEnableSortPriceFromFactors(false);
            }
            if (experiments.isExp("sort-ignore-snippets")) {
                geoReq.setEnableSortPriceFromSnippets(false);
            }
            if (experiments.isExp("sort-fallback-to-factors")) {
                geoReq.setSortPriceFallbackToFactors(true);
            }
            var margin = experiments.getIntValue("sort-margin");
            if (margin != null) {
                sortStabilityMargin = margin;
            }
        }

        if (isHotelsNearbyEnabled) {
            if (selectedSortType.isByDistance()) {
                if (experiments.isExp("1569-no-disable-sort-hack") ||
                        req.getBbox() == null ||
                        req.getBbox().extendByRelativeValue(0.1).contains(sortByDistanceOrigin)) {
                    geoReq.setSort("distance");
                }
            }
        }
    }

    private void addFixedTop(GeoSearchReq geoReq) {
        if (result.isTopHotelSelected()) {
            geoReq.setFixedTop(List.of(getTopHotelPermalink()));
        }
    }

    private Permalink getTopHotelPermalink() {
        return HotelSearchUtils.getTopHotelPermalink(hotelSlugService, req);
    }
}
