package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.Timer;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.metrics.MetricsUtils;

@Data
@Slf4j
public class HotelSearchStats {
    public enum PollingFinishReason {
        NO_OFFERCACHE_RESPONSE,
        ENOUGH_HOTELS_COLLECTED,
        TOO_MANY_GEOPAGES_SCANNED,
        LAST_GEOPAGE_REACHED,
        GEOSEARCH_REPLIED_NOT_IMPLEMENTED,
    }

    HotelSearchStats() {
    }

    void report() {
        try {
            reportToLog();
            reportToMetrics();
        } catch (Exception exc) {
            log.error("{}: Unable to report stats", logId, exc);
        }
    }

    private void reportToLog() {
        var statsMapping = new HashMap<String, Object>() {
            {
                put("pollingFinished", pollingFinished);
                put("pollingFinishReason", pollingFinishReason);
                put("totalPollingTime", totalPollingTime);
                put("iterationTime", iterationTime);
                put("epoch", epoch);
                put("iteration", iteration);
                put("totalRetries", totalRetries);
                put("frontPage", frontPage);
                put("headHotelsCount", headHotelsCount);
                put("tailHotelsCount", tailHotelsCount);
                put("foundHotelCount", foundHotelCount);
                put("totalPricedHotelsOnAllGeoPages", totalPricedHotelsOnAllGeoPages);
                put("totalHotelsOnAllGeoPages", totalHotelsOnAllGeoPages);
                put("hasTrueResultSize", hasTrueResultSize);
            }
        };
        log.info("{}: HotelSearchStats: {}", logId, statsMapping);
        try {
            log.info("{}: HotelSearchStatsEvent: {}", logId, new ObjectMapper().writeValueAsString(statsMapping));
        } catch (JsonProcessingException e) {
            // ignored
        }
    }

    private void reportToMetrics() {
        incrementCounter("hotels.search.calls");
        incrementCounter("hotels.search.pollingFinished", pollingFinished);
        if (pollingFinished) {
            incrementCounter("hotels.search.hasTrueResultSizeOnFinishedPolling", hasTrueResultSize);
            incrementCounter("hotels.search.pollingFinishReason", Tags.of("reason", pollingFinishReason.toString()));

            reportMediumTime("hotels.search.totalPollingTime", totalPollingTime);
        } else {
            incrementCounter("hotels.search.pollingFinishReason", Tags.of("reason", "NOT_FINISHED"));

            reportSmallDistribution("hotels.search.headHotelsCount", headHotelsCount);
            reportSmallDistribution("hotels.search.tailHotelsCount", tailHotelsCount);
        }

        if (iteration == 0) {
            incrementCounter("hotels.search.hasTrueResultSizeOnIter0", hasTrueResultSize);
            incrementCounter("hotels.search.hasFilters", hasFilters);

            reportSmallDistribution("hotels.search.epoch", epoch);
            reportSmallDistribution("hotels.search.frontPage", frontPage);
        }

        reportSmallTime("hotels.search.iterationTime", iterationTime);

        reportSmallDistribution("hotels.search.iteration", iteration);
        reportSmallDistribution("hotels.search.totalRetries", totalRetries);

        for (var pollingFinishedTag : new String[]{String.valueOf(pollingFinished), "_ALL_"}) {
            var currTags = Tags.of("pollingFinished", pollingFinishedTag);
            reportLargeDistribution("hotels.search.foundHotelCount", foundHotelCount, currTags);
        }

        incrementCounter("hotels.search.emptyResponse", totalPricedHotelsOnAllGeoPages != null && totalPricedHotelsOnAllGeoPages == 0);

        if (totalHotelsOnAllGeoPages != null && totalPricedHotelsOnAllGeoPages != null && totalHotelsOnAllGeoPages != 0) {
            reportSmallDistribution("hotels.search.totalPricedHotelsOnAllGeoPages", totalPricedHotelsOnAllGeoPages);
            reportSmallDistribution("hotels.search.totalHotelsOnAllGeoPages", totalHotelsOnAllGeoPages);
            reportPctDistribution("hotels.search.pricedHotelsOnAllGeoPagesPct", 100.0 * totalPricedHotelsOnAllGeoPages / totalHotelsOnAllGeoPages);
        }
    }

    private void reportSmallTime(String name, Duration duration) {
        reportSmallTime(name, duration, Tags.empty());
    }

    private void reportSmallTime(String name, Duration duration, Tags tags) {
        if (duration == null) {
            log.error("Duration is null for {}", name);
            return;
        }
        Timer.builder(name)
                .serviceLevelObjectives(Duration.ofMillis(5),
                        Duration.ofMillis(10),
                        Duration.ofMillis(25),
                        Duration.ofMillis(50),
                        Duration.ofMillis(75),
                        Duration.ofMillis(100),
                        Duration.ofMillis(150),
                        Duration.ofMillis(200),
                        Duration.ofMillis(300),
                        Duration.ofMillis(400),
                        Duration.ofMillis(500),
                        Duration.ofMillis(1000))
                .tags(tags)
                .publishPercentiles(MetricsUtils.higherPercentiles())
                .register(Metrics.globalRegistry)
                .record(duration);
    }

    private void reportMediumTime(String name, Duration duration) {
        if (duration == null) {
            log.error("Duration is null for {}", name);
            return;
        }
        Timer.builder(name)
                .serviceLevelObjectives(MetricsUtils.mediumDurationSla())
                .publishPercentiles(MetricsUtils.higherPercentiles())
                .register(Metrics.globalRegistry)
                .record(duration);
    }

    private void reportSmallDistribution(String name, double value) {
        reportDistribution(name, value, Tags.empty(), 1, 2, 3, 4, 5, 10, 15, 20, 25, 30, 50, 100, 150, 200, 300, 500, 1000);
    }

    private void reportMediumDistribution(String name, double value) {
        reportDistribution(name, value, Tags.empty(), 50, 200, 400, 600, 800, 1000, 1200, 1400, 1600, 1800, 2000, 2500, 3000);
    }

    private void reportLargeDistribution(String name, double value, Tags tags) {
        reportDistribution(name, value, tags, 10, 20, 30, 40, 50, 100, 150, 200, 300, 500, 1000, 2000, 5000, 10000, 50000, 100000, 1000000);
    }

    private void reportPctDistribution(String name, double value, Tags tags) {
        reportDistribution(name, value, tags, 1, 5, 10, 15, 25, 50, 75, 85, 90, 99, 100);
    }

    private void reportPctDistribution(String name, double value) {
        reportPctDistribution(name, value, Tags.empty());
    }

    private void reportDistribution(String name, double value, Tags tags, long... sla) {
        DistributionSummary.builder(name)
                .serviceLevelObjectives(Arrays.stream(sla).asDoubleStream().toArray())
                .tags(tags)
                .register(Metrics.globalRegistry)
                .record(value);
    }

    private void incrementCounter(String name) {
        Counter.builder(name)
                .register(Metrics.globalRegistry)
                .increment();
    }

    private void incrementCounter(String name, Tags tags) {
        Counter.builder(name)
                .tags(tags)
                .register(Metrics.globalRegistry)
                .increment();
    }

    private void incrementCounter(String name, boolean value) {
        incrementCounter(name, Tags.of("value", String.valueOf(value)));
    }

    String logId;

    private boolean pollingFinished;
    private boolean hasTrueResultSize;
    private boolean hasFilters;
    private PollingFinishReason pollingFinishReason;
    private Duration totalPollingTime;
    private Duration iterationTime;
    private int epoch;
    private int iteration;
    private int totalRetries;
    private int frontPage;
    private int headHotelsCount;
    private int tailHotelsCount;
    private int foundHotelCount;
    private Integer totalPricedHotelsOnAllGeoPages;
    private Integer totalHotelsOnAllGeoPages;
}
