package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.protobuf.Int64Value;
import com.google.protobuf.UInt32Value;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;

import ru.yandex.geobase6.LookupException;
import ru.yandex.geobase6.RegionHash;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.models.Linguistics;
import ru.yandex.travel.api.models.Region;
import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.Hotel;
import ru.yandex.travel.api.models.hotels.HotelFilter;
import ru.yandex.travel.api.models.hotels.HotelOffer;
import ru.yandex.travel.api.models.hotels.OfferSearchParamsWithBbox;
import ru.yandex.travel.api.models.hotels.interfaces.HotelIdentifierProvider;
import ru.yandex.travel.api.models.hotels.interfaces.ImageParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.RequestAttributionProvider;
import ru.yandex.travel.api.models.hotels.interfaces.SearchFilterParamsProvider;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.geocounter.GeoCounterService;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterReq;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.promo.CachedActivePromosService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.tugc.model.HotelFavoriteInfosRsp;
import ru.yandex.travel.commons.experiments.KVExperiments;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.common.promo.mir.MirUtils;
import ru.yandex.travel.hotels.common.refunds.RefundType;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.offercache.api.TOCPansion;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.geocounter_service.ESortType;
import ru.yandex.travel.hotels.proto.geocounter_service.TBbox;
import ru.yandex.travel.hotels.proto.geocounter_service.TCoordinates;
import ru.yandex.travel.hotels.proto.geocounter_service.TCryptaInfo;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetHotelsRequest;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetHotelsResponse;
import ru.yandex.travel.hotels.proto.hotel_filters.THotelFilter;

@Slf4j
public class HotelSearchUtils {
    public static CompletableFuture<CountHotelsRspV1> doGeoCounterRequest(Logger log,
                                                                          String logId,
                                                                          Instant started,
                                                                          SearchHotelsReqV1 req,
                                                                          HotelsFilteringService hotelsFilteringService,
                                                                          UaasSearchExperiments uaasSearchExperiments,
                                                                          Experiments experiments,
                                                                          GeoCounterService geoCounter,
                                                                          OfferSearchParamsWithBbox offerSearchParamsWithBbox,
                                                                          CommonHttpHeaders commonHttpHeaders) {
        if (offerSearchParamsWithBbox == null) {
            log.info("{}: GeoCounter is not possible", logId);
            return CompletableFuture.completedFuture(null);
        }
        if (req.getDisableGeocounter()) {
            log.info("{}: GeoCounter is disabled for this request", logId);
            return CompletableFuture.completedFuture(null);
        }
        log.info("{}: Doing request to geoCounter", logId);
        GeoCounterReq gcReq = hotelsFilteringService.prepareGeoCounterReq(req,
                offerSearchParamsWithBbox.getOfferSearchParams(), offerSearchParamsWithBbox.getBbox(),
                experiments.isExp("4813"), started,
                HotelSearchUtils.getForceFilterLayout(req), req.getGeoId(), commonHttpHeaders);
        gcReq.setNameSuffix(logId);
        return geoCounter.getCounts(gcReq).thenApply(gcRsp -> {
            log.info("{}: Processed response from geoCounter in {} ms", logId, gcRsp.getResponseTime().toMillis());
            return hotelsFilteringService.composeCountHotelsRsp(offerSearchParamsWithBbox.getOfferSearchParams(),
                    req, gcRsp, started, HotelSearchUtils.getForceFilterLayout(req), commonHttpHeaders, experiments,
                    req.getSelectedSortId(), req.getSortOrigin(), req.getGeoId(), req.isOnlyCurrentGeoId());
        });
    }

    public static Region fillRegion(RegionsService regionsService, String domain, Integer geoId) {
        if (geoId != null) {
            try {
                return regionsService.getRegion(geoId, domain, "ru");
            } catch (LookupException e) {
                log.warn("Failed to get region with id={}", geoId);
            }
        }
        var region = new Region();
        region.setGeoId(GeoBaseHelpers.DELETED_REGION);
        region.setType(GeoBaseHelpers.HIDDEN_REGION_TYPE);
        var linguistics = new Linguistics();
        linguistics.setNominativeCase("Неизвестный регион");
        linguistics.setGenitiveCase("Неизвестного региона");
        linguistics.setDativeCase("Неизвестному региону");
        linguistics.setPrepositionalCase("Неизвестном регионе");
        linguistics.setPreposition("в");
        linguistics.setAccusativeCase("Неизвестный регион");
        linguistics.setInstrumentalCase("Неизвестным регионом");
        region.setLinguistics(linguistics);
        return region;
    }

    public static SearchHotelsRspV1.ShortRegion fillShortRegion(GeoBase geoBase, String domain, Integer geoId) {
        SearchHotelsRspV1.ShortRegion result = new SearchHotelsRspV1.ShortRegion();
        String unknownRegionName = "Неизвестный регион";
        if (geoId == null) {
            result.setName(unknownRegionName);
        } else {
            List<String> parts = new ArrayList<>(4);
            int currentGeoId = geoId;
            while (true) {
                RegionHash hash;
                try {
                    hash = geoBase.getRegionById(currentGeoId, domain);
                } catch (LookupException e) {
                    log.warn("Failed to get region with id={}", currentGeoId);
                    parts.add(unknownRegionName);
                    break;
                }
                Integer type = hash.getAttr("type").getInteger();
                boolean isBigEnough = GeoBaseHelpers.regionIsBig(type);
                if (currentGeoId == geoId || isBigEnough) {
                    parts.add(hash.getString("name"));
                }
                if (isBigEnough || currentGeoId == GeoBaseHelpers.WORLD_REGION) {
                    break;
                }
                currentGeoId = geoBase.getParentId(currentGeoId, domain);
            }
            result.setName(String.join(", ", Lists.reverse(parts)));
        }
        return result;
    }

    public static SearchHotelsRspV1.SortInfo buildSortInfo(SortTypeRegistry sortTypeRegistry, String selectedSortId,
                                                           Coordinates sortOrigin) {
        var sortInfo = new SearchHotelsRspV1.SortInfo();
        sortInfo.setSelectedSortId(selectedSortId);
        sortInfo.setSortOrigin(sortOrigin != null ? sortOrigin.toString() : null);
        sortInfo.setAvailableSortTypeGroups(sortTypeRegistry.getSortTypeGroups().stream()
                .map(sortTypeGroup -> new SearchHotelsRspV1.SortTypeGroup(
                        sortTypeGroup.getId(),
                        sortTypeGroup.getName(),
                        sortTypeGroup.getSortTypes().stream().anyMatch(SortTypeRegistry.SortType::isByDistance),
                        sortTypeGroup.getSortTypes().stream()
                                .map(sortType -> new SearchHotelsRspV1.SortType(
                                        sortType.getId(),
                                        sortType.getName(),
                                        sortType.getHint()))
                                .collect(Collectors.toUnmodifiableList())))
                .collect(Collectors.toUnmodifiableList()));
        return sortInfo;
    }

    public static void updateLegacyFilters(SearchHotelsReqV1 req) {
        if (req.getFilterAtoms() != null) {
            req.setFilterAtoms(HotelsPortalUtils.updateLegacyFilters(req.getFilterAtoms()));
        }
    }

    public static void clearNonPersistentFilters(SearchHotelsReqV1 req,
                                                 HotelsFilteringService hotelsFilteringService,
                                                 Instant started,
                                                 CommonHttpHeaders commonHttpHeaders,
                                                 CachedActivePromosService cachedActivePromosService) {
        if (req.getFilterAtoms() != null) {
            var persistentAtoms = hotelsFilteringService.getPersistentAtoms(started,
                    HotelSearchUtils.getForceFilterLayout(req), req.getGeoId(), commonHttpHeaders);
            req.setFilterAtoms(req.getFilterAtoms().stream()
                    .filter(persistentAtoms::contains)
                    .collect(Collectors.toUnmodifiableList()));
        }
        clearInvalidFilters(req, hotelsFilteringService, started, commonHttpHeaders, cachedActivePromosService);
        req.setOnlyCurrentGeoId(true);
        req.setFilterPriceFrom(null);
        req.setFilterPriceTo(null);
    }

    public static void clearInvalidFilters(SearchHotelsReqV1 req,
                                           HotelsFilteringService hotelsFilteringService,
                                           Instant started,
                                           CommonHttpHeaders commonHttpHeaders,
                                           CachedActivePromosService cachedActivePromosService) {
        if (req.getFilterAtoms() != null) {
            var mirOffersFilterAvailable = req.getCheckoutDate() == null
                    || req.getCheckinDate() == null
                    || MirUtils.areDatesEligible(cachedActivePromosService.getActivePromos(), req.getCheckinDate(), req.getCheckoutDate());
            if (!mirOffersFilterAvailable) {
                var mirAtoms = hotelsFilteringService.getFilterAtoms(HotelsPortalUtils.MIR_OFFERS_FILTER_ID,
                        started, HotelSearchUtils.getForceFilterLayout(req), req.getGeoId(), commonHttpHeaders);
                req.setFilterAtoms(req.getFilterAtoms().stream().filter(x -> !mirAtoms.contains(x)).collect(Collectors.toUnmodifiableList()));
            }
        }
    }

    public static String getForceFilterLayout(SearchFilterParamsProvider searchFilterParamsProvider) {
        if (Strings.isNullOrEmpty(searchFilterParamsProvider.getForceFiltersLayout())) {
            return null;
        }
        return searchFilterParamsProvider.getForceFiltersLayout();
    }

    public static Permalink getTopHotelPermalink(HotelSlugService hotelSlugService, SearchHotelsReqV1 req) {
        return hotelSlugService.findPermalinkByHotelIdentifier(new HotelIdentifierProvider() {
            @Override
            public Permalink getPermalink() {
                return null;
            }

            @Override
            public String getHotelSlug() {
                return req.getTopHotelSlug();
            }
        });
    }

    public static void setCancellationInfoNullIfNotFullyRefundable(HotelOffer offer) {
        var cancellationInfo = offer.getCancellationInfo();
        if (cancellationInfo != null && cancellationInfo.getRefundType() != RefundType.FULLY_REFUNDABLE ) {
            offer.setCancellationInfo(null);
        }
    }

    public static void setMealTypeNullIfNotMealTypeOrUnknown(HotelOffer offer) {
        var mealType = offer.getMealType();
        if (mealType != null) {
            var eMealType = TOCPansion.EPansion.valueOf(mealType.getId());
            if (eMealType != TOCPansion.EPansion.RO && eMealType != TOCPansion.EPansion.UNKNOWN) {
                offer.setMealType(null);
            }
        }
    }

    public static String getGeoSearchBusinessId(THotelFilter filter) {
        return Strings.isNullOrEmpty(filter.getGeoSearchBusinessId()) ? filter.getFeatureId() : filter.getGeoSearchBusinessId();
    }

    public static String getGeoSearchBusinessId(HotelFilter filter) {
        return Strings.isNullOrEmpty(filter.getGeoSearchBusinessId()) ? filter.getFeatureId() : filter.getGeoSearchBusinessId();
    }

    public static TGetHotelsRequest prepareGeoCounterGetHotelsReq(Integer travelGeoId, SearchFilterParamsProvider filterParams,
                                                                  OfferSearchParamsProvider offerParams, BoundingBox bbox,
                                                                  Instant now,
                                                                  String forceFilterLayout, ESortType sortType, Coordinates sortOrigin,
                                                                  String context, boolean poolingEnabled, int pollingIteration,
                                                                  int skip, int limit, String reqId,
                                                                  boolean useSearcher, GeoOriginEnum origin, Permalink topHotelPermalink,
                                                                  UaasSearchExperiments uaasSearchExperiments,
                                                                  RequestAttributionProvider attribution, CommonHttpHeaders headers,
                                                                  GeoBase geoBase, boolean allowRestrictedUserRates,
                                                                  boolean sortOffersUsingPlus,
                                                                  HotelsFilteringService hotelsFilteringService,
                                                                  Map<Integer, Integer> forceCryptaInfo,
                                                                  boolean allowHotelsWithoutPrices) {
        TGetHotelsRequest.Builder builder = TGetHotelsRequest.newBuilder();

        if (bbox != null) {
            builder.setBbox(TBbox.newBuilder()
                    .setLowerLeft(coordinatesToProto(bbox.getLeftDown()))
                    .setUpperRight(coordinatesToProto(bbox.getUpRight()))
                    .build());
        }
        if (travelGeoId != null) {
            builder.setTravelGeoId(travelGeoId);
        }
        builder.addAllFilters(hotelsFilteringService.prepareFilters(filterParams.getFilterAtoms(), now,
                        forceFilterLayout, travelGeoId, headers)
                .stream()
                .flatMap(x -> x.getFilters().stream())
                .map(GeoCounterService::filterModelToPbOrNull)
                .filter(Objects::nonNull)
                .collect(Collectors.toUnmodifiableList()));
        if (filterParams.getFilterPriceFrom() != null || filterParams.getFilterPriceTo() != null) {
            var priceFilterBuilder = THotelFilter.TPriceValue.newBuilder();
            if (filterParams.getFilterPriceFrom() != null) {
                priceFilterBuilder.setTotalPriceFrom(UInt32Value.of(filterParams.getFilterPriceFrom()));
            }
            if (filterParams.getFilterPriceTo() != null) {
                priceFilterBuilder.setTotalPriceTo(UInt32Value.of(filterParams.getFilterPriceTo()));
            }
            builder.addFilters(THotelFilter.newBuilder()
                    .setUniqueId("~price~")
                    .setFeatureId("~price~")
                    .setPriceValue(priceFilterBuilder.build())
                    .build());
        }
        // Add region filter
        if (HotelsPortalUtils.isFilterByGeoIdEnabled(uaasSearchExperiments, travelGeoId, headers) && filterParams.isOnlyCurrentGeoId()) {
            builder.addFilters(THotelFilter.newBuilder()
                    .setUniqueId(travelGeoId == GeoBaseHelpers.MOSCOW_DISTRICT ? "only_moscow_region" : "only_leningrad_region")
                    .setFeatureId("geo_ids")
                    .setListValue(THotelFilter.TListValue.newBuilder().addValue(travelGeoId.toString())));
        }
        builder.setSortType(sortType);
        if (sortOrigin != null) {
            builder.setSortOrigin(coordinatesToProto(sortOrigin));
        }
        builder.setSortOffersUsingPlus(sortOffersUsingPlus);
        boolean isUserDeviceTouch = HotelsPortalUtils.isUserDeviceTouch(headers);
        builder.setAllowMobileRates(isUserDeviceTouch);
        if (offerParams.getCheckinDate() != null) {
            builder.setCheckInDate(offerParams.getCheckinDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        }
        if (offerParams.getCheckoutDate() != null) {
            builder.setCheckOutDate(offerParams.getCheckoutDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        }
        if (offerParams.getAdults() != null) {
            builder.setAges(Ages.build(offerParams.getAdults(), offerParams.getChildrenAges()).toString());
        }

        builder.setUseSearcher(useSearcher);

        builder.setSkip(skip);
        builder.setLimit(limit);

        if (poolingEnabled) {
            var incrementalPolling = TGetHotelsRequest.TIncrementalPolling.newBuilder();
            incrementalPolling.setEnabled(true);
            incrementalPolling.setPollingIteration(pollingIteration);
            if (context != null && !context.isBlank()) {
                incrementalPolling.setPreviousIterationToken(context);
            }
            builder.setIncrementalPolling(incrementalPolling.build());
        }

        builder.setGeoOrigin(origin.getValue());
        builder.setGeoClientId("travel.portal");

        if (topHotelPermalink != null) {
            builder.setTopHotelPermalink(Int64Value.newBuilder().setValue(topHotelPermalink.asLong()).build());
        }

        var debugDataBuilder = TGetHotelsRequest.TDebugData.newBuilder();
        debugDataBuilder.setRequestId(reqId);
        if (forceCryptaInfo != null) {
            var cryptaInfoBuilder = TCryptaInfo.newBuilder();
            for (var kv: forceCryptaInfo.entrySet()) {
                cryptaInfoBuilder.addSegments(TCryptaInfo.TSegment.newBuilder()
                        .setKeywordId(kv.getKey()).setSegmentId(kv.getValue()).build());
            }
            debugDataBuilder.setForceCryptaInfo(cryptaInfoBuilder.build());
        }
        builder.setDebugData(debugDataBuilder.build());

        if (uaasSearchExperiments.isStrikeThroughPrices()) {
            builder.addExperiments(2713);
        }
        if (uaasSearchExperiments.isOstrovokClickoutTouchDisabled() && isUserDeviceTouch) {
            builder.addExperiments(5782);
        }
        if (uaasSearchExperiments.isOstrovokClickoutDisabled()) {
            builder.addBanOpId(EOperatorId.OI_OSTROVOK_VALUE);
        }
        builder.setOnlyBoYWhenBoYAvailable(true);

        builder.addAllKVExperiments(KVExperiments.toProto(headers.getExperiments()));

        var builderAtrribuition = HotelsPortalUtils.prepareOfferCacheAttributionForGeoCounter(attribution, headers, HotelsPortalUtils.determineUserRegion(geoBase, headers, attribution));
        builder.setAttribution(builderAtrribuition.build());
        builder.setRobotRequest(HotelsPortalUtils.isRobotRequest(headers));

        builder.setAllowRestrictedUserRates(allowRestrictedUserRates);

        if (allowHotelsWithoutPrices) {
            builder.setAllowHotelsWithoutPrices(true);
        }

        builder.setWhiteLabelPartnerId(headers.getWhiteLabelPartnerId());

        return builder.build();
    }

    public static Hotel extractHotel(TGetHotelsResponse.THotel gcHotel, GeoHotel geoHotel, HotelFavoriteInfosRsp tugcRsp,
                       ImageParamsProvider imageParams, ImageWhitelistDataProvider imageWhitelistDataProvider,
                       HotelSlugService hotelSlugService, AmenityService amenityService,
                       HotelImagesService hotelImagesService, HotelsPortalProperties hotelsPortalProperties,
                       Experiments experiments, UaasSearchExperiments uaasSearchExperiments) {
        var hotel = new Hotel();
        hotel.setPermalink(Permalink.of(gcHotel.getPermalink()));
        hotel.setHotelSlug(hotelSlugService.findMainSlugByPermalink(hotel.getPermalink()));
        hotel.setName(gcHotel.getName());
        hotel.setCategory(new Hotel.Category(Long.toString(gcHotel.getRubric().getId()), gcHotel.getRubric().getName()));
        hotel.setCoordinates(new Coordinates(gcHotel.getCoordinates().getLon(), gcHotel.getCoordinates().getLat()));
        hotel.setAddress(gcHotel.getAddress());
        hotel.setStars(gcHotel.getStars() == 0 ? null : gcHotel.getStars());
        hotel.setRating(gcHotel.getRating());
        hotel.setTotalImageCount(gcHotel.getTotalImageCount());
        hotel.setMainAmenities(amenityService.buildMainAmenities(gcHotel.getFeaturesList()));

        //hotel.setAmenityGroups(Collections.emptyList());
        //hotel.setCityName();
        if (gcHotel.hasDistanceMeters()) {
            hotel.setDistanceMeters((double)gcHotel.getDistanceMeters());
            hotel.setDistanceText(HotelsPortalUtils.formatDistanceMeters(gcHotel.getDistanceMeters()));
        }
        hotel.setIsFavorite(tugcRsp != null && tugcRsp.getIsFavorite().getOrDefault(hotel.getPermalink(), false));

        if (geoHotel != null) {
            HotelsPortalUtils.HotelInfo hotelInfo = HotelsPortalUtils.extractHotel(
                    amenityService, hotelSlugService, hotelImagesService,null, geoHotel, imageParams,
                    true, null, experiments, hotelsPortalProperties.getHotelCategoriesToSearch(),
                    false, false, uaasSearchExperiments, imageWhitelistDataProvider,
                    hotelsPortalProperties.isShowNearestStationOnSnippet()
            );
            Hotel extractedHotel = hotelInfo.getHotel();
            hotel.setTotalTextReviewCount(extractedHotel.getTotalTextReviewCount());
            hotel.setTotalImageCount(extractedHotel.getTotalImageCount());
            hotel.setImages(extractedHotel.getImages());
            if (hotelsPortalProperties.isShowNearestStationOnSnippet()) {
                hotel.setNearestStations(extractedHotel.getNearestStations());
            }
            hotel.setHasVerifiedOwner(extractedHotel.getHasVerifiedOwner());
            hotel.setCityName(extractedHotel.getCityName());
            hotel.setName(extractedHotel.getName());
            hotel.setGeoFeature(extractedHotel.getGeoFeature());
        } else {
            hotel.setTotalTextReviewCount(0);
            hotel.setTotalImageCount(0);
            hotel.setImages(List.of());
            hotel.setHasVerifiedOwner(false);
            if (hotel.getName() == null) {
                hotel.setName("");
            }
        }

        return hotel;
    }

    private static TCoordinates coordinatesToProto(Coordinates coordinates) {
        return TCoordinates.newBuilder()
                .setLat(coordinates.getLat())
                .setLon(coordinates.getLon())
                .build();
    }
}
