package ru.yandex.travel.api.endpoints.hotels_portal;

import java.io.IOException;
import java.io.InputStreamReader;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import com.google.protobuf.TextFormat;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.HotelAdditionalFilter;
import ru.yandex.travel.api.models.hotels.HotelFilter;
import ru.yandex.travel.api.models.hotels.HotelFilterGroup;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.PriceCounts;
import ru.yandex.travel.api.models.hotels.SearchFilterAndTextParams;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.SearchFilterParamsProvider;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterReq;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterRsp;
import ru.yandex.travel.api.services.hotels.promo.CachedActivePromosService;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.common.promo.mir.MirUtils;
import ru.yandex.travel.hotels.proto.hotel_filters.THotelFilter;
import ru.yandex.travel.proto.hotel_filters_config.TFilterInfoConfig;

@Component
@Slf4j
public class HotelsFilteringService {

    @Data
    @AllArgsConstructor
    private static class Config {
        private final List<TFilterInfoConfig.TQuickFilterReference> quickFilterRefs;
        private final Map<String, TFilterInfoConfig.TQuickFilter> quickFiltersMap;
        private final List<TFilterInfoConfig.TBasicFilterGroup> detailedFilters;
        private final Map<String, TFilterInfoConfig.TBasicFilterGroup> detailedFiltersMap;
        private final Set<String> persistentAtoms;
        private final List<TFilterInfoConfig.TFiltersLayoutBatch> filtersLayoutBatches;
    }

    @EqualsAndHashCode
    @Data
    @AllArgsConstructor
    @ToString
    private static class ConfigKey {
        private String layoutId;
        private boolean mirEligible;
        private boolean isTouch;
        private boolean isBlackFriday;
        private boolean isRequestedGeoIdFiltersEnabled;
        private boolean isMoscowDistrictExp;
        private boolean isWhiteLabel;
    }

    public final static String FAKE_BUSINESS_ID_PREFIX = "FAKE-ID-";
    public final static String PARTNER_FILTER_FEATURE_ID = "hotel_provider";
    public final static String YANDEX_OFFERS_FILTER_BUSINESS_ID = "FAKE-ID-yandex-offers";
    public final static String MIR_OFFERS_FILTER_BUSINESS_ID = "FAKE-ID-mir-offers";
    public final static String CATEGORY_FILTER_GROUP_ID = "category_id";

    private final static String BACKEND_ID_PREFIX = "BACKEND-";
    private final static String ONLY_BOY_OFFERS_FOR_WHITE_LABEL_BACKEND_ID = "BACKEND-only-boy-offers-for-white-label";
    private final CachedActivePromosService cachedActivePromosService;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final GeoBase geoBase;

    private final Map<ConfigKey, Config> configs;
    private final List<TFilterInfoConfig.TBasicFilterGroup> backendFilters;

    public HotelsFilteringService(HotelsPortalProperties portalProperties,
                                  CachedActivePromosService cachedActivePromosService,
                                  ExperimentDataProvider uaasExperimentDataProvider,
                                  GeoBase geoBase) throws IOException {
        var is = this.getClass().getClassLoader().getResourceAsStream("portal-search-filters.pb.txt");
        if (is == null) {
            throw new RuntimeException("Portal search filters description not found");
        }

        var reader = new InputStreamReader(is);
        var filersConfigBuilder = TFilterInfoConfig.newBuilder();
        TextFormat.merge(reader, filersConfigBuilder);
        reader.close();

        this.configs = new HashMap<>();
        this.cachedActivePromosService = cachedActivePromosService;
        this.uaasExperimentDataProvider = uaasExperimentDataProvider;
        this.geoBase = geoBase;
        var filterInfoConfig = filersConfigBuilder.build();
        var defaultFilterLayoutTags = new HashSet<>(portalProperties.getFilterLayoutTags());
        for (var layout: filterInfoConfig.getFiltersLayoutsList()) {
            doTwice(mirEligible ->
                doTwice(isTouch ->
                    doTwice(blackFriday ->
                        doTwice(isRequestedGeoIdFiltersEnabled ->
                            doTwice(isMoscowAreaExp ->
                                doTwice(isWhiteLabel ->
                                    addFilterConfig(filterInfoConfig, defaultFilterLayoutTags, layout, mirEligible,
                                                isTouch, blackFriday, isRequestedGeoIdFiltersEnabled,
                                                isMoscowAreaExp, isWhiteLabel)
                                )
                            )
                        )
                    )
                )
            );
        }
        this.backendFilters = addBackendFilters(filterInfoConfig);
    }

    private static void doTwice(Consumer<Boolean> c) {
        c.accept(true);
        c.accept(false);
    }

    private void addFilterConfig(TFilterInfoConfig filterInfoConfig, Set<String> defaultTags,
                                 TFilterInfoConfig.TFiltersLayout layout, boolean mirEligible, boolean isTouch,
                                 boolean isBlackFriday, boolean isRequestedGeoIdFiltersEnabled,
                                 boolean isMoscowAreaExp, boolean isWhiteLabel) {
        var tags = new HashSet<>(defaultTags);
        tags.add(mirEligible ? "mir" : "not-mir");
        tags.add(isTouch ? "touch" : "desktop");
        tags.add(isBlackFriday ? "black-friday" : "no-black-friday");
        tags.add(isRequestedGeoIdFiltersEnabled ? "only-requested-geo-id" : "without-only-requested-geo-id");
        tags.add(isMoscowAreaExp ? "is-moscow-area-exp" : "no-moscow-area-exp");
        tags.add(isWhiteLabel ? "white-label" : "no-white-label");

        var currBatches = layout.getBatchesList()
                .stream()
                .filter(x -> tags.containsAll(x.getTagList()))
                .collect(Collectors.toUnmodifiableList());

        var currQuickFilterRefs = layout.getQuickFiltersRefsList()
                .stream()
                .filter(x -> tags.containsAll(x.getTagList()))
                .collect(Collectors.toUnmodifiableList());

        var usedFilters = currBatches
                .stream()
                .flatMap(x -> x.getItemsList().stream())
                .filter(TFilterInfoConfig.TFiltersLayoutBatchItem::hasGroupItem)
                .map(x -> x.getGroupItem().getId())
                .collect(Collectors.toUnmodifiableSet());

        var quickFiltersMap = filterInfoConfig.getQuickFiltersList()
                .stream()
                .collect(Collectors.toUnmodifiableMap(TFilterInfoConfig.TQuickFilter::getId, x -> x));
        var detailedFilters = filterInfoConfig.getDetailedFiltersList().stream()
                .filter(x -> usedFilters.contains(x.getId()))
                .collect(Collectors.toUnmodifiableList());
        var detailedFiltersMap = detailedFilters.stream()
                .collect(Collectors.toMap(TFilterInfoConfig.TBasicFilterGroup::getId, x -> x));

        var currPersistentAtomsSet = detailedFilters
                .stream()
                .flatMap(x -> x.getItemsList().stream())
                .filter(TFilterInfoConfig.TBasicFilter::getPersistent)
                .flatMap(x -> filterToAtomList(x).stream())
                .collect(Collectors.toUnmodifiableSet());

        var allKnownQuickFilterIds = filterInfoConfig.getQuickFiltersList().stream()
                .map(TFilterInfoConfig.TQuickFilter::getId)
                .collect(Collectors.toUnmodifiableSet());

        var allKnownFilterIds = filterInfoConfig.getDetailedFiltersList().stream()
                .map(TFilterInfoConfig.TBasicFilterGroup::getId)
                .collect(Collectors.toUnmodifiableSet());

        var notMatchedQuickFilter = currQuickFilterRefs.stream()
                .filter(x -> !allKnownQuickFilterIds.contains(x.getId()) && x.getType() == TFilterInfoConfig.EQuickControlType.CT_QuickFilter)
                .map(TFilterInfoConfig.TQuickFilterReference::getId)
                .findFirst();
        Preconditions.checkState(notMatchedQuickFilter.isEmpty(), "Found unknown quick filter in layout: '%s'", notMatchedQuickFilter.orElse(""));
        var notMatchedFilter = usedFilters.stream().filter(x -> !allKnownFilterIds.contains(x)).findFirst();
        Preconditions.checkState(notMatchedFilter.isEmpty(), "Found unknown filter in layout: '%s'", notMatchedQuickFilter.orElse(""));

        configs.put(
            new ConfigKey(layout.getId(), mirEligible, isTouch, isBlackFriday,
                    isRequestedGeoIdFiltersEnabled, isMoscowAreaExp, isWhiteLabel),
            new Config(currQuickFilterRefs, quickFiltersMap, detailedFilters, detailedFiltersMap, currPersistentAtomsSet, currBatches)
        );
    }

    private List<TFilterInfoConfig.TBasicFilterGroup> addBackendFilters(TFilterInfoConfig filterInfoConfig) {
        return filterInfoConfig.getDetailedFiltersList().stream()
                .filter(x -> x.getId().startsWith(BACKEND_ID_PREFIX))
                .collect(Collectors.toList());
    }

    public SearchHotelsRspV1.FilterInfo composeDefaultFilterInfo(OfferSearchParamsProvider searchParams, Instant now,
                                                                 String forceFilterLayout, Integer requestedGeoId,
                                                                 Boolean isOnlyCurrentGeoId, CommonHttpHeaders headers) {
        return composeFilterInfo(searchParams, Collections.emptyList(), Collections.emptyMap(), 0, null, false, now,
                forceFilterLayout, requestedGeoId, isOnlyCurrentGeoId, headers);
    }

    public SearchHotelsRspV1.SearchControlInfo composeDefaultControlInfo(OfferSearchParamsProvider searchParams, Instant now,
                                                                         String forceFilterLayout, Integer requestedGeoId,
                                                                         CommonHttpHeaders headers) {
        var controlInfo = new SearchHotelsRspV1.SearchControlInfo();

        controlInfo.setQuickControls(
            composeQuickControls(searchParams, getCurrentConfig(now, forceFilterLayout, requestedGeoId, headers))
        );

        return controlInfo;
    }

    public SearchHotelsRspV1.SearchControlInfo composeControlInfo(OfferSearchParamsProvider searchParams, Instant now,
                                                                  String forceFilterLayout, Integer requestedGeoId,
                                                                  CommonHttpHeaders headers) {
        var controlInfo = new SearchHotelsRspV1.SearchControlInfo();

        controlInfo.setQuickControls(
            composeQuickControls(searchParams, getCurrentConfig(now, forceFilterLayout, requestedGeoId, headers))
        );

        return controlInfo;
    }

    public SearchHotelsRspV1.ResetFilterInfo composeResetFilterInfo(SearchFilterParamsProvider filterParams, Instant now,
                                                                    String forceFilterLayout, Integer requestedGeoId,
                                                                    CommonHttpHeaders headers) {
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        SearchHotelsRspV1.ResetFilterInfo resetFilterInfo = new SearchHotelsRspV1.ResetFilterInfo();
        var requestedFilterGroups = getSelectedFilterGroups(
                filterParams.getFilterAtoms(), now, forceFilterLayout, requestedGeoId, headers);
        boolean hasPriceFilter = filterParams.getFilterPriceTo() != null || filterParams.getFilterPriceFrom() != null;
        boolean hasGeoIdFilter = filterParams.isOnlyCurrentGeoId()
                && HotelsPortalUtils.isFilterByGeoIdEnabled(uaasSearchExperiments, requestedGeoId, headers);
        int requestedSimpleFiltersCount = requestedFilterGroups.stream().mapToInt(filterGroup -> filterGroup.getItemsList().size()).sum();
        int requestedFiltersCount = requestedSimpleFiltersCount + (hasPriceFilter ? 1 : 0) + (hasGeoIdFilter ? 1 : 0);

        List<SearchHotelsRspV1.ResetFilterInfo.ResetFilterAction> actions = new ArrayList<>();

        resetFilterInfo.setActions(actions);

        if (requestedFiltersCount == 0) {
            resetFilterInfo.setResetFilterReason(SearchHotelsRspV1.ResetFilterInfo.EResetFilterReason.ZOOMING_MAP);

            return resetFilterInfo;
        }

        List<String> atomIds = new ArrayList<>();

        if (requestedFiltersCount > 1) {
            var resetAllAction = new SearchHotelsRspV1.ResetFilterInfo.ResetFilterAction();
            resetAllAction.setName("Сбросить все");
            resetAllAction.setNeedResetFilterPrice(true);
            resetAllAction.setAtomsOff(atomIds);
            resetAllAction.setEffect("reset:all");

            actions.add(resetAllAction);
        }

        requestedFilterGroups.forEach(detailedFilterConfig ->
                detailedFilterConfig.getItemsList().forEach(filter -> {
                    var filterAction = new SearchHotelsRspV1.ResetFilterInfo.ResetFilterAction();
                    var filterAtomsIds = filterToAtomList(filter);
                    String actionName = filter.hasSpecialNameForResetAction() ? filter.getSpecialNameForResetAction() : filter.getName();

                    filterAction.setName(actionName);
                    filterAction.setNeedResetFilterPrice(false);
                    filterAction.setNeedResetGeoIdFilter(false);
                    filterAction.setAtomsOff(filterAtomsIds);
                    filterAction.setEffect("reset:simple");

                    atomIds.addAll(filterAtomsIds);
                    actions.add(filterAction);
                }));

        if (hasGeoIdFilter) {
            var resetGeoIdFilterAction = new SearchHotelsRspV1.ResetFilterInfo.ResetFilterAction();
            String geoIdFilterName = "";
            switch (requestedGeoId) {
                case GeoBaseHelpers.MOSCOW_DISTRICT:
                    geoIdFilterName = "Московская область";
                    break;
                case GeoBaseHelpers.LENINGRAD_DISTRICT:
                    geoIdFilterName = "Ленинградская область";
                    break;
                default: GeoBaseHelpers.getRegionName(geoBase, "ru", requestedGeoId);
            }
            resetGeoIdFilterAction.setName(geoIdFilterName);
            resetGeoIdFilterAction.setAtomsOff(new ArrayList<>());
            resetGeoIdFilterAction.setNeedResetFilterPrice(false);
            resetGeoIdFilterAction.setNeedResetGeoIdFilter(true);
            resetGeoIdFilterAction.setEffect("reset:simple");

            actions.add(resetGeoIdFilterAction);
        }

        if (hasPriceFilter) {
            var resetPriceAction = new SearchHotelsRspV1.ResetFilterInfo.ResetFilterAction();
            resetPriceAction.setName("Цена");
            resetPriceAction.setAtomsOff(new ArrayList<>());
            resetPriceAction.setNeedResetFilterPrice(true);
            resetPriceAction.setNeedResetGeoIdFilter(false);
            resetPriceAction.setEffect("reset:simple");

            actions.add(resetPriceAction);
        }

        resetFilterInfo.setResetFilterReason(SearchHotelsRspV1.ResetFilterInfo.EResetFilterReason.FILTERS);

        return resetFilterInfo;
    }

    public Set<String> getFilterAtoms(String filterId, Instant now, String forceFilterLayout, Integer requestedGeoId, CommonHttpHeaders headers) {
        return getCurrentConfig(now, forceFilterLayout, requestedGeoId, headers).getDetailedFilters()
                .stream()
                .flatMap(detailedFilterConfig ->
                        detailedFilterConfig.getItemsList()
                                .stream()
                                .filter(basicFilterConfig -> basicFilterConfig.getFilter().getUniqueId().equals(filterId))
                                .map(HotelsFilteringService::filterToAtomList)
                                .flatMap(Collection::stream))
                .collect(Collectors.toUnmodifiableSet());
    }

    public Set<String> getPersistentAtoms(Instant now, String forceFilterLayout, Integer requestedGeoId, CommonHttpHeaders headers) {
        return getCurrentConfig(now, forceFilterLayout, requestedGeoId, headers).getPersistentAtoms();
    }

    private Set<String> getFilterAtomSet(List<String> list) {
        return list == null ? Collections.emptySet() : Set.copyOf(list);
    }

    private List<TFilterInfoConfig.TBasicFilterGroup> getSelectedFilterGroups(List<String> selectedAtomsList, Instant now, String forceFilterLayout,
                                                                              Integer requestedGeoId, CommonHttpHeaders headers) {
        Set<String> selectedAtoms = getFilterAtomSet(selectedAtomsList);
        Set<String> knownAtoms = getFilters(now, forceFilterLayout, requestedGeoId, headers).stream()
                .flatMap(detailedFilterConfig -> detailedFilterConfig.getItemsList().stream().map(HotelsFilteringService::filterToAtomList).flatMap(Collection::stream))
                .collect(Collectors.toUnmodifiableSet());
        List<String> unknownAtoms = selectedAtoms.stream().filter(x -> !knownAtoms.contains(x)).collect(Collectors.toUnmodifiableList());
        if (!unknownAtoms.isEmpty()) {
            log.warn("Unknown atoms: {}", unknownAtoms);
        }
        return getFilters(now, forceFilterLayout, requestedGeoId, headers).stream()
                .map(detailedFilterConfig -> {
                    var filters = detailedFilterConfig
                            .getItemsList()
                            .stream()
                            .filter(basicFilter -> {
                                var atoms = filterToAtomList(basicFilter);
                                var selectedCount = atoms.stream().filter(selectedAtoms::contains).count();
                                if (selectedCount > 0 && selectedCount < atoms.size()) {
                                    throw new TravelApiBadRequestException(String.format("Found partially matched filter " +
                                            "(basic " +
                                            "filter id: %s)", basicFilter.getFilter().getUniqueId()));
                                }
                                return selectedCount == atoms.size() && !atoms.isEmpty(); // Skipping filters without atoms: they are always true
                            })
                            .collect(Collectors.toUnmodifiableList());

                    if (filters.size() == 0) {
                        return null;
                    }

                    return detailedFilterConfig.toBuilder().clearItems().addAllItems(filters).build();
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toUnmodifiableList());
    }

    private List<TFilterInfoConfig.TBasicFilterGroup> getActiveBackendFilters(CommonHttpHeaders headers) {
        if (HotelsPortalUtils.isWhiteLabelActive(headers)) {
            return backendFilters.stream()
                    .filter(x -> x.getId().equals(ONLY_BOY_OFFERS_FOR_WHITE_LABEL_BACKEND_ID))
                    .collect(Collectors.toList());
        } else {
            return List.of();
        }
    }

    public List<HotelFilterGroup> prepareFilters(List<String> selectedAtomsList, Instant now, String forceFilterLayout,
                                                 Integer requestedGeoId, CommonHttpHeaders headers) {
        List<TFilterInfoConfig.TBasicFilterGroup> selectedFilters = getSelectedFilterGroups(selectedAtomsList, now, forceFilterLayout, requestedGeoId, headers);
        List<TFilterInfoConfig.TBasicFilterGroup> activeBackendFilters = getActiveBackendFilters(headers);
        return Stream.concat(selectedFilters.stream(), activeBackendFilters.stream())
                .map(detailedFilterConfig -> {
                    var filtersForRequest = detailedFilterConfig.getItemsList()
                            .stream().map(HotelsFilteringService::configFilterToFilterModelOrNull)
                            .filter(Objects::nonNull)
                            .collect(Collectors.toUnmodifiableList());

                    if (!filtersForRequest.isEmpty()) {
                        if (detailedFilterConfig.getType() == TFilterInfoConfig.TBasicFilterGroup.EBasicFilterGroupType.Or) {
                            return List.of(new HotelFilterGroup(filtersForRequest.get(0).getFeatureId(),
                                    filtersForRequest));
                        } else {
                            if (detailedFilterConfig.getType() == TFilterInfoConfig.TBasicFilterGroup.EBasicFilterGroupType.Single && filtersForRequest.size() > 1) {
                                throw new TravelApiBadRequestException(String.format("Found several matched filters in " +
                                        "group of type 'single' " +
                                        "(ids: %s)", String.join(", ",
                                        filtersForRequest.stream().map(HotelFilter::getUniqueId).collect(Collectors.toUnmodifiableList()))));
                            }
                            return filtersForRequest.stream().map(filter -> new HotelFilterGroup(filter.getFeatureId(), List.of(filter))).collect(Collectors.toUnmodifiableList());
                        }
                    }
                    return new ArrayList<HotelFilterGroup>();
                })
                .flatMap(Collection::stream)
                .collect(Collectors.toUnmodifiableList());
    }

    private List<HotelAdditionalFilter> prepareAdditionalFilters(Instant now, String forceFilterLayout, Integer requestedGeoId, CommonHttpHeaders headers) {
        return getFilters(now, forceFilterLayout, requestedGeoId, headers)
                .stream()
                .flatMap(detailedFilterConfig -> detailedFilterConfig.getItemsList()
                        .stream()
                        .map(basicFilterConfig -> {
                            var filterModel = configFilterToFilterModelOrNull(basicFilterConfig);
                            if (filterModel == null) {
                                return null;
                            }
                            return new HotelAdditionalFilter(
                                    filterModel.getFeatureId(),
                                    toModelAdditionalFilterType(detailedFilterConfig.getType()),
                                    filterModel);
                        })
                        .filter(Objects::nonNull))
                .collect(Collectors.toUnmodifiableList());
    }

    private HotelAdditionalFilter.Type toModelAdditionalFilterType(TFilterInfoConfig.TBasicFilterGroup.EBasicFilterGroupType type) {
        switch (type) {
            case Or:
                return HotelAdditionalFilter.Type.or;
            case And:
                return HotelAdditionalFilter.Type.and;
            case Single:
                return HotelAdditionalFilter.Type.single;
            default:
                throw new IllegalStateException(String.format("Unknown additional filter type: %s", type));
        }
    }

    public GeoCounterReq prepareGeoCounterReq(SearchFilterParamsProvider filterParams,
                                              OfferSearchParamsProvider offerParams, BoundingBox bbox,
                                              boolean useOfferBus, Instant now,
                                              String forceFilterLayout, Integer requestedGeoId, CommonHttpHeaders headers) {
        GeoCounterReq gcReq = new GeoCounterReq();
        gcReq.setCheckinDate(offerParams.getCheckinDate());
        gcReq.setCheckoutDate(offerParams.getCheckoutDate());
        gcReq.setAges(Ages.build(offerParams.getAdults(), offerParams.getChildrenAges()).toString());
        gcReq.setBbox(bbox);
        gcReq.setFilterPriceFrom(filterParams.getFilterPriceFrom());
        gcReq.setFilterPriceTo(filterParams.getFilterPriceTo());
        gcReq.setInitialFilterGroups(prepareFilters(filterParams.getFilterAtoms(), now, forceFilterLayout, requestedGeoId, headers)
                .stream()
                .filter(group -> !group.getUniqueId().equals(PARTNER_FILTER_FEATURE_ID))
                .collect(Collectors.toUnmodifiableList()));
        gcReq.setAdditionalFilters(prepareAdditionalFilters(now, forceFilterLayout, requestedGeoId, headers)
                .stream()
                .filter(additionalFilter -> !additionalFilter.getFilter().getFeatureId().equals(PARTNER_FILTER_FEATURE_ID))
                .collect(Collectors.toUnmodifiableList()));
        gcReq.setUseOfferBus(useOfferBus);
        return gcReq;
    }

    public CountHotelsRspV1 composeCountHotelsRsp(OfferSearchParamsProvider searchParams,
                                                  SearchFilterParamsProvider filterParams, GeoCounterRsp gcRsp,
                                                  Instant now, String forceFilterLayout, CommonHttpHeaders headers,
                                                  Experiments experiments, String selectedSortId, Coordinates sortOrigin,
                                                  Integer requestedGeoId, boolean isOnlyRequestedGeoId) {
        var counts = gcRsp.getCounts();
        var rsp = new CountHotelsRspV1();
        rsp.setFoundHotelCount((int) counts.getMatchedCount());
        var countsMap = new HashMap<String, Long>();
        counts.getAdditionalFilterCountsList().forEach(predefinedFilterCount -> {
            var id = predefinedFilterCount.getUniqueId();
            var count = predefinedFilterCount.getCount();
            countsMap.put(id, count);
        });
        var priceCounts = !counts.hasPriceCounts() ? null : new PriceCounts(
                counts.getPriceCounts().getMinPriceEstimate(),
                counts.getPriceCounts().getMaxPriceEstimate(),
                counts.getPriceCounts().getHistogramBoundsList(),
                counts.getPriceCounts().getHistogramCountsList());
        rsp.setFilterInfo(composeFilterInfo(searchParams, filterParams.getFilterAtoms(), countsMap, counts.getMatchedCount(),
                priceCounts, true, now, forceFilterLayout, requestedGeoId, isOnlyRequestedGeoId, headers));
        rsp.setSearchControlInfo(composeControlInfo(searchParams, now, forceFilterLayout, requestedGeoId, headers));

        var searchFilterParams = new SearchFilterAndTextParams();
        searchFilterParams.setFilterAtoms(filterParams.getFilterAtoms());
        searchFilterParams.setFilterPriceFrom(filterParams.getFilterPriceFrom());
        searchFilterParams.setFilterPriceTo(filterParams.getFilterPriceTo());
        rsp.getFilterInfo().setParams(searchFilterParams);

        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        SortTypeRegistry sortTypeRegistry = SortTypeRegistry.getSortTypeRegistry(
                HotelsPortalUtils.isHotelsNearbyEnabled(headers),
                HotelsPortalUtils.getSortTypeLayout(uaasSearchExperiments),
                uaasSearchExperiments.isNewRanking(),
                uaasSearchExperiments.isPersonalRanking() || experiments.isExp("personal-ranking"),
                HotelsPortalUtils.getExplorationRankingCategory(uaasSearchExperiments, experiments),
                HotelsPortalUtils.getRealTimeRanking(uaasSearchExperiments, experiments, false));
        rsp.setSortInfo(HotelSearchUtils.buildSortInfo(sortTypeRegistry, selectedSortId, null));
        return rsp;
    }

    private Config getCurrentConfig(Instant now, String forceFilterLayoutId, Integer requestedGeoId, CommonHttpHeaders headers) {
        var activePromosRsp = cachedActivePromosService.getActivePromos();
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        var configKey = new ConfigKey(
                Objects.requireNonNullElse(forceFilterLayoutId, "default"),
                MirUtils.isActive(activePromosRsp),
                HotelsPortalUtils.isUserDeviceTouch(headers),
                activePromosRsp != null && activePromosRsp.hasBlackFriday() && activePromosRsp.getBlackFriday().getActive(),
                HotelsPortalUtils.isFilterByGeoIdEnabled(uaasSearchExperiments, requestedGeoId, headers),
                uaasSearchExperiments.isMoskowAreaEnabled(),
                HotelsPortalUtils.isWhiteLabelActive(headers)
        );
        var config = configs.get(configKey);
        if (config == null) {
            if (forceFilterLayoutId != null) { // got from request -> bad request
                throw new TravelApiBadRequestException("Filter config not found by key " + configKey);
            } else { // got from config -> bad state
                throw new IllegalStateException("Filter config not found by key " + configKey);
            }
        }
        return config;
    }

    private List<TFilterInfoConfig.TBasicFilterGroup> getFilters(Instant now, String forceFilterLayout,
                                                                 Integer requestedGeoId, CommonHttpHeaders headers) {
        return getCurrentConfig(now, forceFilterLayout, requestedGeoId, headers).getDetailedFilters();
    }

    private boolean isMirOffersFilterAvailable(OfferSearchParamsProvider searchParams) {
        return searchParams == null
                || searchParams.getCheckoutDate() == null
                || searchParams.getCheckinDate() == null
                || MirUtils.areDatesEligible(cachedActivePromosService.getActivePromos(), searchParams.getCheckinDate(), searchParams.getCheckoutDate());
    }

    private List<SearchHotelsRspV1.QuickControl> composeQuickControls(OfferSearchParamsProvider searchParams, Config currentConfig) {
        return currentConfig.getQuickFilterRefs().stream()
            .map(quickFilterRef -> {
                if (quickFilterRef.getType() == TFilterInfoConfig.EQuickControlType.CT_QuickSort) {
                    var quickSort = new SearchHotelsRspV1.QuickSort();
                    quickSort.setId(quickFilterRef.getId());
                    quickSort.setEnabled(true);

                    return quickSort;
                }

                if (quickFilterRef.getType() == TFilterInfoConfig.EQuickControlType.CT_QuickPriceFilter) {
                    var quickPriceFilter = new SearchHotelsRspV1.QuickPriceFilter();
                    quickPriceFilter.setId(quickFilterRef.getId());
                    quickPriceFilter.setName("Цена");
                    quickPriceFilter.setEnabled(true);

                    return quickPriceFilter;
                }

                if (quickFilterRef.getType() == TFilterInfoConfig.EQuickControlType.CT_QuickFilter) {
                    var quickFilterConfig = currentConfig.getQuickFiltersMap().get(quickFilterRef.getQuickFilterId());

                    return quickFilterConfig != null
                            ? composeQuickFilter(quickFilterConfig, isMirOffersFilterAvailable(searchParams))
                            : null;
                }

                return null;
            })
            .filter(Objects::nonNull)
            .collect(Collectors.toUnmodifiableList());
    }

    private SearchHotelsRspV1.QuickFilter composeQuickFilter(TFilterInfoConfig.TQuickFilter quickFilterConfig, boolean mirOffersFilterAvailable) {
        var quickFilter = new SearchHotelsRspV1.QuickFilter();
        quickFilter.setId(quickFilterConfig.getId());
        if (quickFilter.getId().startsWith(HotelsPortalUtils.MIR_OFFERS_QUICK_FILTER_ID)) {
            // TODO(alexcrush) Remove after FRONTEND will remove their id-based hacks
            quickFilter.setId(HotelsPortalUtils.MIR_OFFERS_QUICK_FILTER_ID);
        }
        if (quickFilter.getId().startsWith("covid-safety-quick")) {
            // TODO(alexcrush) Remove after FRONTEND will remove their id-based hacks
            quickFilter.setId("covid-safety-quick");
        }
        quickFilter.setName(quickFilterConfig.getName());
        if (quickFilterConfig.hasHint()) {
            quickFilter.setHint(quickFilterConfig.getHint());
        }
        quickFilter.setEffect(quickFilterConfig.getEffect());
        if (quickFilter.getId().equals(HotelsPortalUtils.MIR_OFFERS_QUICK_FILTER_ID) && !mirOffersFilterAvailable) {
            quickFilter.setEnabled(false); // https://st.yandex-team.ru/TRAVELBACK-1308
            if (quickFilterConfig.hasHintForDisabled()) {
                quickFilter.setHint(quickFilterConfig.getHintForDisabled());
            }
        } else {
            quickFilter.setEnabled(true);
        }
        quickFilter.setAtomsOn(quickFilterConfig.getAtomsOnList());
        quickFilter.setAtomsOff(quickFilterConfig.getAtomsOffList());
        return quickFilter;
    }

    private SearchHotelsRspV1.FilterInfo composeFilterInfo(OfferSearchParamsProvider searchParams,
                                                           List<String> selectedAtomsList,
                                                           Map<String, Long> countsMap,
                                                           long matchedCount,
                                                           PriceCounts priceCounts,
                                                           boolean addCountInfo,
                                                           Instant now,
                                                           String forceFilterLayout,
                                                           Integer requestedGeoId,
                                                           boolean isOnlyCurrentGeoId,
                                                           CommonHttpHeaders headers) {
        Set<String> selectedAtoms = getFilterAtomSet(selectedAtomsList);
        var filterInfo = new SearchHotelsRspV1.FilterInfo();
        var currentConfig = getCurrentConfig(now, forceFilterLayout, requestedGeoId, headers);
        var mirOffersFilterAvailable = isMirOffersFilterAvailable(searchParams);

        filterInfo.setQuickFilters(currentConfig.getQuickFilterRefs().stream()
                .filter(tQuickFilterRef -> tQuickFilterRef.getType() == TFilterInfoConfig.EQuickControlType.CT_QuickFilter)
                .map(quickFilterRef -> {
                    var quickFilterConfig = currentConfig.getQuickFiltersMap().get(quickFilterRef.getQuickFilterId());

                    return quickFilterConfig != null
                            ? composeQuickFilter(quickFilterConfig, mirOffersFilterAvailable)
                            : null;
                })
                .filter(x -> x != null)
                .collect(Collectors.toUnmodifiableList()));

        var filtersMap = currentConfig.getDetailedFiltersMap();
        filterInfo.setDetailedFiltersBatches(currentConfig.getFiltersLayoutBatches().stream()
                .filter(batchConfig -> batchConfig.getTagList().stream().noneMatch(tag -> Objects.equals(tag, "only-requested-geo-id")))
                .map(batchConfig -> {
                    var batch = new SearchHotelsRspV1.DetailedFiltersBatch();
                    batch.setItems(batchConfig.getItemsList().stream()
                            .map(itemConfig -> {
                                var item = new SearchHotelsRspV1.DetailedFiltersItem();
                                if (itemConfig.hasPriceItem()) {
                                    item.setType(SearchHotelsRspV1.DetailedFiltersBatchItemType.PRICE);
                                } else if (itemConfig.hasGroupItem()) {
                                    item.setType(SearchHotelsRspV1.DetailedFiltersBatchItemType.GROUP);
                                    item.setDetailedFilters(composeBasicFilterGroup(
                                            filtersMap.get(itemConfig.getGroupItem().getId()), selectedAtoms,
                                            countsMap, matchedCount, addCountInfo, mirOffersFilterAvailable));
                                }
                                return item;
                            })
                            .collect(Collectors.toUnmodifiableList()));
                    return batch;
                })
                .collect(Collectors.toUnmodifiableList()));
        filterInfo.setDetailedFilters(filterInfo.getDetailedFiltersBatches().stream().flatMap(x -> x.getItems().stream()).collect(Collectors.toUnmodifiableList()));

        var geoIdFilterEnabled = currentConfig.getFiltersLayoutBatches().stream().anyMatch(batchConfig ->
                batchConfig.getTagList().stream().anyMatch(tag -> Objects.equals(tag, "only-requested-geo-id")));
        if (geoIdFilterEnabled) {
            var geoIdFilter = new SearchHotelsRspV1.GeoIdFilter();
            geoIdFilter.setSelected(isOnlyCurrentGeoId);
            String filterName;
            switch (requestedGeoId) {
                case GeoBaseHelpers.MOSCOW_DISTRICT:
                    filterName = "Московская область";
                    break;
                case GeoBaseHelpers.LENINGRAD_DISTRICT:
                    filterName = "Ленинградская область";
                    break;
                default:
                    filterName = GeoBaseHelpers.getRegionName(geoBase, "ru", requestedGeoId);
            }
            geoIdFilter.setName(filterName);
            filterInfo.setGeoIdFilter(geoIdFilter);
        }

        if (priceCounts != null) {
            filterInfo.setPriceFilter(composePriceFilter(priceCounts));
        }

        return filterInfo;
    }

    private SearchHotelsRspV1.PriceFilter composePriceFilter(PriceCounts priceCounts) {
        var priceFilter = new SearchHotelsRspV1.PriceFilter();
        priceFilter.setCurrency(Price.Currency.RUB);
        priceFilter.setMinPriceEstimate((int) priceCounts.getMinPriceEstimate());
        priceFilter.setMaxPriceEstimate((int) priceCounts.getMaxPriceEstimate());
        priceFilter.setHistogramBounds(priceCounts.getHistogramBounds());
        priceFilter.setHistogramCounts(priceCounts.getHistogramCounts());

        return priceFilter;
    }

    private static SearchHotelsRspV1.BasicFilterGroup composeBasicFilterGroup(TFilterInfoConfig.TBasicFilterGroup detailedFilterConfig,
                                                                              Set<String> selectedAtoms,
                                                                              Map<String, Long> countsMap,
                                                                              long matchedCount,
                                                                              boolean addCountInfo,
                                                                              boolean mirOffersFilterAvailable) {
        var basicFilterGroup = new SearchHotelsRspV1.BasicFilterGroup();
        basicFilterGroup.setId(detailedFilterConfig.getId());
        basicFilterGroup.setName(detailedFilterConfig.getName());
        basicFilterGroup.setType(detailedFilterConfig.getType() == TFilterInfoConfig.TBasicFilterGroup.EBasicFilterGroupType.Single
                ? SearchHotelsRspV1.BasicFilterGroupType.SINGLE
                : SearchHotelsRspV1.BasicFilterGroupType.MULTI);
        var usePlus = detailedFilterConfig.getType() == TFilterInfoConfig.TBasicFilterGroup.EBasicFilterGroupType.Or
                && detailedFilterConfig.getItemsList().stream().map(HotelsFilteringService::filterToAtomList).anyMatch(selectedAtoms::containsAll);
        var assumeEmptyAtomsFilterSelected = detailedFilterConfig.getType() == TFilterInfoConfig.TBasicFilterGroup.EBasicFilterGroupType.Single
                && detailedFilterConfig.getItemsList().stream().map(HotelsFilteringService::filterToAtomList).filter(x -> !x.isEmpty()).noneMatch(selectedAtoms::containsAll);
        basicFilterGroup.setItems(detailedFilterConfig.getItemsList().stream().map(basicFilterConfig -> {
            var basicFilter = new SearchHotelsRspV1.BasicFilter();
            basicFilter.setId(basicFilterConfig.getFilter().getUniqueId());
            basicFilter.setName(basicFilterConfig.getName());
            basicFilter.setEffect(basicFilterConfig.getEffect());
            basicFilter.setAtoms(filterToAtomList(basicFilterConfig));
            var enabled = true; // https://st.yandex-team.ru/HOTELS-4694#5de7daa76aa73033bd620a3e
            if (basicFilterConfig.getFilter().getUniqueId().equals(HotelsPortalUtils.MIR_OFFERS_FILTER_ID) && !mirOffersFilterAvailable) {
                enabled = false; // https://st.yandex-team.ru/TRAVELBACK-1308
            }
            basicFilter.setEnabled(enabled);
            if (addCountInfo) {
                var count = countsMap.getOrDefault(basicFilterConfig.getFilter().getUniqueId(), null);
                var filterSelected = selectedAtoms.containsAll(filterToAtomList(basicFilterConfig)) &&
                        (!filterToAtomList(basicFilterConfig).isEmpty() || assumeEmptyAtomsFilterSelected);
                var disableHint = count == null || filterSelected || !enabled;
                if (!disableHint) {
                    basicFilter.setHint(usePlus ? ("+" + (count - matchedCount)) : count.toString());
                } else {
                    basicFilter.setHint("");
                }
            } else {
                basicFilter.setHint("");
            }
            return basicFilter;
        }).collect(Collectors.toUnmodifiableList()));
        return basicFilterGroup;
    }

    private static HotelFilter configFilterToFilterModelOrNull(TFilterInfoConfig.TBasicFilter filter) {
        if (filter.getFilter().hasComparableValue()) {
            var comparableValue = filter.getFilter().getComparableValue();
            return new HotelFilter(
                    filter.getFilter().getUniqueId(),
                    filter.getFilter().getFeatureId(),
                    filter.getFilter().getGeoSearchBusinessId(),
                    new HotelFilter.NumericValue(toModelNumericFilterMode(comparableValue.getMode()),
                            comparableValue.getValue()),
                    null,
                    null);
        } else if (filter.getFilter().hasListValue()) {
            return new HotelFilter(
                    filter.getFilter().getUniqueId(),
                    filter.getFilter().getFeatureId(),
                    filter.getFilter().getGeoSearchBusinessId(),
                    null,
                    new HotelFilter.ListValue(filter.getFilter().getListValue().getValueList()),
                    null);
        } else if (filter.getFilter().hasIgnoredValue()) {
            return new HotelFilter(
                    filter.getFilter().getUniqueId(),
                    filter.getFilter().getFeatureId(),
                    filter.getFilter().getGeoSearchBusinessId(),
                    null,
                    null,
                    new HotelFilter.IgnoredValue());
        }
        log.error("Unknown filter type in config. Id: {}", filter.getFilter().getUniqueId());
        return null;
    }

    private static HotelFilter.NumericValue.Mode toModelNumericFilterMode(THotelFilter.TComparableValue.EMode mode) {
        switch (mode) {
            case Less:
                return HotelFilter.NumericValue.Mode.less;
            case Greater:
                return HotelFilter.NumericValue.Mode.greater;
            case LessOrEqual:
                return HotelFilter.NumericValue.Mode.lessOrEqual;
            case GreaterOrEqual:
                return HotelFilter.NumericValue.Mode.greaterOrEqual;
            default:
                throw new IllegalStateException(String.format("Unknown comparable value mode: %s", mode));
        }
    }

    private static String convertEModeToString(THotelFilter.TComparableValue.EMode mode) {
        switch (mode) {
            case Less:
                return "-";
            case Greater:
                return "+";
            case LessOrEqual:
                return "-0";
            case GreaterOrEqual:
                return "0+";
            default:
                throw new IllegalStateException(String.format("Unknown comparable value mode: %s", mode));
        }
    }

    private static List<String> filterToAtomList(TFilterInfoConfig.TBasicFilter basicFilter) {
        var filter = basicFilter.getFilter();
        if (filter.hasListValue()) {
            return filter.getListValue().getValueList()
                    .stream()
                    .map(value -> String.format("%s:%s", HotelSearchUtils.getGeoSearchBusinessId(filter), value))
                    .collect(Collectors.toUnmodifiableList());
        } else if (filter.hasComparableValue()) {
            var comparableValue = filter.getComparableValue();
            return Collections.singletonList(String.format("%s:%.2f%s", HotelSearchUtils.getGeoSearchBusinessId(filter),
                    comparableValue.getValue(),
                    convertEModeToString(comparableValue.getMode())));
        } else if (filter.hasIgnoredValue()) {
            return Collections.emptyList();
        } else {
            throw new IllegalStateException(String.format("Found filter with no value (basic filter id: %s)",
                    basicFilter.getFilter().getUniqueId()));
        }
    }
}
