package ru.yandex.travel.api.endpoints.hotels_portal;

import javax.servlet.http.HttpServletRequest;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.misc.io.http.HttpException;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelErrorRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddHotelReviewBodyV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddHotelReviewReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddHotelReviewRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewImagesReqBodyV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewImagesReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewImagesRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DoesCityStaticPageExistReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DoesCityStaticPageExistRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.EditHotelReviewBodyV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.EditHotelReviewReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.EditHotelReviewRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCalendarPricesReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCalendarPricesRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsReqV2;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsReqV3;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsRspV2;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsRspV3;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsOffersReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsOffersRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelImagesReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelImagesRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelInfoReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelInfoRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelOffersReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelOffersRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelReviewsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelReviewsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetLegacyHotelReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetLegacyHotelRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSharedFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSharedFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSimilarHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSimilarHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.LogSuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.LogSuggestRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.RemoveFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.RemoveFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SetHotelReviewReactionReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SetHotelReviewReactionRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.ShareFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.ShareFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.UploadHotelReviewImageReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.UploadHotelReviewImageRspV1;
import ru.yandex.travel.api.exceptions.FavoriteHotelLimitExceededException;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.hotels.common.Permalink;

@RestController
@RequestMapping("/api/hotels_portal")
@Api(value = "HotelsPortal", description = "Hotels Portal Controller")
@RequiredArgsConstructor
public class HotelsPortalController {

    private final ResponseProcessor responseProcessor;
    private final HotelsPortalImpl impl;

    @ExceptionHandler(HotelNotFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<String> handleHotelNotFoundException(HotelNotFoundException e) {
        return ResponseEntity.status(HttpStatus.NOT_FOUND).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(HttpException.class)
    @ResponseStatus(HttpStatus.BAD_GATEWAY)
    public ResponseEntity<String> handleHttpException(HttpException e) {
        return ResponseEntity.status(HttpStatus.BAD_GATEWAY).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(FavoriteHotelLimitExceededException.class)
    @ResponseStatus(HttpStatus.UNPROCESSABLE_ENTITY)
    public ResponseEntity<AddFavoriteHotelErrorRspV1> handleFavoriteHotelLimitExceededException(FavoriteHotelLimitExceededException e) {
        var rsp = new AddFavoriteHotelErrorRspV1();
        rsp.setError(AddFavoriteHotelErrorRspV1.AddToFavoritesError.HOTEL_LIMIT_EXCEEDED);
        return ResponseEntity.status(HttpStatus.UNPROCESSABLE_ENTITY).body(rsp);
    }

    @RequestMapping(value = {"/v1/get_legacy_hotel"}, method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<GetLegacyHotelRspV1> getLegacyHotelV1(@Validated @BindFromQuery GetLegacyHotelReqV1 req)
            throws HotelNotFoundException {
        return responseProcessor.replyWithFuture( "HotelsPortalGetLegacyHotelV1", () -> impl.getLegacyHotelById(req.getId()));
    }

    @RequestMapping(value = {"/v1/get_hotel_raw"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Информация про отель в сыром виде")
    public DeferredResult<String> getHotelRawV1(@RequestParam(name = "permalink") @ApiParam(value = "Permalink отеля", required = true) Permalink permalink) {
        return responseProcessor.replyWithFuture( "HotelsPortalGetHotelRawV1", () -> impl.getHotelRaw(permalink, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/get_hotel_reviews"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Отзывы про отель")
    public DeferredResult<GetHotelReviewsRspV1> getHotelReviewsV1(@Validated @BindFromQuery GetHotelReviewsReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalGetHotelReviewsV1", () -> impl.getHotelReviews(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/set_hotel_review_reaction"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Реакция на отзывы про отель")
    public DeferredResult<SetHotelReviewReactionRspV1> setHotelReviewReactionV1(@Validated @BindFromQuery SetHotelReviewReactionReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalSetHotelReviewReactionV1", () -> impl.setHotelReviewReaction(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/upload_hotel_review_image"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Загрузка фотографий отзывов про отель")
    public DeferredResult<UploadHotelReviewImageRspV1> uploadHotelReviewImageV1(@Validated @BindFromQuery UploadHotelReviewImageReqV1 req, @RequestParam("image") MultipartFile image) {
        return responseProcessor.replyWithFuture( "UploadHotelReviewImageV1", () -> impl.uploadHotelReviewImage(req, image, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/delete_hotel_review_images"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Удаление фотографий отзывов про отель")
    public DeferredResult<DeleteHotelReviewImagesRspV1> deleteHotelReviewImagesV1(@Validated @BindFromQuery DeleteHotelReviewImagesReqV1 req, @RequestBody @Validated DeleteHotelReviewImagesReqBodyV1 body) {
        return responseProcessor.replyWithFuture( "DeleteHotelReviewImagesV1", () -> impl.deleteHotelReviewImages(req, body, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/add_hotel_review"}, method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
    @ApiOperation(value = "Создание отзыва про отель")
    public DeferredResult<AddHotelReviewRspV1> addHotelReviewV1(@Validated @BindFromQuery AddHotelReviewReqV1 req, @RequestBody @Validated AddHotelReviewBodyV1 body) {
        return responseProcessor.replyWithFuture( "AddHotelReviewV1", () -> impl.addHotelReview(req, body, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/edit_hotel_review"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Редактирование отзыва про отель")
    public DeferredResult<EditHotelReviewRspV1> editHotelReviewV1(@Validated @BindFromQuery EditHotelReviewReqV1 req, @RequestBody @Validated EditHotelReviewBodyV1 body) {
        return responseProcessor.replyWithFuture( "EditHotelReviewV1", () -> impl.editHotelReview(req, body, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/delete_hotel_review"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Удаление отзыва про отель")
    public DeferredResult<DeleteHotelReviewRspV1> deleteHotelReviewV1(@Validated @BindFromQuery DeleteHotelReviewReqV1 req) {
        return responseProcessor.replyWithFuture( "DeleteHotelReviewV1", () -> impl.deleteHotelReview(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/get_hotel_images"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Фотографии отеля")
    public DeferredResult<GetHotelImagesRspV1> getHotelImagesV1(@Validated @BindFromQuery GetHotelImagesReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalGetHotelImagesV1", () -> impl.getHotelImages(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/get_hotel_info"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Информация про отель")
    public DeferredResult<GetHotelInfoRspV1> getHotelInfoV1(@Validated @BindFromQuery GetHotelInfoReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalGetHotelInfoV1", () -> impl.getHotelInfo(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/get_hotel_offers"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Информация про офферы отеля")
    public DeferredResult<GetHotelOffersRspV1> getHotelOffersV1(@Validated @BindFromQuery GetHotelOffersReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalGetHotelOffersV1", () -> impl.getHotelOffers(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/get_similar_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Информация про похожие отели")
    public DeferredResult<GetSimilarHotelsRspV1> getSimilarHotelsV1(@Validated @BindFromQuery GetSimilarHotelsReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalGetSimilarHotelsV1", () -> impl.getSimilarHotels(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/search_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Поиск отелей в регионе")
    public DeferredResult<SearchHotelsRspV1> searchHotelsV1(@Validated @BindFromQuery SearchHotelsReqV1 req, HttpServletRequest httpServletRequest) {
        return responseProcessor.replyWithFuture( "HotelsPortalSearchHotelsV1", () -> impl.searchHotels(req, httpServletRequest, CommonHttpHeaders.get(), UserCredentials.get()));
    }
    @RequestMapping(value = {"/v1/count_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Поиск отелей в регионе - считалочка")
    public DeferredResult<CountHotelsRspV1> countHotelsV1(@Validated @BindFromQuery CountHotelsReqV1 req) {
        return responseProcessor.replyWithFuture( "HotelsPortalCountHotelsV1", () -> impl.countHotels(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/suggest"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Поисковые подсказки")
    public DeferredResult<SuggestRspV1> suggestV1(@Validated @BindFromQuery SuggestReqV1 req) {
        return responseProcessor.replyWithFuture("HotelsPortalSuggestV1", () -> impl.suggest(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/log_suggest"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Логирование поисковой подсказки, выбранной пользователем")
    public DeferredResult<LogSuggestRspV1> logSuggestV1(@Validated @BindFromQuery LogSuggestReqV1 req) {
        return responseProcessor.replyWithFuture("HotelsPortalLogSuggestV1", () -> impl.logSuggest(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/does_city_static_page_exist"}, method = RequestMethod.GET)
    @ApiOperation(value = "Определение, существует ли лендинг по слагу")
    public DeferredResult<DoesCityStaticPageExistRspV1> doesCityStaticPageExist(@Validated @BindFromQuery DoesCityStaticPageExistReqV1 req) {
        return responseProcessor.replyWithFuture("DoesCityStaticPageExist", () -> impl.doesCityStaticPageExist(req));
    }

    @RequestMapping(value = {"/v1/city_static_page"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение данных для рендеринга статической страницы города")
    public DeferredResult<GetCityStaticPageRspV1> getCityStaticPageV1(@Validated @BindFromQuery GetCityStaticPageReqV1 req) {
        return responseProcessor.replyWithFuture("GetCityStaticPageV1", () -> impl.getCityStaticPage(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/add_favorite_hotel"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Добавление отеля в избранное")
    public DeferredResult<AddFavoriteHotelRspV1> addFavoriteHotelV1(@Validated @BindFromQuery AddFavoriteHotelReqV1 req) {
        return responseProcessor.replyWithFuture("AddFavoriteHotelV1", () -> impl.addFavoriteHotel(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/remove_favorite_hotels"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Удаление отелей из избранного")
    public DeferredResult<RemoveFavoriteHotelsRspV1> removeFavoriteHotelsV1(@Validated @BindFromQuery RemoveFavoriteHotelsReqV1 req) {
        return responseProcessor.replyWithFuture("RemoveFavoriteHotelsV1", () -> impl.removeFavoriteHotels(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/share_favorite_hotels"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Построение данных для возможности поделиться избранным")
    public DeferredResult<ShareFavoriteHotelsRspV1> shareFavoriteHotelsV1(@Validated @BindFromQuery ShareFavoriteHotelsReqV1 req) {
        return responseProcessor.replyWithFuture("ShareFavoriteHotelsV1", () -> impl.shareFavoriteHotels(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/get_favorite_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение избранных отелей")
    public DeferredResult<GetFavoriteHotelsRspV1> getFavoriteHotelsV1(@Validated @BindFromQuery GetFavoriteHotelsReqV1 req) {
        return responseProcessor.replyWithFuture("ShareFavoriteHotelsV1", () -> impl.getFavoriteHotels(req, CommonHttpHeaders.get()));
    }

    @RequestMapping(value = {"/v1/get_shared_favorite_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение избранных отелей")
    public DeferredResult<GetSharedFavoriteHotelsRspV1> getSharedFavoriteHotelsV1(@Validated @BindFromQuery GetSharedFavoriteHotelsReqV1 req) {
        return responseProcessor.replyWithFuture("GetSharedFavoriteHotelsV1", () -> impl.getSharedFavoriteHotels(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/get_favorite_hotels_offers"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Поллинг офферов избранных отелей")
    public DeferredResult<GetFavoriteHotelsOffersRspV1> getFavoriteHotelsOffersV1(@Validated @BindFromQuery GetFavoriteHotelsOffersReqV1 req) {
        return responseProcessor.replyWithFuture("GetFavoriteHotelsOffersV1", () -> impl.getFavoriteHotelsOffers(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/get_cross_sale_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение отелей для кросс-сейла на транспортных вертикалях")
    public DeferredResult<GetCrossSaleHotelsRspV1> getCrossSaleHotelsV1(@Validated @BindFromQuery GetCrossSaleHotelsReqV1 req) {
        return responseProcessor.replyWithFuture("GetCrossSaleHotelsV1", () -> impl.getCrossSaleHotels(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    //todo (liinda): оторвать после RASPTICKETS-22061
    @RequestMapping(value = {"/v2/get_cross_sale_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение отелей для кросс-сейла на транспортных вертикалях с поллингом")
    public DeferredResult<GetCrossSaleHotelsRspV2> getCrossSaleHotelsV2(@Validated @BindFromQuery GetCrossSaleHotelsReqV2 req) {
        return responseProcessor.replyWithFuture("GetCrossSaleHotelsV2", () -> impl.getCrossSaleHotels(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v3/get_cross_sale_hotels"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение отелей для кросс-сейла на транспортных вертикалях с поллингом")
    public DeferredResult<GetCrossSaleHotelsRspV3> getCrossSaleHotelsV3(@Validated @BindFromQuery GetCrossSaleHotelsReqV3 req) {
        return responseProcessor.replyWithFuture("GetCrossSaleHotelsV3", () -> impl.getCrossSaleHotels(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }

    @RequestMapping(value = {"/v1/get_calendar_prices"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение списка цен для отеля")
    public DeferredResult<GetCalendarPricesRspV1> getCalendarPricesV1(@Validated @BindFromQuery GetCalendarPricesReqV1 req) {
        return responseProcessor.replyWithFuture("GetCalendarPricesV1", () -> impl.getCalendarPrices(req, CommonHttpHeaders.get(), UserCredentials.get()));
    }
}
