package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.multipart.MultipartFile;
import yandex.maps.proto.search.business.Business;

import ru.yandex.travel.api.endpoints.hotels_portal.breadcrumbs.BreadcrumbsService;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddFavoriteHotelRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddHotelReviewBodyV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddHotelReviewReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.AddHotelReviewRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewImagesReqBodyV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewImagesReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewImagesRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DeleteHotelReviewRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DoesCityStaticPageExistReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DoesCityStaticPageExistRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.EditHotelReviewBodyV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.EditHotelReviewReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.EditHotelReviewRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCalendarPricesReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCalendarPricesRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsReqV2;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsReqV3;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsRspV2;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCrossSaleHotelsRspV3;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsOffersReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsOffersRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelImagesReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelImagesRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelInfoReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelInfoRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelOffersReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelOffersRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelReviewsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetHotelReviewsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetLegacyHotelRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSharedFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSharedFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSimilarHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetSimilarHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.LogSuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.LogSuggestRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.RemoveFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.RemoveFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SetHotelReviewReactionReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SetHotelReviewReactionRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.ShareFavoriteHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.ShareFavoriteHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.UploadHotelReviewImageReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.UploadHotelReviewImageRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.MetricUtils;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.CrossSaleHotelsBlock;
import ru.yandex.travel.api.models.hotels.Hotel;
import ru.yandex.travel.api.models.hotels.HotelOffer;
import ru.yandex.travel.api.models.hotels.HotelRatings;
import ru.yandex.travel.api.models.hotels.HotelReviewsInfo;
import ru.yandex.travel.api.models.hotels.HotelWithOffers;
import ru.yandex.travel.api.models.hotels.OfferCacheMetadata;
import ru.yandex.travel.api.models.hotels.OffersPollingInfo;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.RegionSearchHotelsRequestData;
import ru.yandex.travel.api.models.hotels.interfaces.DebugOfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.RequestAttributionProvider;
import ru.yandex.travel.api.models.hotels.seo.HotelSchemaOrgInfo;
import ru.yandex.travel.api.models.hotels.seo.OpenGraphInfo;
import ru.yandex.travel.api.models.hotels.seo.SeoInfo;
import ru.yandex.travel.api.services.crosslinks.CrosslinksService;
import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.geo.CrossSearchPointProvider;
import ru.yandex.travel.api.services.geo.model.PointId;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.calendar_prices.CalendarPricesSearchContext;
import ru.yandex.travel.api.services.hotels.calendar_prices.CalendarPricesService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.geocounter.GeoCounterService;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterReq;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.legacy.LegacyHotelInfo;
import ru.yandex.travel.api.services.hotels.legacy.LegacyHotelsService;
import ru.yandex.travel.api.services.hotels.min_prices.MinMaxPricesService;
import ru.yandex.travel.api.services.hotels.min_prices.RegionMinPricesService;
import ru.yandex.travel.api.services.hotels.region_images.RegionImagesService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.search_log.SearchLogItem;
import ru.yandex.travel.api.services.hotels.search_log.SearchLogger;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.static_pages.RegionHotelsSearcher;
import ru.yandex.travel.api.services.hotels.static_pages.RegionPagesService;
import ru.yandex.travel.api.services.hotels.tugc.TugcService;
import ru.yandex.travel.api.services.hotels.tugc.model.HotelFavoriteInfosRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDigestRsp;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.api.services.seo.SeoExperimentsDataProvider;
import ru.yandex.travel.api.services.seo.SeoExperimentsElement;
import ru.yandex.travel.api.services.seo.SeoExperimentsOption;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.common.token.TokenCodec;
import ru.yandex.travel.hotels.common.token.TravelToken;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelUgcFeatures;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchSingleHotelRsp;
import ru.yandex.travel.hotels.geosearch.model.OfferCacheRequestParams;
import ru.yandex.travel.hotels.offercache.api.EOfferDeduplicationMode;
import ru.yandex.travel.hotels.offercache.api.TReadReq;
import ru.yandex.travel.hotels.offercache.api.TReadResp;
import ru.yandex.travel.hotels.proto.region_pages.EGeoSearchSortType;
import ru.yandex.travel.http.ReqAnsLoggerInterceptor;

@Component
@RequiredArgsConstructor
@EnableConfigurationProperties(HotelsPortalProperties.class)
@Slf4j
public class HotelsPortalImpl {

    private final static String CROSS_SALE_REGION_IMAGE_SIZE = "region-desktop";
    private final static String HOTEL_DIRECT_SEARCH_REASON = "hotelDirectSearch";
    private static final String locale = "ru";
    private static final String domain = "ru";

    private final HotelsPortalProperties config;
    private final LocalizationService localizationService;
    private final GeoSearchService geoSearch;
    private final ObjectMapper mapper;
    private final HotelsSearchService hotelsSearchService;
    private final GeoCounterService geoCounter;
    private final HotelsFilteringService hotelsFilteringService;
    private final SuggestService suggestService;
    private final BreadcrumbsService breadcrumbsService;
    private final LegacyHotelsService legacyHotelService;
    private final TokenCodec tokenCodec;
    private final AmenityService amenityService;
    private final GeoBase geoBase;
    private final HotelSlugService hotelSlugService;
    private final RegionPagesService regionPagesService;
    private final TugcService tugcService;
    private final FavoritesService favoritesService;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final ReviewService reviewService;
    private final HotelImagesService hotelImagesService;
    private final RegionHotelsSearcher regionHotelsSearcher;
    private final RegionsService regionsService;
    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final RegionMinPricesService regionMinPricesService;
    private final RegionImagesService regionImagesService;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;
    private final MinMaxPricesService minMaxPricesService;
    private final CrossSearchPointProvider pointProvider;
    private final CrosslinksService crosslinksService;
    private final SearchLogger searchLogger;
    private final CalendarPricesService calendarPricesService;
    private final SeoExperimentsDataProvider seoExperimentsDataProvider;

    private final Counter unknownHotelCategoryCounter =
            Counter.builder("hotels.unknownCategory").register(Metrics.globalRegistry);

    private CompletableFuture<GetLegacyHotelRspV1> getLegacyHotel(Long id, LegacyHotelInfo lhi) {
        if (lhi == null) {
            return CompletableFuture.failedFuture(new HotelNotFoundException());
        }
        CompletableFuture<GeoHotel> geoHotelFuture;
        if (lhi.getPermalink() != null) {
            geoHotelFuture = legacyHotelService.getGeoInfoByPermalink(lhi.getPermalink());
        } else {
            log.info("No permalink for hotel {}, unable to return prices and features", id);
            geoHotelFuture = CompletableFuture.completedFuture(null);
        }
        return geoHotelFuture.handle((geoHotel, thr) -> {
            if (thr != null) {
                log.error("Failed to fetch old hotel info for permalink {}", lhi.getPermalink(), thr);
            }
            Double cachedMinPrice = null;
            List<String> activeBooleanFeatureNames = Collections.emptyList();
            if (geoHotel != null && geoHotel.getOfferCacheResponse() != null) {
                if (geoHotel.getOfferCacheResponse().getPricesCount() > 0) {
                    cachedMinPrice = (double) geoHotel.getOfferCacheResponse().getPrices(0).getPrice();
                }
                activeBooleanFeatureNames = legacyHotelService.getActiveBooleanTopFeatureNames(geoHotel);
            }
            String oid = String.format("b:%s", lhi.getPermalink());
            String query = String.format("%s - %s", lhi.getName(), lhi.getAddress());
            String url = HotelsPortalUtils.getSerpUrl(query, null, oid, "travel-portal",
                    "legacy-hotel-page-redirect");
            GetLegacyHotelRspV1.Price price = new GetLegacyHotelRspV1.Price();
            price.setCurrency(GetLegacyHotelRspV1.Currency.RUB);
            if (cachedMinPrice != null) {
                price.setAmount(cachedMinPrice);
                price.setType(GetLegacyHotelRspV1.PriceType.MINIMUM);// TODO remove it !!!
            } else if (lhi.getMedianPrice() != null) {
                price.setAmount(lhi.getMedianPrice());
                price.setType(GetLegacyHotelRspV1.PriceType.MEDIAN);
            } else {
                price = null;
            }
            String hotelSlug = null;
            if (lhi.getPermalink() != null) {
                hotelSlug = hotelSlugService.findMainSlugByPermalink(lhi.getPermalink());
            }
            return new GetLegacyHotelRspV1(id, lhi.getName(), lhi.getAddress(), lhi.getCountry(),
                    lhi.getImage(),
                    lhi.getPermalink(),
                    hotelSlug,
                    price, lhi.getStars(), lhi.getRating(), url, activeBooleanFeatureNames);
        });
    }

    public CompletableFuture<GetLegacyHotelRspV1> getLegacyHotelById(Long id) throws HotelNotFoundException {
        return getLegacyHotel(id, legacyHotelService.getStaticHotelInfo(id));
    }

    public CompletableFuture<String> getHotelRaw(Permalink permalink, CommonHttpHeaders headers) {
        GeoOriginEnum origin = getHotelPageGeoOrigin(headers);
        GeoSearchReq p = GeoSearchReq.byPermalink(origin, permalink, headers);
        p.setIncludeSpravPhotos(true);
        p.setIncludeUgcAnswers(true);
        p.setIncludeSimilarHotels(true);
        p.setIncludeOfferCache(true);
        p.setIncludeCategoryIds(true);
        p.setIncludeRating(true);
        p.setIncludePhotos(true);
        p.setIncludeNearByStops(true);
        p.setIncludeFeatureGroups(true);
        return geoSearch.querySingleHotel(p).thenApply(geoHotel -> {
            try {
                return mapper.writeValueAsString(geoHotel);
            } catch (Exception e) {
                return null;
            }
        });
    }

    public CompletableFuture<GetHotelReviewsRspV1> getHotelReviews(GetHotelReviewsReqV1 req, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("getHotelReviews", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            String reqId = getRequestId(headers);

            return reviewService.getDigest(permalink, reqId, req.getTextReviewLimit(), req.getTextReviewOffset(),
                    req.getKeyPhraseFilter(), req.getTextReviewRanking(), config.getMaxTextReviewOffset(), req.isEnabledTestUgc()
            ).thenApply(ugcDigestRsp -> {
                GetHotelReviewsRspV1 rsp = new GetHotelReviewsRspV1();
                rsp.setReviewsInfo(ReviewExtractor.extractHotelReviewsInfo(ugcDigestRsp, permalink, reqId,
                        req.getKeyPhraseLimit(), req.getTextReviewLimit(),
                        req.getTextReviewOffset(), config.getMaxTextReviewOffset(), req.getKeyPhraseFilter()));
                return rsp;
            });
        });
    }

    public CompletableFuture<SetHotelReviewReactionRspV1> setHotelReviewReaction(SetHotelReviewReactionReqV1 req, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("setHotelReviewReaction", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);

            return reviewService.setReviewReaction(permalink, req.getReviewId(), req.getUserReaction(),
                    getRequestId(headers), req.isEnabledTestUgc()
            ).thenApply(ugcRsp -> new SetHotelReviewReactionRspV1());
        });
    }

    public CompletableFuture<UploadHotelReviewImageRspV1> uploadHotelReviewImage(UploadHotelReviewImageReqV1 req, MultipartFile image, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("uploadHotelReviewImage", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);

            return reviewService.uploadImage(image, permalink, getRequestId(headers), req.isEnabledTestUgc())
                .thenApply(ugcRsp -> {
                    UploadHotelReviewImageRspV1 rsp = new UploadHotelReviewImageRspV1();
                    rsp.setImageId(ugcRsp.getId());
                    rsp.setSizes(ReviewExtractor.extractUploadedHotelReviewImageSizes(ugcRsp.getSizes().getSize()));

                    return rsp;
                });
        });
    }

    public CompletableFuture<DeleteHotelReviewImagesRspV1> deleteHotelReviewImages(DeleteHotelReviewImagesReqV1 req, DeleteHotelReviewImagesReqBodyV1 body, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("deleteHotelReviewImages", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            List<String> imageIds = body.getImageIds();

            if (imageIds == null || imageIds.isEmpty()) {
                return CompletableFuture.completedFuture(new DeleteHotelReviewImagesRspV1());
            }

            return reviewService.deleteImages(imageIds, permalink, getRequestId(headers), req.isEnabledTestUgc())
                    .thenApply(ugcRsp -> new DeleteHotelReviewImagesRspV1());
        });
    }

    public CompletableFuture<AddHotelReviewRspV1> addHotelReview(AddHotelReviewReqV1 req, AddHotelReviewBodyV1 body, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("addHotelReview", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);

            return changeHotelReview(permalink, null, body.getReviewText(), body.getReviewRating(),
                    body.getReviewImageIds(), getRequestId(headers), req.isEnabledTestUgc()
            ).thenApply(textReview -> {
                AddHotelReviewRspV1 rsp = new AddHotelReviewRspV1();
                rsp.setTextReview(textReview);

                return rsp;
            });
        });
    }

    public CompletableFuture<EditHotelReviewRspV1> editHotelReview(EditHotelReviewReqV1 req, EditHotelReviewBodyV1 body, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("editHotelReview", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);

            return changeHotelReview(permalink, req.getReviewId(), body.getReviewText(), body.getReviewRating(),
                    body.getReviewImageIds(), getRequestId(headers), req.isEnabledTestUgc()
            ).thenApply(textReview -> {
                EditHotelReviewRspV1 rsp = new EditHotelReviewRspV1();
                rsp.setTextReview(textReview);

                return rsp;
            });
        });
    }

    private CompletableFuture<HotelReviewsInfo.TextReview> changeHotelReview(Permalink permalink, String reviewId, String reviewText, int reviewRating,
                                                                             List<String> reviewImageIds, String requestId, boolean isEnabledTestUgc) {
        return reviewService.changeReview(permalink, reviewId, reviewText, reviewRating, reviewImageIds, requestId, isEnabledTestUgc)
                .thenApply(ugcRsp -> ReviewExtractor.extractHotelReview(ugcRsp.getReview(), permalink, requestId, true, null));
    }

    public CompletableFuture<DeleteHotelReviewRspV1> deleteHotelReview(DeleteHotelReviewReqV1 req, CommonHttpHeaders headers) {
        return HotelsPortalUtils.handleExceptions("deleteHotelReview", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);

            return reviewService.deleteReview(permalink, req.getReviewId(), getRequestId(headers), req.isEnabledTestUgc())
                    .thenApply(textReview -> new DeleteHotelReviewRspV1());
        });
    }

    public CompletableFuture<GetHotelImagesRspV1> getHotelImages(GetHotelImagesReqV1 req, CommonHttpHeaders headers) {
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        return HotelsPortalUtils.handleExceptions("getHotelImages", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            GeoOriginEnum origin = getHotelPageGeoOrigin(headers);
            GeoSearchReq p = GeoSearchReq.byPermalink(origin, permalink, headers);
            p.setIncludeSpravPhotos(true);
            p.setIncludePhotos(true);
            p.setParentReqId(req.getParentRequestId());
            return geoSearch.querySingleHotel(p).thenApply(geoRsp -> {
                GeoHotel geoHotel = geoRsp.getHotel();
                GetHotelImagesRspV1 rsp = new GetHotelImagesRspV1();
                var imgInfo = HotelsPortalUtils.extractHotelImages(
                        geoHotel, req, hotelImagesService, uaasSearchExperiments, imageWhitelistDataProvider
                );
                rsp.setImages(imgInfo.images);
                rsp.setTotalImageCount(imgInfo.totalImageCount);
                return rsp;
            });
        });
    }

    // TODO(alexcrush): rework for new amenities (main & grouped)
    public HotelRatings extractHotelRatings(GeoHotel geoHotel) {
        HotelRatings result = new HotelRatings();
        List<HotelRatings.FeatureRating> featureRatings = new ArrayList<>();
        result.setFeatureRatings(featureRatings);
        if (geoHotel.getUgcFeatures() == null) {
            return result;
        }
        Set<String> amenityIds = geoHotel.getGeoObjectMetadata().getFeatureList().stream()
                // Only use features, which are not boolean, or boolean with true value
                .filter(feature -> !feature.getValue().hasBooleanValue() || feature.getValue().getBooleanValue())
                .map(Business.Feature::getId).collect(Collectors.toSet());
        HotelRatings.FeatureRating teaserFeature = null;
        int teaserPositivePercent = 0;
        for (GeoHotelUgcFeatures.Feature geoFeature : geoHotel.getUgcFeatures().getFeatureList()) {
            HotelsPortalProperties.HotelRatingConfig ratingConfig = config.getRatings().get(geoFeature.getFeatureId());
            if (ratingConfig == null) {
                continue;
            }
            if (ratingConfig.getForAmenity() != null && !ratingConfig.getForAmenity().isEmpty()) {
                boolean found = false;
                for (String am : ratingConfig.getForAmenity()) {
                    if (amenityIds.contains(am)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    continue;
                }
            }
            int denom = geoFeature.getStat().getNegative() + geoFeature.getStat().getPositive();
            if (denom <= 0) {
                continue;
            }
            HotelRatings.FeatureRating feature = new HotelRatings.FeatureRating();
            featureRatings.add(feature);
            feature.setId(geoFeature.getFeatureId());
            feature.setName(ratingConfig.getRatingText());
            int positivePercent = 100 * geoFeature.getStat().getPositive() / denom;
            feature.setPositivePercent(positivePercent);
            if (teaserFeature == null || positivePercent > teaserPositivePercent) {
                teaserFeature = feature;
                teaserPositivePercent = positivePercent;
            }
        }
        featureRatings.sort((x, y) -> Integer.compare(y.getPositivePercent(), x.getPositivePercent()));
        if (teaserFeature != null) {
            HotelsPortalProperties.HotelRatingConfig ratingConfig = config.getRatings().get(teaserFeature.getId());
            result.setTeaser(String.format(ratingConfig.getTeaserTextTemplate(), teaserPositivePercent));
        }
        return result;
    }

    public CompletableFuture<GetHotelInfoRspV1> getHotelInfo(GetHotelInfoReqV1 req, CommonHttpHeaders headers, UserCredentials userCredentials) {
        final String locale = "ru";
        return HotelsPortalUtils.handleExceptions("getHotelInfo", () -> {
            var experiments = new Experiments(req, config);
            var hasSearchParams = req.hasSearchParams() || !config.isAllowNoSearchParamsInHotelInfo() || req.isUseDefaultSearchParams();
            var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);

            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            String requestId = getRequestId(headers);
            CompletableFuture<UgcDigestRsp> ugcFuture = reviewService.getDigest(permalink, requestId, req.getTextReviewLimit(), 0,
                    req.getKeyPhraseFilter(), req.getTextReviewRanking(), config.getMaxTextReviewOffset(), req.isEnabledTestUgc());

            CompletableFuture<HotelFavoriteInfosRsp> tugcFuture;
            if (HotelsPortalUtils.needLikes(headers, userCredentials)) {
                tugcFuture = tugcService.getHotelFavoriteInfos("", userCredentials, List.of(permalink));
            } else {
                tugcFuture = CompletableFuture.completedFuture(null);
            }

            extractBoYToken(req.getToken()); // Just check that we can parse it
            GeoSearchReq geoSearchReq = GeoSearchReq.byPermalink(getHotelPageGeoOrigin(headers), permalink, headers);
            geoSearchReq.setIncludeOfferCache(true);
            geoSearchReq.setIncludeRating(true);
            geoSearchReq.setIncludeSpravPhotos(true);
            geoSearchReq.setIncludeSimilarHotels(true);
            geoSearchReq.setIncludePhotos(true);
            geoSearchReq.setIncludeNearByStops(true);
            geoSearchReq.setIncludeUgcAnswers(true);// колбаски
            geoSearchReq.setIncludeFeatureGroups(true);
            geoSearchReq.setIncludeCategoryIds(true);

            geoSearchReq.setOfferCacheRequestParams(generateOfferCacheRequestParams(req, headers, userCredentials, req, req,
                    req.getLabel(), req.getSearchPagePollingId(),false, "get_hotel_info", req.getDomain(),
                    req.isEnablePermarooms(), req.isEnableDeduplication(), uaasSearchExperiments, hasSearchParams));
            geoSearchReq.setUseProdOfferCache(getUseProdOfferCache(req));

            CompletableFuture<GeoSearchSingleHotelRsp> geoSearchFuture = geoSearch.querySingleHotel(geoSearchReq);

            return CompletableFuture.allOf(geoSearchFuture, ugcFuture, tugcFuture).handle((ignored, ignoredT) -> {
                GeoSearchSingleHotelRsp geoRsp = geoSearchFuture.join();

                checkExistenceOfHotel(geoRsp, req);

                UgcDigestRsp ugcDigestRsp = tryGetUgcDigestRsp(ugcFuture);
                HotelFavoriteInfosRsp tugcRsp = tryGetTugcRsp(tugcFuture);
                var isFavorite = tugcRsp != null && tugcRsp.getIsFavorite().getOrDefault(permalink, false);

                if (ugcDigestRsp == null) {
                    return extractHotelRsp(geoRsp, req, geoSearchReq.isUseProdOfferCache(),
                            requestId, experiments, geoSearchReq.getFilterCategories(), null,
                            0, locale, isFavorite, uaasSearchExperiments, hasSearchParams);
                }

                HotelReviewsInfo reviewsInfo = ReviewExtractor.extractHotelReviewsInfo(ugcDigestRsp, permalink, requestId,
                        req.getKeyPhraseLimit(), req.getTextReviewLimit(), req.getTextReviewOffset(),
                        config.getMaxTextReviewOffset(), req.getKeyPhraseFilter());
                int totalTextReviewCount = reviewsInfo.getTotalTextReviewCount();

                return extractHotelRsp(geoRsp, req, geoSearchReq.isUseProdOfferCache(), requestId,
                        experiments, geoSearchReq.getFilterCategories(), reviewsInfo, totalTextReviewCount, locale,
                        isFavorite, uaasSearchExperiments, hasSearchParams);
            });
        }).thenApply(response -> {
            if (response != null && response.getHotel() != null) {
                logGetHotelInfo(headers, response);
            }
            return response;
        });
    }

    private void logGetHotelInfo(CommonHttpHeaders headers, GetHotelInfoRspV1 response) {
        try {
            searchLogger.log(
                    SearchLogItem.builder()
                            .permalink(response.getHotel().getPermalink().toString())
                            .offerSearchParams(response.getSearchParams())
                            .searchReason(HOTEL_DIRECT_SEARCH_REASON)
                            .foundHotelCount(1)
                            .build(),
                    Instant.now(),
                    headers
            );
        } catch (Exception exc) {
            log.error("logGetHotelInfo failed -> ", exc);
        }
    }

    private String getRequestId(CommonHttpHeaders headers) {
        String reqId = headers.getRequestId();

        return reqId != null ? reqId : UUID.randomUUID().toString();
    }

    private UgcDigestRsp tryGetUgcDigestRsp(CompletableFuture<UgcDigestRsp> ugcFuture) {
        try {
            return ugcFuture.join();
        } catch (Exception exc) {
            log.error("UGC request failed -> no reviews ", exc);
            return null;
        }
    }

    private HotelFavoriteInfosRsp tryGetTugcRsp(CompletableFuture<HotelFavoriteInfosRsp> tugcFuture) {
        try {
            return tugcFuture.join();
        } catch (Exception exc) {
            log.error("TUGC request failed -> no likes", exc);
            return null;
        }
    }

    private void checkExistenceOfHotel(GeoSearchSingleHotelRsp geoRsp, GetHotelInfoReqV1 req) {
        GeoHotel geoHotel = geoRsp.getHotel();

        // If we do not have OC response (=> No ytravel_* signals in permalink
        // And rubric is not known => return 404
        if (geoRsp.getOfferCacheResponseMetadata() == null &&
                geoHotel.getCategoryIds().stream().noneMatch(config.getHotelCategoriesToShow()::contains)) {
            // request filter doesn't work together with filter by business, so use post-filtering
            unknownHotelCategoryCounter.increment();
            throw new HotelNotFoundException(String.format(
                    "GetHotelInfo: By permalink %s got hotel %s with non-hotel categories %s",
                    req.getPermalink(), geoHotel.getGeoObjectMetadata().getId(), geoHotel.getCategoryIds()));
        }
        if (config.getBannedPermalinksToShow().contains(geoHotel.getPermalink().toString())) {
            // note that this check is done AFTER geosearch request to check cluster permalink
            throw new HotelNotFoundException(String.format(
                    "GetHotelInfo: Permalink %s is banned",
                    req.getPermalink()));
        }
    }

    private GetHotelInfoRspV1 extractHotelRsp(GeoSearchSingleHotelRsp geoRsp, GetHotelInfoReqV1 req,
                                              boolean withProdOfferCache,
                                              String requestId, Experiments experiments, List<String> filterCategories,
                                              HotelReviewsInfo reviewsInfo, int totalTextReviewCount, String locale,
                                              boolean isFavorite,
                                              UaasSearchExperiments uaasSearchExperiments, boolean hasSearchParams) {
        GeoHotel geoHotel = geoRsp.getHotel();
        OfferCacheMetadata ocMeta = OfferCacheMetadata.extract(geoRsp.getOfferCacheResponseMetadata(),
                req, withProdOfferCache);
        GetHotelInfoRspV1 rsp = new GetHotelInfoRspV1();
        // searchParams
        if (hasSearchParams) {
            rsp.setSearchParams(ocMeta.getOfferSearchParams());
        }

        //parentRequestId
        rsp.setParentRequestId(requestId);

        //hotel
        HotelsPortalUtils.HotelInfo hotelInfo = HotelsPortalUtils.extractHotel(amenityService, hotelSlugService, hotelImagesService, req,
                geoHotel, req, false, filterCategories, experiments,
                config.getHotelCategoriesToShow(), false, isFavorite, uaasSearchExperiments,
                imageWhitelistDataProvider, true);
        Hotel rspHotel = hotelInfo.getHotel();
        rsp.setHotel(rspHotel);

        // Review info
        rsp.setReviewsInfo(reviewsInfo);
        rsp.getHotel().setTotalTextReviewCount(totalTextReviewCount);

        // hotelDescription - not filled

        rsp.setBreadcrumbs(breadcrumbsService.getHotelBreadcrumbs(req.getDomain(), rspHotel.getCoordinates()));
        rsp.setOffersInfo(HotelsPortalUtils.extractHotelOffersInfo(ocMeta, geoHotel, experiments, hasSearchParams,
                uaasSearchExperiments, config.getOfferBadgePriorities(), config));
        rsp.setSimilarHotelsInfo(HotelsPortalUtils.extractSimilarHotelsInfo(amenityService, hotelSlugService,
                ocMeta, geoRsp, req.getSimilarHotelLimit(), experiments, hasSearchParams, config.getOfferBadgePriorities(), config));
        rsp.setRatingsInfo(extractHotelRatings(geoHotel));

        // SeoInfo
        String mainImageUrl = null;
        if (!rspHotel.getImages().isEmpty()) {
            mainImageUrl = String.format(rspHotel.getImages().get(0).getUrlTemplate(), "orig");
        }

        TReadResp.THotel ocResponse = geoHotel.getOfferCacheResponse();
        boolean hotelIsBoy = ocResponse != null && ocResponse.hasHasBoyPartner() && ocResponse.getHasBoyPartner();

        String seoTitle;
        var seoTitleOption = seoExperimentsDataProvider.getExpReplacement(rspHotel.getHotelSlug(), SeoExperimentsElement.SEO_TITLE);
        if (seoTitleOption == SeoExperimentsOption.WITH_MIN_PRICE) {
            var hotelMinPrice = minMaxPricesService.getMinPriceByPermalink(geoHotel.getPermalink().asLong());
            seoTitle = HotelsPortalUtils.getHotelSeoTitleExpMinPrice(localizationService, geoBase, rspHotel.getName(),
                    rspHotel.getCategory().getId(), rspHotel.getCategory().getName(),
                    rspHotel.getCoordinates().getLat(), rspHotel.getCoordinates().getLon(), locale,
                    req.getDomain(), hotelMinPrice);
        } else {
            seoTitle = HotelsPortalUtils.getHotelSeoTitle(localizationService, geoBase, rspHotel.getName(),
                    rspHotel.getCategory().getId(), rspHotel.getCategory().getName(),
                    rspHotel.getCoordinates().getLat(), rspHotel.getCoordinates().getLon(), locale,
                    req.getDomain(), hotelIsBoy);
        }
        String seoDescription = HotelsPortalUtils.getHotelSeoDescription(
                localizationService, rspHotel.getName(), locale, hotelIsBoy, config.getPlusDefaultPercentDiscount()
        );
        SeoInfo<HotelSchemaOrgInfo> seoInfo = new SeoInfo<>();
        rsp.setSeoInfo(seoInfo);
        seoInfo.setTitle(seoTitle);
        seoInfo.setDescription(seoDescription);
        OpenGraphInfo ogInfo = new OpenGraphInfo();
        seoInfo.setOpenGraph(ogInfo);
        ogInfo.setTitle(seoTitle);
        ogInfo.setDescription(seoDescription);
        ogInfo.setImage(mainImageUrl);
        HotelSchemaOrgInfo schemaOrgInfo = new HotelSchemaOrgInfo();
        seoInfo.setSchemaOrg(schemaOrgInfo);
        schemaOrgInfo.setName(rspHotel.getName());
        schemaOrgInfo.setImage(mainImageUrl);
        String template = localizationService.getLocalizedValueWithoutDefault("SeoHotelPriceRangeTemplate", locale);
        if (hasSearchParams) {
            if (geoHotel.getOfferCacheResponse() != null && geoHotel.getOfferCacheResponse().hasPriceRange() && template != null) {
                schemaOrgInfo.setPriceRange(String.format(template,
                        geoHotel.getOfferCacheResponse().getPriceRange().getMinPrice(),
                        geoHotel.getOfferCacheResponse().getPriceRange().getMaxPrice()));

            }
        } else {
            var minPrice = minMaxPricesService.getMinPriceByPermalink(geoHotel.getPermalink().asLong());
            var maxPrice = minMaxPricesService.getMaxPriceByPermalink(geoHotel.getPermalink().asLong());
            if (minPrice.isPresent() && maxPrice.isPresent() && template != null) {
                schemaOrgInfo.setPriceRange(String.format(template, minPrice.getAsInt(), maxPrice.getAsInt()));
            }
        }
        schemaOrgInfo.setAddress(rspHotel.getAddress());
        schemaOrgInfo.setRatingValue(rspHotel.getRating());
        schemaOrgInfo.setReviewCount(rspHotel.getTotalTextReviewCount());

        // seoBreadcrumbs
        rsp.setSeoBreadcrumbs(breadcrumbsService.getHotelSeoBreadcrumbs(
                req.getDomain(),
                rspHotel.getCoordinates(),
                rspHotel.getName(),
                rspHotel.getHotelSlug()
        ));

        Map<String, String> pageParams = ExtraVisitAndUserParamsUtils.initParamsMap();
        ExtraVisitAndUserParamsUtils.fillPermalink(pageParams, rspHotel.getPermalink().asLong());
        ExtraVisitAndUserParamsUtils.fillBoolean(pageParams, "imagesMayBeChangedByWatermark", hotelInfo.isImagesMayBeChangedByWatermark());
        rsp.setExtraVisitAndUserParams(ExtraVisitAndUserParamsUtils.createForHotelPage(pageParams));

        return rsp;
    }

    private TravelToken extractBoYToken(String token) {
        if (token == null) {
            return null;
        }
        try {
            return tokenCodec.decode(token);
        } catch (Exception exc) {
            throw new TravelApiBadRequestException("Failed to decode token", exc);
        }
    }

    private GeoOriginEnum getHotelPageGeoOrigin(CommonHttpHeaders headers) {
        return HotelsPortalUtils.selectGeoOriginByDevice(headers, GeoOriginEnum.HOTEL_PAGE_DESKTOP,
                GeoOriginEnum.HOTEL_PAGE_TOUCH);
    }

    private OfferCacheRequestParams generateOfferCacheRequestParams(RequestAttributionProvider attribution,
                                                                    CommonHttpHeaders headers,
                                                                    UserCredentials userCredentials,
                                                                    OfferSearchParamsProvider offerSearchParams,
                                                                    DebugOfferSearchParamsProvider debugOfferSearchParams,
                                                                    String labelHash,
                                                                    String searchPagePollingId,
                                                                    boolean allowPastDates,
                                                                    String debugId,
                                                                    String domain,
                                                                    boolean enablePermarooms,
                                                                    boolean enableOfferDeduplication,
                                                                    UaasSearchExperiments uaasSearchExperiments,
                                                                    boolean hasSearchParams) {
        var exps = new Experiments(attribution, config);
        Integer userRegion = HotelsPortalUtils.determineUserRegion(geoBase, headers, attribution);
        TReadReq.Builder ocReqBuilder = HotelsPortalUtils.prepareOfferCacheRequestParams(
                attribution, headers, userCredentials, userRegion, offerSearchParams, debugOfferSearchParams,
                labelHash, searchPagePollingId, true, exps, uaasSearchExperiments, debugId, hasSearchParams,
                config.isSortOffersUsingPlus());
        ocReqBuilder.setFull(true);
        ocReqBuilder.setOfferDeduplicationMode(enableOfferDeduplication ? EOfferDeduplicationMode.ODM_ByPermaroom : EOfferDeduplicationMode.ODM_None);
        ocReqBuilder.setAllowPastDates(allowPastDates);
        ocReqBuilder.setEnableCatRoom(enablePermarooms);
        if (HotelsPortalUtils.isCatRoomForAllPartners(uaasSearchExperiments, exps)) {
            ocReqBuilder.setAutoCatRoomForAll(true);
            ocReqBuilder.setCatRoomMaxOtherPct(25);
            ocReqBuilder.setCatRoomShowOther(false);
        } else {
            ocReqBuilder.setAutoCatRoomOnlyForBoY(true);
        }
        ocReqBuilder.setUseNewCatRoom(true);
        ocReqBuilder.setShowPermaroomsWithNoOffers(true);
        ocReqBuilder.setAdjustDefaultSubKeyForMirPromo(true);
        // Не хотим показывать "прочие" офферы
        // ocReqBuilder.setCatRoomShowOther(false); - пока отключено, приводит к расхождению офферов между Full=0 & 1
        ocReqBuilder.setUseBnovoRooms(true);
        return OfferCacheRequestParams.build(ocReqBuilder);
    }

    private boolean getUseProdOfferCache(DebugOfferSearchParamsProvider req) {
        return config.isEnableDebugParams() && req.isDebugUseProdOffers();
    }

    public CompletableFuture<GetHotelOffersRspV1> getHotelOffers(GetHotelOffersReqV1 req, CommonHttpHeaders headers,
                                                                 UserCredentials userCredentials) {
        return HotelsPortalUtils.handleExceptions("getHotelOffers", () -> {
            var experiments = new Experiments(req, config);
            var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            extractBoYToken(req.getToken()); // Just check that we can parse it

            GeoSearchReq p = GeoSearchReq.byPermalink(getHotelPageGeoOrigin(headers), permalink, headers);
            p.setIncludeOfferCache(true);
            p.setIncludeSpravPhotos(true);
            p.setParentReqId(req.getParentRequestId());
            p.setOfferCacheRequestParams(generateOfferCacheRequestParams(req, headers, userCredentials, req, req,
                    req.getLabel(), req.getSearchPagePollingId(), true, "get_hotel_offers", req.getDomain(),
                    req.isEnablePermarooms(), req.isEnableDeduplication(), uaasSearchExperiments, true));
            p.setUseProdOfferCache(getUseProdOfferCache(req));

            return geoSearch.querySingleHotel(p).thenApply(geoRsp -> {
                OfferCacheMetadata ocMeta = OfferCacheMetadata.extract(geoRsp.getOfferCacheResponseMetadata(),
                        req, p.isUseProdOfferCache());
                GetHotelOffersRspV1 rsp = new GetHotelOffersRspV1();

                rsp.setOffersInfo(HotelsPortalUtils.extractHotelOffersInfo(ocMeta, geoRsp.getHotel(), experiments,
                        true, uaasSearchExperiments, config.getOfferBadgePriorities(), config));

                var pageParams = ExtraVisitAndUserParamsUtils.initParamsMap();
                ExtraVisitAndUserParamsUtils.fillPermalink(pageParams, geoRsp.getHotel().getPermalink().asLong());
                ExtraVisitAndUserParamsUtils.fillHotelRefs(pageParams, ocMeta, geoRsp.getHotel());
                ExtraVisitAndUserParamsUtils.fillHotelOffers(pageParams, ocMeta, geoRsp.getHotel());
                ExtraVisitAndUserParamsUtils.fillSearchParams(pageParams, ocMeta.getOfferSearchParams());

                Coordinates coordinates = HotelsPortalUtils.extractHotelCoordinates(geoRsp.getHotel());
                Integer geoId = null;
                if (coordinates != null) {
                    geoId = GeoBaseHelpers.getRegionIdByLocationOrNull(geoBase, coordinates.getLat(), coordinates.getLon());
                }
                ExtraVisitAndUserParamsUtils.fillGeoParams(pageParams, req.getDomain(), geoId, geoBase);
                rsp.setExtraVisitAndUserParams(ExtraVisitAndUserParamsUtils.createForHotelPage(pageParams));
                return rsp;
            });
        });
    }

    public CompletableFuture<GetSimilarHotelsRspV1> getSimilarHotels(GetSimilarHotelsReqV1 req,
                                                                     CommonHttpHeaders headers,
                                                                     UserCredentials userCredentials) {
        return HotelsPortalUtils.handleExceptions("getSimilarHotels", () -> {
            var experiments = new Experiments(req, config);
            var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            GeoOriginEnum origin = getHotelPageGeoOrigin(headers);
            GeoSearchReq p = GeoSearchReq.byPermalink(origin, permalink, headers);
            p.setIncludeOfferCache(true);
            p.setParentReqId(req.getParentRequestId());
            p.setIncludeSimilarHotels(true);
            p.setIncludePhotos(true);
            p.setUseProdOfferCache(getUseProdOfferCache(req));
            p.setOfferCacheRequestParams(generateOfferCacheRequestParams(req, headers, userCredentials, req, req,
                    req.getSearchPagePollingId(), null, true, "get_similar_hotels", req.getDomain(),
                    false, true, uaasSearchExperiments, true));

            return geoSearch.querySingleHotel(p).thenApply(geoRsp -> {
                OfferCacheMetadata ocMeta = OfferCacheMetadata.extract(geoRsp.getOfferCacheResponseMetadata(),
                        req, p.isUseProdOfferCache());

                GetSimilarHotelsRspV1 rsp = new GetSimilarHotelsRspV1();
                rsp.setSimilarHotelsInfo(HotelsPortalUtils.extractSimilarHotelsInfo(amenityService, hotelSlugService,
                        ocMeta, geoRsp, req.getSimilarHotelLimit(), experiments, true, config.getOfferBadgePriorities(), config));
                return rsp;
            });
        });
    }

    public CompletableFuture<SearchHotelsRspV1> searchHotels(SearchHotelsReqV1 req, HttpServletRequest httpServletRequest, CommonHttpHeaders headers, UserCredentials userCredentials) {
        var additionalSearchHotelsLogData = new AdditionalSearchHotelsLogData();
        httpServletRequest.setAttribute(ReqAnsLoggerInterceptor.ADDITIONAL_INFO_ATTR_NAME, additionalSearchHotelsLogData);
        MetricUtils.addRequestMetricTag(httpServletRequest, "is_robot_request", Boolean.toString(HotelsPortalUtils.isRobotRequest(headers)), true);
        return hotelsSearchService.search(httpServletRequest, req, headers, userCredentials, additionalSearchHotelsLogData)
                .thenApply(response -> {
                    if (response != null) {
                        logSearchHotels(req, headers, response);
                    }
                    return response;
                });
    }

    private void logSearchHotels(SearchHotelsReqV1 req, CommonHttpHeaders headers, SearchHotelsRspV1 response) {
        try {
            var logBuilder = SearchLogItem.builder();
            if (req.getStartSearchReason() != null) {
                logBuilder.searchReason(req.getStartSearchReason().toString());
            }
            if (req.getGeoId() != null && req.getGeoId() != 0) {
                logBuilder.geoId(req.getGeoId());
            }

            searchLogger.log(
                    logBuilder
                            .offerSearchParams(response.getOfferSearchParams())
                            .foundHotelCount(response.getFoundHotelCount())
                            .build(),
                    Instant.now(),
                    headers
            );
        } catch (Exception exc) {
            log.error("logSearchHotels failed -> ", exc);
        }
    }

    public CompletableFuture<CountHotelsRspV1> countHotels(CountHotelsReqV1 req, CommonHttpHeaders headers) {
        var now = Instant.now();
        var experiments = new Experiments(req, config);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        GeoCounterReq gcReq = hotelsFilteringService.prepareGeoCounterReq(req, req, req.getBbox(),
                experiments.isExp("4813"), now,
                HotelSearchUtils.getForceFilterLayout(req), null, headers);
        gcReq.setNameSuffix("countHotels");
        return geoCounter.getCounts(gcReq).handle((gcRsp, exception) -> {
            if (exception != null) {
                log.error("Failed to get response from geocounter", exception);
                var rsp = new CountHotelsRspV1();
                rsp.setFoundHotelCount(0);
                rsp.setSearchControlInfo(hotelsFilteringService.composeDefaultControlInfo(req, now,
                        HotelSearchUtils.getForceFilterLayout(req), null, headers));
                rsp.setFilterInfo(hotelsFilteringService.composeDefaultFilterInfo(req, now,
                        HotelSearchUtils.getForceFilterLayout(req), null, false, headers));
                SortTypeRegistry sortTypeRegistry = SortTypeRegistry.getSortTypeRegistry(
                        HotelsPortalUtils.isHotelsNearbyEnabled(headers),
                        HotelsPortalUtils.getSortTypeLayout(uaasSearchExperiments),
                        uaasSearchExperiments.isNewRanking(),
                        uaasSearchExperiments.isPersonalRanking() || experiments.isExp("personal-ranking"),
                        HotelsPortalUtils.getExplorationRankingCategory(uaasSearchExperiments, experiments),
                        HotelsPortalUtils.getRealTimeRanking(uaasSearchExperiments, experiments, false));
                rsp.setSortInfo(HotelSearchUtils.buildSortInfo(sortTypeRegistry, sortTypeRegistry.getDefaultSortType().getId(), null));
                return rsp;
            } else {
                return hotelsFilteringService.composeCountHotelsRsp(req, req, gcRsp, now,
                        HotelSearchUtils.getForceFilterLayout(req), headers, experiments, req.getSelectedSortId(),
                        req.getSortOrigin(), null, false);
            }
        });
    }

    public CompletableFuture<SuggestRspV1> suggest(SuggestReqV1 req, CommonHttpHeaders headers) {
        return suggestService.processSuggestRequest(req, headers);
    }

    public CompletableFuture<LogSuggestRspV1> logSuggest(LogSuggestReqV1 req, CommonHttpHeaders headers) {
        return suggestService.processLogSuggestRequest(req, headers);
    }

    public CompletableFuture<DoesCityStaticPageExistRspV1> doesCityStaticPageExist(DoesCityStaticPageExistReqV1 req) {
        return regionPagesService.doesCityStaticPageExist(req);
    }

    public CompletableFuture<GetCityStaticPageRspV1> getCityStaticPage(GetCityStaticPageReqV1 req,
                                                                       CommonHttpHeaders headers,
                                                                       UserCredentials userCredentials) {
        return regionPagesService.getRegionStaticPage(req, headers, userCredentials);
    }

    public CompletableFuture<AddFavoriteHotelRspV1> addFavoriteHotel(AddFavoriteHotelReqV1 req, CommonHttpHeaders commonHttpHeaders) {
        checkRequiredYandexUid();
        return favoritesService.addFavoriteHotel(req,commonHttpHeaders, UserCredentials.get());
    }

    public CompletableFuture<RemoveFavoriteHotelsRspV1> removeFavoriteHotels(RemoveFavoriteHotelsReqV1 req, CommonHttpHeaders commonHttpHeaders) {
        checkRequiredYandexUid();
        return favoritesService.removeFavoriteHotels(req, UserCredentials.get());
    }

    public CompletableFuture<ShareFavoriteHotelsRspV1> shareFavoriteHotels(ShareFavoriteHotelsReqV1 req, CommonHttpHeaders commonHttpHeaders) {
        checkRequiredYandexUid();
        return favoritesService.shareFavoriteHotels(req, UserCredentials.get());
    }

    public CompletableFuture<GetFavoriteHotelsRspV1> getFavoriteHotels(GetFavoriteHotelsReqV1 req, CommonHttpHeaders commonHttpHeaders) {
        checkRequiredYandexUid();
        return favoritesService.getFavoriteHotels(req, commonHttpHeaders, UserCredentials.get());
    }

    public CompletableFuture<GetSharedFavoriteHotelsRspV1> getSharedFavoriteHotels(GetSharedFavoriteHotelsReqV1 req, CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        return favoritesService.getSharedFavoriteHotels(req, commonHttpHeaders, userCredentials);
    }

    public CompletableFuture<GetFavoriteHotelsOffersRspV1> getFavoriteHotelsOffers(GetFavoriteHotelsOffersReqV1 req, CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        return favoritesService.getFavoriteHotelsOffers(req, commonHttpHeaders, userCredentials);
    }

    public CompletableFuture<GetCrossSaleHotelsRspV1> getCrossSaleHotels(GetCrossSaleHotelsReqV1 req,
                                                                         CommonHttpHeaders commonHttpHeaders,
                                                                         UserCredentials userCredentials) {
        int geoId = getSettlementGeoId(req.getSettlementId());
        if (geoId <= 0) {
            return CompletableFuture.completedFuture(new GetCrossSaleHotelsRspV1());
        }

        return crosslinksService.getHotelsBlock(
                geoId, req.getTotalHotelLimit(), req.getDomain(), commonHttpHeaders, userCredentials
        ).thenApply(x -> new GetCrossSaleHotelsRspV1(x.getData()));
    }

    public CompletableFuture<GetCrossSaleHotelsRspV2> getCrossSaleHotels(GetCrossSaleHotelsReqV2 req,
                                                                         CommonHttpHeaders commonHttpHeaders,
                                                                         UserCredentials userCredentials) {
        int geoId = getSettlementGeoId(req.getSettlementId());
        if (geoId <= 0) {
            return CompletableFuture.completedFuture(new GetCrossSaleHotelsRspV2());
        }

        var requestData = new RegionSearchHotelsRequestData();
        requestData.setGeoId(geoId);
        requestData.setLimit(req.getTotalHotelLimit());
        requestData.setSortType(EGeoSearchSortType.CHEAP_FIRST);
        requestData.setCheckinDate(req.getCheckinDate());
        requestData.setCheckoutDate(req.getCheckoutDate());
        requestData.setAdults(req.getAdults());
        requestData.setChildrenAges(req.getChildrenAges());

        return regionHotelsSearcher.searchHotels(geoId, requestData, commonHttpHeaders, userCredentials,
                "CrossSaleWithOffers").thenApply(result -> prepareCrossSaleRsp(req, geoId, result));
    }

    public CompletableFuture<GetCrossSaleHotelsRspV3> getCrossSaleHotels(GetCrossSaleHotelsReqV3 req,
                                                                         CommonHttpHeaders commonHttpHeaders,
                                                                         UserCredentials userCredentials) {

        int geoId = getSettlementGeoIdByPointKey(req.getPointKey(), locale, domain);

        var requestData = new RegionSearchHotelsRequestData();
        requestData.setGeoId(geoId);
        requestData.setLimit(req.getTotalHotelLimit());
        requestData.setSortType(EGeoSearchSortType.CHEAP_FIRST);
        requestData.setCheckinDate(req.getCheckinDate());
        requestData.setCheckoutDate(req.getCheckoutDate());
        requestData.setAdults(req.getAdults());
        requestData.setChildrenAges(req.getChildrenAges());

        return regionHotelsSearcher.searchHotels(geoId, requestData, commonHttpHeaders, userCredentials,
                "CrossSaleWithOffers").thenApply(result -> prepareCrossSaleRsp(req, geoId, result));
    }

    private GetCrossSaleHotelsRspV2 prepareCrossSaleRsp(GetCrossSaleHotelsReqV2 req, int geoId,
                                                        RegionHotelsSearcher.SearchResult searchResult) {
        var hotelsWithOffers = searchResult
                .getHotels()
                .stream()
                .map(RegionHotelsSearcher.SearchResultHotel::getHotelWithOffers)
                .filter(hotel -> hotel.getOffers().size() > 0)
                .collect(Collectors.toList());

        if (hotelsWithOffers.size() == 0) {
            return new GetCrossSaleHotelsRspV2();
        }

        var rsp = new GetCrossSaleHotelsRspV2();
        rsp.setHasData(true);
        rsp.setOfferSearchProgress(searchResult.getOcMeta().getProgress());
        if (!searchResult.getOcMeta().getProgress().isFinished()) {
            rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, new Experiments(req, config)));
        }

        var region = regionsService.getRegion(geoId, req.getDomain(), "ru");
        rsp.setRegion(region);

        var minPrice = hotelsWithOffers.stream()
                .map(HotelWithOffers::getOffers)
                .flatMap(Collection::stream)
                .map(HotelOffer::getPrice)
                .min(Comparator.comparing(Price::getValue));

        if (minPrice.isEmpty()) {
            return new GetCrossSaleHotelsRspV2();
        }
        rsp.setMinPriceInRegion(minPrice.get());

        var bbox = HotelsPortalUtils.getBBoxByHotels(geoId, req.getDomain(), geoBase,
                hotelsWithOffers.stream()
                        .map(HotelWithOffers::getHotel)
                        .collect(Collectors.toList()));
        rsp.setBboxAsString(bbox);
        rsp.setBboxAsStruct(List.of(bbox.getLeftDown(), bbox.getUpRight()));

        rsp.setHotels(hotelsWithOffers.stream()
                .sorted(Comparator.comparing(this::getCrossSaleHotelPriority))
                .collect(Collectors.toList()));

        rsp.setTotalHotelCount(searchResult.getTotalHotelsFound());
        rsp.setRegionImageUrl(regionImagesService.getImageUsingTreeWithSize(geoId, CROSS_SALE_REGION_IMAGE_SIZE));
        return rsp;
    }

    private GetCrossSaleHotelsRspV3 prepareCrossSaleRsp(GetCrossSaleHotelsReqV3 req, int geoId,
                                                        RegionHotelsSearcher.SearchResult searchResult) {
        var pollingProgressBuilder = OffersPollingInfo.builder()
                .offerSearchProgress(searchResult.getOcMeta().getProgress());
        if (!searchResult.getOcMeta().getProgress().isFinished()) {
            pollingProgressBuilder = pollingProgressBuilder.nextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, new Experiments(req, config)));
        }

        var rsp = new GetCrossSaleHotelsRspV3();
        rsp.setSearchProgress(pollingProgressBuilder.build());

        var hotelsWithOffers = searchResult
                .getHotels()
                .stream()
                .map(RegionHotelsSearcher.SearchResultHotel::getHotelWithOffers)
                .filter(hotel -> hotel.getOffers().size() > 0)
                .collect(Collectors.toList());

        if (hotelsWithOffers.size() == 0) {
            return rsp;
        }

        var crossSaleBlock = new CrossSaleHotelsBlock();
        var region = regionsService.getRegion(geoId, req.getDomain(), "ru");
        crossSaleBlock.setRegion(region);

        var minPrice = hotelsWithOffers.stream()
                .map(HotelWithOffers::getOffers)
                .flatMap(Collection::stream)
                .map(HotelOffer::getPrice)
                .min(Comparator.comparing(Price::getValue));

        if (minPrice.isEmpty()) {
            return rsp;
        }
        crossSaleBlock.setMinPriceInRegion(minPrice.get());

        var bbox = HotelsPortalUtils.getBBoxByHotels(geoId, req.getDomain(), geoBase,
                hotelsWithOffers.stream()
                        .map(HotelWithOffers::getHotel)
                        .collect(Collectors.toList()));
        crossSaleBlock.setBboxAsString(bbox);
        crossSaleBlock.setBboxAsStruct(List.of(bbox.getLeftDown(), bbox.getUpRight()));

        crossSaleBlock.setHotels(hotelsWithOffers.stream()
                .sorted(Comparator.comparing(this::getCrossSaleHotelPriority))
                .collect(Collectors.toList()));

        crossSaleBlock.setTotalHotelCount(searchResult.getTotalHotelsFound());
        crossSaleBlock.setRegionImageUrl(regionImagesService.getImageUsingTreeWithSize(geoId, CROSS_SALE_REGION_IMAGE_SIZE));
        rsp.setCrossSale(crossSaleBlock);
        return rsp;
    }

    private Integer getSettlementGeoId(String settlementId) {
        TSettlement settlement;
        try {
            settlement = trainSettlementDataProvider.getById(parseSettlementId(settlementId));
        } catch (NoSuchElementException e) {
            throw new TravelApiBadRequestException(String.format("Unknown settlementId: %s", settlementId));
        }
        return settlement.getGeoId();
    }

    private Integer getSettlementGeoIdByPointKey(String pointKey, String locale, String domain) {
        var pointId = PointId.builder().pointKey(pointKey).build();
        var pointInfo = pointProvider.getByPointKey(pointId, locale, domain);

        if (pointInfo == null || (pointInfo.getGeoId() <= 0)) {
            throw new TravelApiBadRequestException(String.format("Not found settlementGeoId for pointKey=%s", pointKey));
        }
        return pointInfo.getGeoId();
    }

    public int parseSettlementId(String settlementId) {
        TravelPreconditions.checkRequestArgument(settlementId.startsWith("c"),"Invalid settlement_id");
        try {
            return Integer.parseInt(settlementId.substring(1));
        } catch (NumberFormatException | StringIndexOutOfBoundsException e) {
            throw new TravelApiBadRequestException("Invalid settlement_id");
        }
    }

    private Integer getCrossSaleHotelPriority(HotelWithOffers hotelWithOffers) {
        var sortGroup = 2; // pessimize priority for hotels without yandex plus
        if (hotelWithOffers.getOffers().size() == 0) {
            return sortGroup;
        }

        var offer = hotelWithOffers.getOffers().get(0);
        if (offer.getOfferYandexPlusInfo() == null) {
            return sortGroup;
        }

        if (hotelWithOffers.getOffers().get(0).getOfferYandexPlusInfo().getPoints() > 0) {
            sortGroup = 1; // improve priority for hotels with yandex plus
        }
        return sortGroup;
    }

    private void checkRequiredYandexUid() {
        if (UserCredentials.get().getYandexUid() == null) {
            throw new TravelApiBadRequestException("Missing yandex uid");
        }
    }

    public CompletableFuture<GetCalendarPricesRspV1> getCalendarPrices(GetCalendarPricesReqV1 req,
                                                                       CommonHttpHeaders headers,
                                                                       UserCredentials userCredentials) {
        Preconditions.checkState(config.isCalendarPricesEnabled(), "Calendar prices disabled");
        if (!calendarPricesService.acquireThrottler()){
            return CompletableFuture.failedFuture(new HttpServerErrorException(HttpStatus.SERVICE_UNAVAILABLE, "over rate limit"));
        }
        var experiments = new Experiments(req, config);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        var geoOrigin = HotelsPortalUtils.selectGeoOriginByDevice(
                headers, GeoOriginEnum.CALENDAR_DESKTOP, GeoOriginEnum.CALENDAR_TOUCH);

        return HotelsPortalUtils.handleExceptions("getCalendarPrices", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            return calendarPricesService.getCalendarPrices(
                    new CalendarPricesService.BaseCalendarParams(
                            req.getStartDate(), req.getEndDate(), req.getCheckinDate(), req.getCheckoutDate(), permalink
                    ), req, userCredentials, headers, experiments, uaasSearchExperiments, geoOrigin
            ).thenApply(prices -> {
                var rsp = new GetCalendarPricesRspV1();
                rsp.setPrices(prices.getPrices().stream()
                        .map(GetCalendarPricesRspV1.IPriceInfo::new)
                        .collect(Collectors.toList()));
                rsp.setFinished(prices.getAllFinished());
                var context = CalendarPricesSearchContext.Companion.deserialize(req.getContext());
                if (!prices.getAllFinished()) {
                    rsp.setNextPollingRequestDelayMs(
                            calendarPricesService.getNextIterationPollingDelayMs(context, experiments)
                    );
                }
                context.incrementPollingIteration();
                rsp.setContext(context.serialize());
                return rsp;
            }).whenComplete((result, error) -> calendarPricesService.releaseThrottler());
        });
    }
}
