package ru.yandex.travel.api.endpoints.hotels_portal;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.CRC32;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapLikeType;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tags;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.javamoney.moneta.Money;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;
import yandex.maps.proto.common2.geometry.GeometryOuterClass;
import yandex.maps.proto.photos2.Photos2;
import yandex.maps.proto.search.business.Business;
import yandex.maps.proto.search.masstransit_2x.Masstransit2X;
import yandex.maps.proto.search.photos_2x.Photos2X;
import yandex.maps.proto.search.related_places.RelatedPlaces;

import ru.yandex.geobase6.LookupException;
import ru.yandex.geobase6.RegionHash;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.models.Linguistics;
import ru.yandex.travel.api.models.hotels.Badge;
import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.CancellationInfoAggregate;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.Hotel;
import ru.yandex.travel.api.models.hotels.HotelFilter;
import ru.yandex.travel.api.models.hotels.HotelFilterGroup;
import ru.yandex.travel.api.models.hotels.HotelImage;
import ru.yandex.travel.api.models.hotels.HotelOffer;
import ru.yandex.travel.api.models.hotels.HotelOffersInfo;
import ru.yandex.travel.api.models.hotels.HotelWithOffers;
import ru.yandex.travel.api.models.hotels.OfferCacheMetadata;
import ru.yandex.travel.api.models.hotels.OfferSearchProgress;
import ru.yandex.travel.api.models.hotels.OperatorInfo;
import ru.yandex.travel.api.models.hotels.PansionAggregate;
import ru.yandex.travel.api.models.hotels.PermaroomInfo;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.RoomAmenity;
import ru.yandex.travel.api.models.hotels.RoomAmenityGroup;
import ru.yandex.travel.api.models.hotels.RoomArea;
import ru.yandex.travel.api.models.hotels.RoomBedConfigurationItem;
import ru.yandex.travel.api.models.hotels.RoomBedGroup;
import ru.yandex.travel.api.models.hotels.ShortHotelOffersInfo;
import ru.yandex.travel.api.models.hotels.SimilarHotelsInfo;
import ru.yandex.travel.api.models.hotels.interfaces.DebugOfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.HotelIdentifierProvider;
import ru.yandex.travel.api.models.hotels.interfaces.ImageParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.api.models.hotels.interfaces.RequestAttributionProvider;
import ru.yandex.travel.api.models.hotels.interfaces.ShortHotelOffersInfoSetter;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.commons.experiments.KVExperiments;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundType;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelPhoto;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchSingleHotelRsp;
import ru.yandex.travel.hotels.geosearch.model.GeoSimilarHotel;
import ru.yandex.travel.hotels.geosearch.model.OfferCacheRequestParams;
import ru.yandex.travel.hotels.offercache.api.TBadge;
import ru.yandex.travel.hotels.offercache.api.TOCPansion;
import ru.yandex.travel.hotels.offercache.api.TPrice;
import ru.yandex.travel.hotels.offercache.api.TReadReq;
import ru.yandex.travel.hotels.offercache.api.TReadResp;
import ru.yandex.travel.hotels.offercache.api.TRefundRule;
import ru.yandex.travel.hotels.offercache.api.TStrikethroughPrice;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.ERefundType;
import ru.yandex.travel.hotels.proto.THotelImage;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetHotelsResponse;
import ru.yandex.travel.white_label.proto.EWhiteLabelPartnerId;


@Slf4j
public class HotelsPortalUtils {

    private final static int CRC_BYTES = 4;
    private final static Base64.Encoder BASE64_ENCODER = Base64.getUrlEncoder().withoutPadding();
    private final static Pattern SEO_TITLE_CATEGORY_REGEXP = Pattern.compile("\\b(отель|мотель|гостиница|" +
                    "гостиничный|санаторий|дом отдыха|база отдыха|хостел|турбаза|кемпинг|гостевой|гестхаус|пансион|" +
                    "вилла|бунгало|эллинг|комплекс)\\b",
            Pattern.UNICODE_CHARACTER_CLASS);
    private final static Pattern SEO_TITLE_MASCULINE_REGEXP = Pattern.compile("(ий|ый|ой)$",
            Pattern.UNICODE_CHARACTER_CLASS);
    private final static Pattern BOTS_USER_AGENT_PATTERN = Pattern.compile("googlebot|yandexbot|bingbot|SemrushBot" +
            "|AhrefsBot|sindresorhus/got|Mail\\.RU_Bot|petalbot|YaDirectFetcher|AdsBot-Google|YaK|YandexDirect|DuckDuckGo", Pattern.CASE_INSENSITIVE);
    // sindresorhus/got - requests from sprav
    // YaK - "Mozilla/5.0 (compatible; YaK/1.0; http://linkfluence.com/; bot@linkfluence.com)"

    public final static String ROOM_AREA_FEATURE_ID = "room-size-square-meters";
    public final static String MIR_OFFERS_QUICK_FILTER_ID = "mir-offers-quick";
    public final static String MIR_OFFERS_FILTER_ID = "mir-offers";
    public final static String FORCE_OFFER_DISCOUNT_EXP = "force-offer-discount-value-pct";
    public final static int MAX_BADGES_PER_OFFER = 2;
    public final static int MAX_BADGES_PER_HOTEL = 2;
    public final static Map<String, String> LEGACY_FILTERS_REMAPPING = new HashMap<>() {{
        put("hotel_breakfast_included:1", "hotel_pansion_with_offerdata:hotel_pansion_breakfast_included");
    }};
    private final static String USER_DEVICE_TOUCH = "touch";
    private final static String USER_DEVICE_DESKTOP = "desktop";
    private final static Double MIN_VISIBLE_DISCOUNT_PERCENTAGE = 5.00;

    @Data
    public static class FixedImageParamsProvider implements ImageParamsProvider {
        private Integer imageLimit = null;
        private int imageOffset = 0;
        private Set<String> imageSizes = null;
        private boolean onlyTopImages = true;
    }

    @Value
    public static class HotelImagesInfo {
        public List<HotelImage> images;
        public int totalImageCount;
        public boolean imagesMayBeChangedByWhitelist;
    }

    @Value
    public static class HotelInfo {
        public Hotel hotel;
        public boolean imagesMayBeChangedByWatermark;
    }

    public static SimilarHotelsInfo extractSimilarHotelsInfo(AmenityService amenityService,
                                                             HotelSlugService hotelSlugService,
                                                             OfferCacheMetadata ocMeta,
                                                             GeoSearchSingleHotelRsp geoRsp,
                                                             Integer limit,
                                                             Experiments experiments,
                                                             boolean includeOffers,
                                                             Map<String, Integer> offerBadgePriorities,
                                                             HotelsPortalProperties config) {
        SimilarHotelsInfo result = new SimilarHotelsInfo();
        if (geoRsp.getHotel().getSimilarHotels() == null) {
            result.setHotels(Collections.emptyList());
            result.setOperatorById(Collections.emptyMap());
            var progress = new OfferSearchProgress();
            progress.setFinished(true);
            progress.setPartnersComplete(0);
            progress.setPartnersTotal(0);
            progress.setFinishedPartners(Collections.emptyList());
            progress.setPendingPartners(Collections.emptyList());
            result.setOfferSearchProgress(progress);
            return result;
        }

        result.setOperatorById(ocMeta.getOperatorById());
        result.setOfferSearchProgress(ocMeta.getProgress());
        if (!ocMeta.getProgress().isFinished()) {
            result.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, experiments));
        }
        List<HotelWithOffers> hotelsWithOffers = new ArrayList<>();
        result.setHotels(hotelsWithOffers);
        for (GeoSimilarHotel geoSimilarHotel : geoRsp.getHotel().getSimilarHotels()) {
            if (limit != null && hotelsWithOffers.size() >= limit) {
                break;
            }
            HotelWithOffers hotelWithOffers = new HotelWithOffers();
            hotelsWithOffers.add(hotelWithOffers);
            Hotel hotel = new Hotel();
            hotelWithOffers.setHotel(hotel);

            RelatedPlaces.PlaceInfo pi = geoSimilarHotel.getPlaceInfo();
            hotel.setPermalink(geoSimilarHotel.getPermalink());
            hotel.setHotelSlug(hotelSlugService.findMainSlugByPermalink(hotel.getPermalink()));
            hotel.setName(pi.getName());
            Hotel.Category category = new Hotel.Category();
            category.setName(pi.getCategory());
            // оу, айдишника нема
            hotel.setCategory(category);
            if (pi.hasPoint()) {
                Coordinates coordinates = new Coordinates();
                coordinates.setLat(pi.getPoint().getLat());
                coordinates.setLon(pi.getPoint().getLon());
                hotel.setCoordinates(coordinates);
            }
            hotel.setAddress(pi.getAddress());
            List<HotelImage> images = new ArrayList<>();
            hotel.setImages(images);
            if (pi.hasPhotoUrlTemplate()) {
                HotelImage image = new HotelImage();
                images.add(image);
                image.setUrlTemplate(pi.getPhotoUrlTemplate());
                // Больше ничего нет, сорри
            }
            amenityService.fillForSimilarHotel(hotel, geoSimilarHotel);
            hotel.setRating(geoSimilarHotel.getRatingAt0To5Scale());
            if (geoSimilarHotel.getExtension() != null) {
                hotel.setTotalTextReviewCount(geoSimilarHotel.getExtension().getReviewCount());
            }

            hotelWithOffers.setSearchIsFinished(result.getOfferSearchProgress().isFinished());
            if (includeOffers) {
                hotelWithOffers.setOffers(extractHotelOffers(ocMeta, geoSimilarHotel.getOfferCacheResponse(),
                        offerBadgePriorities, experiments));
            }
        }
        return result;
    }

    public static HotelImage extractHotelImage(GeoHotelPhoto geoPhoto, Set<String> allowedSizes) {
        HotelImage image = new HotelImage();
        image.setUrlTemplate(geoPhoto.getPhoto().getUrlTemplate());
        image.setId(geoPhoto.getBase().getId());
        image.setTags(geoPhoto.getPhoto().getTagList());
        List<HotelImage.Size> sizes = new LinkedList<>();
        image.setSizes(sizes);
        for (Photos2.Entry.Image geoImage : geoPhoto.getPhoto().getImageList()) {
            if (allowedSizes != null && !allowedSizes.contains(geoImage.getSize())) {
                continue;
            }
            HotelImage.Size size = new HotelImage.Size();
            size.setSize(geoImage.getSize());
            size.setWidth(geoImage.getWidth());
            size.setHeight(geoImage.getHeight());
            sizes.add(size);
        }
        return image;
    }

    private static List<HotelImage> getImagesForPermaroom(TReadResp.TPermaroom permaroom) {
        return permaroom.getPhotosList().stream().map(photo -> {
            var image = new HotelImage();
            image.setUrlTemplate(photo.getUrlTemplate());
            var urlParts = photo.getUrlTemplate().split("/");
            image.setId(urlParts[4] + '-' + urlParts[5]);
            image.setSizes(photo.getSizesList().stream().map(photoSize -> {
                var size = new HotelImage.Size();
                size.setSize(photoSize.getSize());
                size.setHeight(photoSize.getHeight());
                size.setWidth(photoSize.getWidth());
                return size;
            }).collect(Collectors.toUnmodifiableList()));
            return image;
        }).collect(Collectors.toUnmodifiableList());
    }

    private static List<HotelOffer> extractHotelOffers(OfferCacheMetadata ocMeta,
                                                       TReadResp.THotel protoHotel,
                                                       Map<String, Integer> offerBadgePriorities,
                                                       Experiments experiments) {
        List<HotelOffer> allOffers = new LinkedList<>();
        if (protoHotel != null) {
            for (TPrice protoPrice : protoHotel.getPricesList()) {
                allOffers.add(extractOffer(ocMeta, protoPrice, offerBadgePriorities, experiments));
            }
        }
        return allOffers;
    }

    @Data
    @AllArgsConstructor
    static class AmenityInfo {
        boolean hasTopFeatureImportance;
        double topFeatureImportance;
        String category;
        String categoryName;
        RoomAmenity roomAmenity;
    }

    @Data
    @AllArgsConstructor
    private static class OfferAggregates {
        PansionAggregate pansionAggregate;
        CancellationInfoAggregate cancellationInfoAggregate;
    }

    public static OfferAggregates getOfferAggregates(List<HotelOffer> offers) {
        var hasPansion = offers.stream()
                .map(x -> {
                    try {
                        return x.getMealType() == null ? TOCPansion.EPansion.UNKNOWN : TOCPansion.EPansion.valueOf(x.getMealType().getId());
                    } catch (IllegalArgumentException e) {
                        log.error("Unknown pansion type '{}' in offer '{}'", x.getMealType().getId(), x.getId(), e);
                        return TOCPansion.EPansion.UNKNOWN;
                    }
                })
                .anyMatch(x -> x != TOCPansion.EPansion.RO && x != TOCPansion.EPansion.UNKNOWN);

        var hasFreeCancellation = offers.stream()
                .anyMatch(x -> x.getCancellationInfo() != null && x.getCancellationInfo().getRefundType() == RefundType.FULLY_REFUNDABLE);

        var hasRefundableWithPenalty = offers.stream()
                .anyMatch(x -> x.getCancellationInfo() != null && x.getCancellationInfo().getRefundType() == RefundType.REFUNDABLE_WITH_PENALTY);

        if (hasPansion && hasFreeCancellation) {
            return new OfferAggregates(PansionAggregate.PANSION_AVAILABLE, CancellationInfoAggregate.FULLY_REFUNDABLE_AVAILABLE);
        }

        if (hasPansion) {
            return new OfferAggregates(PansionAggregate.PANSION_AVAILABLE, null);
        }

        if (hasFreeCancellation) {
            return new OfferAggregates(null, CancellationInfoAggregate.FULLY_REFUNDABLE_AVAILABLE);
        }

        if (hasRefundableWithPenalty) {
            return new OfferAggregates(null, CancellationInfoAggregate.REFUNDABLE_WITH_PENALTY_AVAILABLE);
        }

        return new OfferAggregates(null, CancellationInfoAggregate.NON_REFUNDABLE_AVAILABLE);
    }

    public static HotelOffersInfo extractHotelOffersInfo(OfferCacheMetadata ocMeta,
                                                         GeoHotel geoHotel,
                                                         Experiments experiments,
                                                         boolean includeOffers,
                                                         UaasSearchExperiments uaasSearchExperiments,
                                                         Map<String, Integer> offerBadgePriorities,
                                                         HotelsPortalProperties config) {
        TReadResp.THotel protoHotel = geoHotel.getOfferCacheResponse();
        HotelOffersInfo result = new HotelOffersInfo();

        if (protoHotel != null && protoHotel.getShowPermarooms()) {
            result.setGroupBy(HotelOffersInfo.OfferGroupType.ROOMS);
            ArrayList<PermaroomInfo> rooms = new ArrayList<>();
            result.setRooms(rooms);

            for (TReadResp.TPermaroom permaroom : protoHotel.getPermaroomsList()) {
                PermaroomInfo permaroomInfo = new PermaroomInfo();
                permaroomInfo.setId(permaroom.getId());
                permaroomInfo.setName(permaroom.getName());
                permaroomInfo.setImages(getImagesForPermaroom(permaroom));
                permaroomInfo.setDescription(permaroom.getDescription());
                var amenityInfoGroups = extractGrouppedAmenityInfo(extractRoomAmenities(permaroom));
                permaroomInfo.setMainAmenities(exractMainAmenities(amenityInfoGroups, config.getMaxMainAmenities()));
                permaroomInfo.setAmenityGroups(extractRoomAmenitiesGroupped(amenityInfoGroups));
                permaroomInfo.setBedGroups(extractRoomBedGroups(permaroom));

                var areaFeature = permaroom.getIntegerFeaturesList().stream().filter(x -> x.getId().equals(ROOM_AREA_FEATURE_ID)).findFirst();
                if (areaFeature.isPresent()) {
                    permaroomInfo.setArea(new RoomArea(areaFeature.get().getValue(), RoomArea.RoomAreaUnits.SQUARE_METERS));
                }

                permaroomInfo.setBadges(permaroom.getBadgesList().stream()
                        .map(BadgeUtils::extractBadge)
                        .collect(Collectors.toUnmodifiableList()));
                rooms.add(permaroomInfo);
            }
        }

        result.setOfferSearchProgress(ocMeta.getProgress());
        if (!includeOffers) {
            result.setBannerType(HotelOffersInfo.BannerType.NONE);
            result.setOperatorById(ocMeta.getOperatorById());
            return result;
        }

        if (!ocMeta.getProgress().isFinished()) {
            result.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMsWithoutIteration(config, experiments));
        }
        var extractedOffers = extractHotelOffers(ocMeta, protoHotel, offerBadgePriorities, experiments);

        if (HotelsPortalUtils.isCatRoomForAllPartners(uaasSearchExperiments, experiments) && result.getRooms() != null) {
            extractedOffers
                    .stream()
                    .filter(x -> !x.isYandexOffer())
                    .forEach(offer -> offer.setDiscountInfo(null));
            result.setMainOffers(extractedOffers);
            result.setPartnerOffers(List.of());
        } else {
            result.setMainOffers(extractedOffers.stream()
                    .filter(HotelOffer::isYandexOffer)
                    .collect(Collectors.toUnmodifiableList())
            );
            result.setPartnerOffers(extractedOffers
                    .stream()
                    .filter(x -> !x.isYandexOffer())
                    .collect(Collectors.groupingBy(HotelOffer::getOperatorId))
                    .values()
                    .stream()
                    .map(offers -> {
                        var aggregates = getOfferAggregates(offers);
                        var offer = offers.get(0);
                        HotelOffer.MealType defaultOfferMealType = offer.getMealType();
                        TOCPansion.EPansion mt = null;
                        if (defaultOfferMealType != null) {
                            try {
                                mt = TOCPansion.EPansion.valueOf(defaultOfferMealType.getId());
                            } catch (IllegalArgumentException e) {
                                log.error("Unknown pansion type '{}' in offer '{}'", offer.getMealType().getId(), offer.getId(), e);
                                mt = TOCPansion.EPansion.UNKNOWN;
                            }
                            if (mt == TOCPansion.EPansion.RO || mt == TOCPansion.EPansion.UNKNOWN) {
                                defaultOfferMealType = null;
                            }
                        }

                        HotelOffer.CancellationInfo defaultOfferCancellationInfo = offer.getCancellationInfo();
                        if (defaultOfferCancellationInfo != null && defaultOfferCancellationInfo.getRefundType() != RefundType.FULLY_REFUNDABLE) {
                            defaultOfferCancellationInfo = null;
                        }

                        if (!result.getMainOffers().isEmpty()) {
                            offer.setDiscountInfo(null);
                        }
                        return new HotelOffersInfo.HotelPartnerOffersInfo(
                                offer.getOperatorId(),
                                aggregates.pansionAggregate,
                                aggregates.cancellationInfoAggregate,
                                offer,
                                defaultOfferMealType,
                                defaultOfferCancellationInfo);
                    })
                    .sorted(Comparator.comparingInt(x -> x.getDefaultOffer().getPrice().getValue()))
                    .collect(Collectors.toUnmodifiableList()));
        }

        if (!extractedOffers.isEmpty()) {
            var allPriceValues = Stream.concat(result.getMainOffers().stream(), result.getPartnerOffers().stream().map(x -> x.getDefaultOffer()))
                    .map(x -> x.getPrice().getValue())
                    .collect(Collectors.toUnmodifiableList());
            var currencies = extractedOffers.stream().map(x -> x.getPrice().getCurrency()).distinct().collect(Collectors.toUnmodifiableList());
            if (currencies.size() > 1) {
                log.error("Found different currencies in offers ({}) for permalink {}", currencies, geoHotel.getPermalink());
            }
            var aggregates = getOfferAggregates(extractedOffers);
            result.setAggregatedOfferInfo(new HotelOffersInfo.AggregatedOfferInfo(
                    new Price(Collections.min(allPriceValues), currencies.get(0)),
                    new Price(Collections.max(allPriceValues), currencies.get(0)),
                    aggregates.pansionAggregate,
                    aggregates.cancellationInfoAggregate
            ));
        }

        if (extractedOffers.stream().anyMatch(x -> x.getBadges() != null && x.getBadges().stream().anyMatch(badge -> badge.getId().equals("mir_cashback")))) {
            result.setBannerType(HotelOffersInfo.BannerType.MIR_CASHBACK);
        } else if (protoHotel != null && protoHotel.getSuggestMirPromoDateChange()) {
            result.setBannerType(HotelOffersInfo.BannerType.POSSIBLE_MIR_CASHBACK);
        } else if (extractedOffers.stream().anyMatch(x -> x.getBadges() != null && x.getBadges().stream().anyMatch(badge -> badge.getId().equals("welcome_promocode")))) {
            result.setBannerType(HotelOffersInfo.BannerType.WELCOME_PROMOCODE);
        } else {
            result.setBannerType(HotelOffersInfo.BannerType.NONE);
        }

        if (!extractedOffers.isEmpty()) {
            result.setDefaultOffer(extractedOffers.get(0));
        }
        if (protoHotel != null) {
            result.setOfferCount(protoHotel.getFullPriceCount());
            result.setOperatorCount(protoHotel.getFullOperatorCount());
        } else {
            result.setOfferCount(0);
            result.setOperatorCount(0);
        }

        if (result.getRooms() != null) {
            for (var room : result.getRooms()) {
                var aggregates = getOfferAggregates(result.getMainOffers().stream().filter(x -> x.getRoomId().equals(room.getId())).collect(Collectors.toUnmodifiableList()));
                room.setPansionAggregate(aggregates.getPansionAggregate());
                room.setCancellationInfoAggregate(aggregates.getCancellationInfoAggregate());
            }
        }
        result.setOperatorById(ocMeta.getOperatorById());
        return result;
    }

    private static String formatRoomAmenityName(TReadResp.TPermaroom.TBinaryFeature feature) {
        if (feature.hasDisplayValue() && !Strings.isNullOrEmpty(feature.getDisplayValue())) {
            return feature.getDisplayValue();
        }
        if (feature.getValue()) {
            return feature.getName();
        }
        return feature.getName() + ": нет";
    }

    private static String formatRoomAmenityName(TReadResp.TPermaroom.TEnumFeature feature) {
        if (feature.hasDisplayValue() && !Strings.isNullOrEmpty(feature.getDisplayValue())) {
            return feature.getDisplayValue();
        }
        if (feature.getValueId().equals("yes")) {
            return feature.getName();
        }
        return feature.getName() + ": " + feature.getValueName();
    }

    private static String formatRoomAmenityName(TReadResp.TPermaroom.TIntegerFeature feature) {
        if (feature.hasDisplayValue() && !Strings.isNullOrEmpty(feature.getDisplayValue())) {
            return feature.getDisplayValue();
        }
        return feature.getName() + ": " + feature.getValue();
    }

    private static String formatRoomAmenityName(TReadResp.TPermaroom.TFloatFeature feature) {
        if (feature.hasDisplayValue() && !Strings.isNullOrEmpty(feature.getDisplayValue())) {
            return feature.getDisplayValue();
        }
        return feature.getName() + ": " + String.format("%.2f", feature.getValue());
    }

    private static String formatRoomAmenityName(TReadResp.TPermaroom.TStringFeature feature) {
        if (feature.hasDisplayValue() && !Strings.isNullOrEmpty(feature.getDisplayValue())) {
            return feature.getDisplayValue();
        }
        return feature.getName() + ": " + feature.getValue();
    }

    private static List<AmenityInfo> extractRoomAmenities(TReadResp.TPermaroom permaroom) {
        return Stream.of(
                permaroom.getBinaryFeaturesList().stream()
                        .map(x -> new AmenityInfo(x.hasTopFeatureImportance(), x.getTopFeatureImportance(), x.getCategory(), x.getCategoryName(),
                                new RoomAmenity(x.getId() + ":" + x.getValue(), Strings.isNullOrEmpty(x.getIconId()) ? x.getId() : x.getIconId(), formatRoomAmenityName(x)))),
                permaroom.getEnumFeaturesList().stream()
                        .map(x -> new AmenityInfo(x.hasTopFeatureImportance(), x.getTopFeatureImportance(), x.getCategory(), x.getCategoryName(),
                                new RoomAmenity(x.getId() + ":" + x.getValueId(), Strings.isNullOrEmpty(x.getIconId()) ? x.getId() : x.getIconId(), formatRoomAmenityName(x)))),
                permaroom.getIntegerFeaturesList().stream()
                        .map(x -> new AmenityInfo(x.hasTopFeatureImportance(), x.getTopFeatureImportance(), x.getCategory(), x.getCategoryName(),
                                new RoomAmenity(x.getId() + ":" + x.getValue(), Strings.isNullOrEmpty(x.getIconId()) ? x.getId() : x.getIconId(), formatRoomAmenityName(x)))),
                permaroom.getFloatFeaturesList().stream()
                        .map(x -> new AmenityInfo(x.hasTopFeatureImportance(), x.getTopFeatureImportance(), x.getCategory(), x.getCategoryName(),
                                new RoomAmenity(x.getId() + ":" + x.getValue(), Strings.isNullOrEmpty(x.getIconId()) ? x.getId() : x.getIconId(), formatRoomAmenityName(x)))),
                permaroom.getStringFeaturesList().stream()
                        .map(x -> new AmenityInfo(x.hasTopFeatureImportance(), x.getTopFeatureImportance(), x.getCategory(), x.getCategoryName(),
                                new RoomAmenity(x.getId() + ":" + x.getValue(), Strings.isNullOrEmpty(x.getIconId()) ? x.getId() : x.getIconId(), formatRoomAmenityName(x))))
        )
                .flatMap(x -> x)
                .collect(Collectors.toUnmodifiableList());
    }

    private static final Comparator<AmenityInfo> AMENITY_INFO_COMPARATOR = (AmenityInfo lhs, AmenityInfo rhs) -> {
        if (Math.abs(lhs.topFeatureImportance - rhs.topFeatureImportance) < 1e-6) {
            return lhs.roomAmenity.getId().compareTo(rhs.roomAmenity.getId());
        }
        return Double.compare(lhs.topFeatureImportance, rhs.topFeatureImportance) * -1;
    };

    static List<List<AmenityInfo>> extractGrouppedAmenityInfo(List<AmenityInfo> amenityInfos) {
        return amenityInfos
                .stream()
                .collect(Collectors.groupingBy(AmenityInfo::getCategory, Collectors.toUnmodifiableList()))
                .values()
                .stream()
                .map(x -> x.stream().sorted(AMENITY_INFO_COMPARATOR).collect(Collectors.toUnmodifiableList()))
                .collect(Collectors.toUnmodifiableList());
    }

    static List<RoomAmenity> exractMainAmenities(List<List<AmenityInfo>> grouppedAmenityInfos, int max) {
         return grouppedAmenityInfos.stream()
                 .map(x -> x.get(0))
                 .filter(x -> x.hasTopFeatureImportance)
                 .sorted(AMENITY_INFO_COMPARATOR)
                 .limit(max)
                 .map(x -> x.roomAmenity)
                 .collect(Collectors.toUnmodifiableList());
    }

    static List<RoomAmenityGroup> extractRoomAmenitiesGroupped(List<List<AmenityInfo>> grouppedAmenityInfos) {
        return grouppedAmenityInfos
                .stream()
                .map(x -> new RoomAmenityGroup(
                        x.get(0).category,
                        x.get(0).category, // using category id as icon id
                        x.get(0).categoryName,
                        x.stream().sorted(AMENITY_INFO_COMPARATOR).map(y -> y.roomAmenity).collect(Collectors.toUnmodifiableList())
                ))
                .collect(Collectors.toUnmodifiableList());
    }

    private static List<RoomBedGroup> extractRoomBedGroups(TReadResp.TPermaroom permaroom) {
        return permaroom.getBedGroupsList()
                .stream()
                .map(x -> new RoomBedGroup(
                        x.getId(),
                        x.getConfigurationList()
                                .stream()
                                .map(conf -> new RoomBedConfigurationItem(conf.getType(), conf.getType(), conf.getNameInitialForm(), conf.getNameInflectedForm(), conf.getQuantity()))
                                .collect(Collectors.toUnmodifiableList())
                ))
                .collect(Collectors.toUnmodifiableList());
    }

    public static HotelOffer.DiscountInfo extractOfferDiscountInfo(Price.Currency currency, int price,
                                                                   TStrikethroughPrice protoStPrice,
                                                                   Experiments experiments) {
        if (HotelsPortalUtils.hasForceOfferDiscount(experiments)) {
            var forceDiscountValuePct = experiments.getIntValue(HotelsPortalUtils.FORCE_OFFER_DISCOUNT_EXP);
            var fakeStPrice = price * 100 / (100 - forceDiscountValuePct);
            double percent = (100d * (fakeStPrice - price)) / (double) fakeStPrice;
            HotelOffer.DiscountInfo discountInfo = new HotelOffer.DiscountInfo();
            discountInfo.setStrikethroughPrice(new Price(fakeStPrice, currency));
            discountInfo.setPercent(Math.round(percent * 10d) / 10d);
            discountInfo.setReason(HotelOffer.DiscountInfo.EDiscountReason.RESTRICTED_OFFER);
            discountInfo.setShowDiscountInfo(discountInfo.getPercent() > MIN_VISIBLE_DISCOUNT_PERCENTAGE);
            return discountInfo;
        }
        double percent =
                (100d * (protoStPrice.getPrice() - price)) / (double) protoStPrice.getPrice();
        HotelOffer.DiscountInfo discountInfo = new HotelOffer.DiscountInfo();
        Price stPrice = new Price();
        stPrice.setCurrency(currency);
        stPrice.setValue(protoStPrice.getPrice());
        discountInfo.setStrikethroughPrice(stPrice);
        discountInfo.setPercent(Math.round(percent * 10d) / 10d);
        switch (protoStPrice.getReason()) {
            case SPR_RestrictedOffer:
                discountInfo.setReason(HotelOffer.DiscountInfo.EDiscountReason.RESTRICTED_OFFER);
            default:
                break;
        }
        discountInfo.setShowDiscountInfo(discountInfo.getPercent() > MIN_VISIBLE_DISCOUNT_PERCENTAGE);

        return discountInfo;
    }

    private static HotelOffer extractOffer(OfferCacheMetadata ocMeta, TPrice protoPrice, Map<String, Integer> offerBadgePriorities, Experiments experiments) {
        HotelOffer offer = new HotelOffer();
        offer.setId(protoPrice.getOfferId());
        Price price = new Price();
        price.setValue(protoPrice.getPrice());
        price.setCurrency(ocMeta.getCurrency());
        offer.setPrice(price);
        // Rest may be missing (for brief hotels for example)
        HotelOffer.CancellationInfo ci = new HotelOffer.CancellationInfo();
        if (protoPrice.hasFreeCancellation()) {
            ci.setHasFreeCancellation(protoPrice.getFreeCancellation());
        }
        if (protoPrice.hasRefundType()) {
            ci.setRefundType(mapRefundProtoToPojo(protoPrice.getRefundType()));
        }
        ci.setRefundRules(mapProtoRefundRulesToPojo(protoPrice.getRefundRuleList(), offer.getPrice().getCurrency()));
        offer.setCancellationInfo(ci);
        if (protoPrice.hasPansion()) {
            HotelOffer.MealType mt = new HotelOffer.MealType();
            mt.setId(protoPrice.getPansion().name());
            mt.setName(ocMeta.getPansionTypes().get(mt.getId()));
            offer.setMealType(mt);
        }
        if (protoPrice.hasOperatorId()) {
            offer.setOperatorId(String.valueOf(protoPrice.getOperatorId()));
            OperatorInfo opInfo = ocMeta.getOperatorById().get(offer.getOperatorId());
            offer.setYandexOffer(opInfo != null && opInfo.isBookOnYandex());
        }
        if (protoPrice.hasPartnerLink()) {
            offer.setLandingUrl(protoPrice.getPartnerLink());
            if (offer.isYandexOffer()) {
                MultiValueMap<String, String> parameters =
                        UriComponentsBuilder.fromUriString(offer.getLandingUrl()).build().getQueryParams();
                offer.setToken(parameters.getFirst("Token"));
            }
        }
        if (protoPrice.hasPermaroomId()) {
            offer.setRoomId(protoPrice.getPermaroomId());
        }
        if (protoPrice.hasRoomType()) {
            if (ocMeta.isUseProdOfferCache()) {
                offer.setName("[PROD!] " + protoPrice.getRoomType());
            } else {
                offer.setName(protoPrice.getRoomType());
            }
        }
        if (protoPrice.getBadgesCount() > 0) {
            offer.setBadges(getBadgesWithPriority(protoPrice.getBadgesList(), offerBadgePriorities, MAX_BADGES_PER_OFFER));
        }

        if (protoPrice.hasStrikethroughPrice() || HotelsPortalUtils.hasForceOfferDiscount(experiments)) {
            offer.setDiscountInfo(extractOfferDiscountInfo(ocMeta.getCurrency(), protoPrice.getPrice(), protoPrice.getStrikethroughPrice(), experiments));
        }

        if (protoPrice.hasYandexPlusInfo()) {
            offer.setOfferYandexPlusInfo(new HotelOffer.OfferYandexPlusInfo(protoPrice.getYandexPlusInfo().getPoints(), protoPrice.getYandexPlusInfo().getEligible()));
        }

        return offer;
    }

    public static List<RefundRule> mapProtoRefundRulesToPojo(List<TRefundRule> refundRuleList, Price.Currency currency) {
        if (refundRuleList == null || refundRuleList.isEmpty()) {
            return null;
        }
        return refundRuleList.stream().map(protoRefundRule -> {
            RefundRule.RefundRuleBuilder refundRuleBuilder = RefundRule.builder()
                    .type(mapRefundProtoToPojo(protoRefundRule.getType()));
            if (protoRefundRule.hasStartsAtTimestampSec()) {
                refundRuleBuilder.startsAt(Instant.ofEpochSecond(protoRefundRule.getStartsAtTimestampSec()));
            }
            if (protoRefundRule.hasEndsAtTimestampSec()) {
                refundRuleBuilder.endsAt(Instant.ofEpochSecond(protoRefundRule.getEndsAtTimestampSec()));
            }
            if (protoRefundRule.hasPenalty()) {
                refundRuleBuilder.penalty(Money.of(protoRefundRule.getPenalty(), ProtoCurrencyUnit.fromCurrencyCode(currency.name())));
            }
            return refundRuleBuilder.build();
        }).collect(Collectors.toList());
    }

    public static RefundType mapRefundProtoToPojo(ERefundType refundType) {
        switch (refundType){
            case RT_FULLY_REFUNDABLE:
                return RefundType.FULLY_REFUNDABLE;
            case RT_REFUNDABLE_WITH_PENALTY:
                return RefundType.REFUNDABLE_WITH_PENALTY;
            case RT_NON_REFUNDABLE:
                return RefundType.NON_REFUNDABLE;
            default:
                return null;
        }
    }

    private static HotelImagesInfo extractHotelImagesFromOverride(ImageParamsProvider params, Collection<THotelImage> imagesOverride) {
        Map<Integer, List<THotelImage>> byOrder = imagesOverride.stream().collect(Collectors.groupingBy(THotelImage::getOrder));
        int limit = params.getImageLimit() == null ? 1000 : params.getImageLimit();
        limit = Math.max(0, limit);
        if (params.isOnlyTopImages()) {
            limit = Math.min(limit, 10);
        }
        List<Integer> orders = byOrder.keySet().stream().sorted().skip(params.getImageOffset()).limit(limit).collect(Collectors.toList());
        List<HotelImage> images = new ArrayList<>();
        for (Integer order: orders) {
            List<THotelImage> pbImages = byOrder.get(order);
            String urlTemplate = pbImages.get(0).getUrl();
            if (urlTemplate.endsWith(pbImages.get(0).getSize())) {
                urlTemplate = urlTemplate.substring(0, urlTemplate.length() - pbImages.get(0).getSize().length());
            }
            if (!urlTemplate.endsWith("/")) {
                urlTemplate += "/";
            }
            urlTemplate += "%s";
            HotelImage img = new HotelImage();
            img.setUrlTemplate(urlTemplate);
            img.setId(String.format("%s/%s", pbImages.get(0).getPermalink(), order));
            img.setTags(Collections.emptyList());
            List<HotelImage.Size> sizes = new ArrayList<>();
            img.setSizes(sizes);
            for (THotelImage pbImage: pbImages) {
                if (params.getImageSizes() != null && !params.getImageSizes().contains(pbImage.getSize())) {
                    continue;
                }
                HotelImage.Size size = new HotelImage.Size();
                size.setSize(pbImage.getSize());
                size.setWidth(pbImage.getWidth());
                size.setHeight(pbImage.getHeight());
                sizes.add(size);
            }
            images.add(img);
        }
        return new HotelImagesInfo(images, byOrder.size(), false);
    }

    private static HotelImagesInfo extractHotelImagesUsingWhitelist(
            ImageParamsProvider params, List<GeoHotelPhoto> spravPhotos, List<String> whitelistUrls
    ) {
        Map<String, GeoHotelPhoto> photoMap = new HashMap<>();
        for (GeoHotelPhoto photo : spravPhotos) {
            photoMap.put(photo.getPhoto().getUrlTemplate(), photo);
        }

        boolean imagesMayBeChangedByWhitelist = false;
        List<HotelImage> images = new ArrayList<>();
        for (String photoUrl : whitelistUrls) {
            GeoHotelPhoto hotelPhoto = photoMap.get(photoUrl);
            if (hotelPhoto != null) {
                int currentIndex = images.size();
                if (currentIndex >= spravPhotos.size() || hotelPhoto != spravPhotos.get(currentIndex)) {
                    imagesMayBeChangedByWhitelist = true;
                }
                images.add(extractHotelImage(hotelPhoto, params.getImageSizes()));
            }
        }
        int totalImageCount = images.size();
        var imagesStream = images.stream().skip(params.getImageOffset());
        if (params.getImageLimit() != null) {
            imagesStream = imagesStream.limit(params.getImageLimit());
        }
        if (params.isOnlyTopImages()) {
            imagesStream = imagesStream.limit(10);
        }

        return new HotelImagesInfo(imagesStream.collect(Collectors.toList()),  totalImageCount, imagesMayBeChangedByWhitelist);
    }

    public static HotelImagesInfo extractHotelImages(
            GeoHotel geoHotel, ImageParamsProvider params, HotelImagesService hotelImagesService,
            UaasSearchExperiments uaasSearchExperiments, ImageWhitelistDataProvider imageWhitelistDataProvider
    ) {
        Collection<THotelImage> imagesOverride = hotelImagesService.getImagesByPermalink(geoHotel.getPermalink());
        if (imagesOverride != null) {
            return extractHotelImagesFromOverride(params, imagesOverride);
        }

        boolean imagesMayBeChangedByWhitelist = false;
        List<GeoHotelPhoto> spravPhotos = geoHotel.getSpravPhotos();
        List<String> whitelistUrls = imageWhitelistDataProvider.getPermalinkImages(geoHotel.getPermalink());
        if (geoHotel.getSpravPhotos() != null && whitelistUrls != null) {
            HotelImagesInfo whitelistedImagesInfo = extractHotelImagesUsingWhitelist(
                    params, spravPhotos, whitelistUrls
            );
            imagesMayBeChangedByWhitelist = whitelistedImagesInfo.imagesMayBeChangedByWhitelist;
            if (uaasSearchExperiments.isHotelImagesNoWatermark()) {
                return whitelistedImagesInfo;
            }
        }

        Set<String> topPhotos = null;
        if (params.isOnlyTopImages() && geoHotel.getPhotos() != null) {
            topPhotos = new HashSet<>();
            for (Photos2X.Photo geoPhoto : geoHotel.getPhotos().getPhotoList()) {
                topPhotos.add(geoPhoto.getUrlTemplate());
            }
        }
        List<HotelImage> images = new LinkedList<>();
        if (geoHotel.getSpravPhotos() != null) {
            int offset = params.getImageOffset();
            for (GeoHotelPhoto geoPhoto : geoHotel.getSpravPhotos()) {
                if (offset > 0) {
                    --offset;
                    continue;
                }
                if (topPhotos != null && !topPhotos.contains(geoPhoto.getPhoto().getUrlTemplate())) {
                    continue;
                }
                if (params.getImageLimit() != null && images.size() >= params.getImageLimit()) {
                    break;
                }
                images.add(extractHotelImage(geoPhoto, params.getImageSizes()));
            }
        }
        int totalImageCount = 0;
        if (geoHotel.getSpravPhotos() != null) {
            totalImageCount = geoHotel.getSpravPhotos().size();
        }
        return new HotelImagesInfo(images, totalImageCount, imagesMayBeChangedByWhitelist);
    }

    public static Coordinates extractCoordinates(GeometryOuterClass.Point point) {
        Coordinates coordinates = new Coordinates();
        coordinates.setLat(point.getLat());
        coordinates.setLon(point.getLon());
        return coordinates;
    }

    public static Coordinates extractHotelCoordinates(GeoHotel geoHotel) {
        if (geoHotel.getGeoObject() == null) {
            return null;
        }
        GeometryOuterClass.Point point = null;
        for (GeometryOuterClass.Geometry geometry : geoHotel.getGeoObject().getGeometryList()) {
            if (geometry.hasPoint()) {
                point = geometry.getPoint();
                break;
            }
        }
        // hotelDescription - not filled
        if (point != null) {
            return extractCoordinates(point);
        }
        return null;
    }

    public static List<Hotel.Station> extractHotelNearestStations(GeoHotel geoHotel) {
        if (geoHotel.getNearByStops() == null) {
            return Collections.emptyList();
        }
        List<Hotel.Station> stations = new ArrayList<>();
        for (Masstransit2X.NearbyStop geoStop : geoHotel.getNearByStops().getStopList()) {
            Hotel.Station station = new Hotel.Station();
            station.setId(geoStop.getStop().getId());
            station.setName(geoStop.getStop().getName());
            station.setCoordinates(extractCoordinates(geoStop.getPoint()));
            station.setDistanceMeters(geoStop.getDistance().getValue());
            station.setDistanceText(geoStop.getDistance().getText());
            var stationType = Hotel.Station.Type.OTHER;
            if (geoStop.getLineAtStopCount() > 0) {
                Masstransit2X.Line geoLine = geoStop.getLineAtStop(0).getLine();// Hm
                if (geoLine.getVehicleTypesCount() > 0) {
                    if ("underground".equals(geoLine.getVehicleTypes(0))) { // Hm
                        stationType = Hotel.Station.Type.METRO;
                        Hotel.MetroLine metroLine = new Hotel.MetroLine();
                        station.setMetroLine(metroLine);
                        metroLine.setId(geoLine.getId());
                        metroLine.setName(geoLine.getName());
                        if (geoLine.getStyle().hasColor()) {
                            metroLine.setColor(String.format("#%06x", geoLine.getStyle().getColor()));
                        }
                    }
                }
            }
            if (stationType == Hotel.Station.Type.OTHER && station.getName() != null && !station.getName().isEmpty()) {
                station.setName(String.format("ост. «%s»", station.getName()));
            }
            station.setType(stationType);
            stations.add(station);
        }
        return stations;
    }

    public static HotelInfo extractHotel(AmenityService amenityService, HotelSlugService hotelSlugService,
                                         HotelImagesService hotelImagesService,
                                         HotelIdentifierProvider hotelId, GeoHotel geoHotel,
                                         ImageParamsProvider imageParams, boolean onlyMainAmenities,
                                         List<String> categoryIdsFromFilters, Experiments experiments,
                                         List<String> hotelCategoryIds, boolean extractDistance,
                                         Boolean isFavorite, UaasSearchExperiments uaasSearchExperiments,
                                         ImageWhitelistDataProvider imageWhitelistDataProvider,
                                         boolean needNearestStations
    ) {
        Hotel rspHotel = new Hotel();
        rspHotel.setPermalink(Permalink.of(geoHotel.getGeoObjectMetadata().getId()));
        rspHotel.setHotelSlug(hotelSlugService.findMainSlug(hotelId, rspHotel.getPermalink()));
        if (extractDistance) {
            if (geoHotel.getGeoObjectMetadata().hasDistance()) {
                rspHotel.setDistanceMeters(geoHotel.getGeoObjectMetadata().getDistance().getValue());
                rspHotel.setDistanceText(geoHotel.getGeoObjectMetadata().getDistance().getText());
            }
        }
        rspHotel.setName(geoHotel.getGeoObjectMetadata().getName());
        Hotel.Category category = new Hotel.Category();
        var categories = geoHotel.getGeoObjectMetadata().getCategoryList();
        if (categoryIdsFromFilters != null) {
            var filteredCategories = filterHotelCategoriesByIds(categories, categoryIdsFromFilters);
            if (!filteredCategories.isEmpty()) {
                categories = filteredCategories;
            }
        }

        List<Business.Category> hotelCategories = filterHotelCategoriesByIds(categories, hotelCategoryIds);

        if (!hotelCategories.isEmpty()) {
            categories = hotelCategories;
        }

        // Intentionally use 1st category
        category.setId(categories.get(0).getClass_());
        category.setName(categories.get(0).getName());
        rspHotel.setCategory(category);
        rspHotel.setAddress(geoHotel.getGeoObjectMetadata().getAddress().getFormattedAddress());
        int stars = geoHotel.getNumStars();
        rspHotel.setStars(stars == 0 ? null : stars);
        rspHotel.setRating(geoHotel.getRatingAt0To5Scale());
        rspHotel.setHasVerifiedOwner(false);
        for (Business.Properties.Item item : geoHotel.getGeoObjectMetadata().getProperties().getItemList()) {
            if ("has_verified_owner".equals(item.getKey()) && "1".equals(item.getValue())) {
                rspHotel.setHasVerifiedOwner(true);
                break;
            }
        }
        if (geoHotel.getRating() != null) {
            rspHotel.setTotalTextReviewCount(geoHotel.getRating().getReviews());
        } else {
            rspHotel.setTotalTextReviewCount(0);
        }
        var imgInfo = extractHotelImages(
                geoHotel, imageParams, hotelImagesService, uaasSearchExperiments, imageWhitelistDataProvider
        );
        rspHotel.setImages(imgInfo.images);
        rspHotel.setTotalImageCount(imgInfo.totalImageCount);
        amenityService.fillForUsualHotel(rspHotel, geoHotel, onlyMainAmenities);
        rspHotel.setCoordinates(extractHotelCoordinates(geoHotel));
        if (needNearestStations) {
            rspHotel.setNearestStations(extractHotelNearestStations(geoHotel));
        }
        rspHotel.setIsFavorite(isFavorite);

        return new HotelInfo(rspHotel, imgInfo.imagesMayBeChangedByWhitelist);
    }

    public static HotelWithOffers extractHotelWithOffers(AmenityService amenityService,
                                                         HotelSlugService hotelSlugService,
                                                         HotelImagesService hotelImagesService,
                                                         OfferCacheMetadata ocMeta, GeoHotel geoHotel,
                                                         ImageParamsProvider imageParams,
                                                         boolean onlyMainAmenities,
                                                         Map<String, Integer> hotelBadgePriorities,
                                                         Map<String, Integer> offerBadgePriorities,
                                                         List<String> categoryIdsFromFilters,
                                                         Experiments experiments,
                                                         List<String> hotelCategoryIds,
                                                         boolean extractDistance,
                                                         boolean highlightedAsSearchedByUser,
                                                         boolean isFavorite,
                                                         UaasSearchExperiments uaasSearchExperiments,
                                                         ImageWhitelistDataProvider imageWhitelistDataProvider,
                                                         boolean needNearestStations) {
        HotelInfo hotelInfo = extractHotel(amenityService, hotelSlugService, hotelImagesService,
                null, geoHotel, imageParams, onlyMainAmenities, categoryIdsFromFilters, experiments,
                hotelCategoryIds, extractDistance, isFavorite, uaasSearchExperiments, imageWhitelistDataProvider,
                needNearestStations
        );
        HotelWithOffers hotelWithOffers = new HotelWithOffers();
        hotelWithOffers.setHotel(hotelInfo.hotel);
        fillShortOfferInfo(hotelWithOffers, ocMeta, geoHotel, hotelBadgePriorities, experiments, offerBadgePriorities);
        hotelWithOffers.setSearchedByUser(highlightedAsSearchedByUser);
        return hotelWithOffers;
    }

    public static ShortHotelOffersInfo extractOnlyOffers(OfferCacheMetadata ocMeta, GeoHotel geoHotel,
                                                         Map<String, Integer> hotelBadgePriorities,
                                                         Map<String, Integer> offerBadgePriorities, Experiments experiments) {
        ShortHotelOffersInfo offers = new ShortHotelOffersInfo();
        fillShortOfferInfo(offers, ocMeta, geoHotel, hotelBadgePriorities, experiments, offerBadgePriorities);
        return offers;
    }

    private static void fillShortOfferInfo(ShortHotelOffersInfoSetter shortHotelOffersInfoSetter,
                                           OfferCacheMetadata ocMeta,
                                           GeoHotel geoHotel,
                                           Map<String, Integer> hotelBadgePriorities,
                                           Experiments experiments,
                                           Map<String, Integer> offerBadgePriorities) {
        if (geoHotel.getOfferCacheResponse() != null) {
            shortHotelOffersInfoSetter.setSearchIsFinished(geoHotel.getOfferCacheResponse().getIsFinished());
            var hotelBadges = extractHotelBadges(geoHotel.getOfferCacheResponse(), hotelBadgePriorities);
            var offers = extractHotelOffers(ocMeta, geoHotel.getOfferCacheResponse(), offerBadgePriorities, experiments);
            if (!offers.isEmpty()) {
                // Temporary hack, because from uses badges from offers[0], not hotel badges
                offers.get(0).setBadges(hotelBadges);
            }
            shortHotelOffersInfoSetter.setOffers(offers);
            shortHotelOffersInfoSetter.setBadges(hotelBadges);
        } else {
            shortHotelOffersInfoSetter.setSearchIsFinished(true);
            shortHotelOffersInfoSetter.setOffers(Collections.emptyList());
        }
    }

    private static List<Badge> extractHotelBadges(TReadResp.THotel protoHotel, Map<String, Integer> hotelBadgePriorities) {
        return getBadgesWithPriority(protoHotel.getBadgesList(), hotelBadgePriorities, MAX_BADGES_PER_HOTEL);
    }

    private static List<Badge> getBadgesWithPriority(List<TBadge> badges, Map<String, Integer> hotelBadgePriorities, int limit) {
        return badges.stream()
                .filter(protoBadge -> hotelBadgePriorities.getOrDefault(protoBadge.getId(), 0) != 0)
                .map(BadgeUtils::extractBadge)
                .sorted((lhs, rhs) -> Integer.compare(hotelBadgePriorities.get(lhs.getId()),
                        hotelBadgePriorities.get(rhs.getId())) * -1)
                .limit(limit)
                .collect(Collectors.toUnmodifiableList());
    }

    public static String getSerpUrl(String query, OfferCacheRequestParams ocParams, String oid,
                                    String utmSource, String utmMedium) {
        RequestBuilder builder = new RequestBuilder()
                .setUrl("https://yandex.ru/search")
                .addQueryParam("text", query);
        if (StringUtils.isNotEmpty(oid)) {
            builder.addQueryParam("oid", oid);
        }
        if (ocParams != null) {
            ocParams.putToQueryParamsAsDynamic(builder);
        }
        if (StringUtils.isNotEmpty(utmSource)) {
            builder.addQueryParam("utm_source", utmSource);
        }
        if (StringUtils.isNotEmpty(utmMedium)) {
            builder.addQueryParam("utm_medium", utmMedium);
        }
        return builder.build().getUrl();
    }

    public static List<String> formatDateRange(LocalDate start, LocalDate end, String lang,
                                               LocalizationService localization) {
        Locale locale = new Locale(lang);
        String year = localization.getLocalizedValue("Year", lang);
        year = localization.getWhom(year);
        StringBuilder startPatternBulder = new StringBuilder("d");
        if (end.getMonth() != start.getMonth() || end.getYear() != start.getYear()) {
            startPatternBulder.append(" MMMM");
        }
        if (end.getYear() != start.getYear()) {
            startPatternBulder.append(String.format(" YYYY %s", year));
        }
        String startPattern = startPatternBulder.toString();
        String endPattern = String.format("d MMMM YYYY %s", year);
        return ImmutableList.of(start.format(DateTimeFormatter.ofPattern(startPattern, locale)),
                end.format(DateTimeFormatter.ofPattern(endPattern, locale)));
    }

    public static String encodeProtoToString(com.google.protobuf.AbstractMessage proto) {
        byte[] dataBytes = proto.toByteArray();
        CRC32 crc32 = new CRC32();
        crc32.update(dataBytes);
        long crc = crc32.getValue();
        byte[] dataWithCrc = new byte[dataBytes.length + CRC_BYTES];
        for (int i = 0; i < CRC_BYTES; ++i) {
            dataWithCrc[i] = (byte) (crc & 0xFF);
            crc >>= 8;
        }
        System.arraycopy(dataBytes, 0, dataWithCrc, CRC_BYTES, dataBytes.length);
        return BASE64_ENCODER.encodeToString(dataWithCrc);
    }

    public static ByteBuffer decodeBytesFromString(String data) {
        byte[] dataWithCrc = Base64.getUrlDecoder().decode(data);
        if (dataWithCrc.length < CRC_BYTES) {
            throw new RuntimeException("Too short proto");
        }
        long crc = 0;
        for (int i = 0; i < CRC_BYTES; ++i) {
            crc <<= 8;
            crc |= dataWithCrc[CRC_BYTES - i - 1] & 0xFF;
        }
        CRC32 crc32 = new CRC32();
        crc32.update(dataWithCrc, CRC_BYTES, dataWithCrc.length - CRC_BYTES);
        if (crc != crc32.getValue()) {
            throw new RuntimeException("Invalid checksum");
        }
        return ByteBuffer.wrap(dataWithCrc, CRC_BYTES, dataWithCrc.length - CRC_BYTES);
    }

    public static BoundingBox getBboxByRegion(RegionHash region) {
        double latCenter = region.getAttr("latitude").getDouble();
        double lonCenter = region.getAttr("longitude").getDouble();
        double latSize = region.getAttr("latitude_size").getDouble();
        double lonSize = region.getAttr("latitude_size").getDouble();
        Coordinates leftDown = new Coordinates();
        leftDown.setLat(latCenter - latSize / 2d);
        leftDown.setLon(lonCenter - lonSize / 2d);
        Coordinates upRight = new Coordinates();
        upRight.setLat(latCenter + latSize / 2d);
        upRight.setLon(lonCenter + lonSize / 2d);
        return BoundingBox.of(leftDown, upRight);
    }

    public static BoundingBox getBboxByGeoId(GeoBase geoBase, int geoId, String domain) {
        return getBboxByRegion(geoBase.getRegionById(geoId, domain));
    }

    public static BoundingBox getBBoxByHotels(Integer geoId, String domain, GeoBase geoBase,
                                              Iterable<Hotel> hotels) {
        BoundingBox bbox = null;
        for (var hotel : hotels) {
            var coordinates = hotel.getCoordinates();
            if (hotel.getCoordinates() != null) {
                bbox = HotelsPortalUtils.extendOrCreateBbox(bbox, coordinates);
            }
        }
        if (bbox == null) {
            bbox = HotelsPortalUtils.getBboxByGeoId(geoBase, geoId, domain);
        }
        return bbox.extendByRelativeValue(0.05, 0.06);
    }

    public static Integer determineUserRegion(GeoBase geoBase, CommonHttpHeaders headers,
                                              RequestAttributionProvider attribution) {
        if (attribution != null && attribution.getUserRegion() != null) {
            return attribution.getUserRegion();
        }
        if (headers.getUserGid() != null) {
            try {
                return Integer.valueOf(headers.getUserGid());
            } catch (NumberFormatException e) {
                log.error("Invalid value of UserGid: {}, error: {}", headers.getUserGid(), e);
            }
        }
        if (headers.getUserIP() != null) {
            try {
                int geoId = geoBase.getRegionIdByIp(headers.getUserIP());
                if (geoId > 0) {
                    return geoId;
                }
                log.warn("Failed to determine region by IP {}", headers.getUserIP());
            } catch (LookupException e) {
                log.error("Failed to lookup region by IP {}, exception {}", headers.getUserIP(), e);
            }
        }
        // give up
        return null;
    }

    public static TReadReq.TAttribution.Builder prepareOfferCacheAttribution(RequestAttributionProvider attribution,
                                                                             CommonHttpHeaders headers,
                                                                             Integer userRegion) {
        var ocAttribution = TReadReq.TAttribution.newBuilder();
        if (attribution.getUtmSource() != null) {
            ocAttribution.setUtmSource(attribution.getUtmSource());
        }
        if (attribution.getUtmMedium() != null) {
            ocAttribution.setUtmMedium(attribution.getUtmMedium());
        }
        if (attribution.getUtmCampaign() != null) {
            ocAttribution.setUtmCampaign(attribution.getUtmCampaign());
        }
        if (attribution.getUtmTerm() != null) {
            ocAttribution.setUtmTerm(attribution.getUtmTerm());
        }
        if (attribution.getUtmContent() != null) {
            ocAttribution.setUtmContent(attribution.getUtmContent());
        }
        if (attribution.getSerpReqId() != null) {
            ocAttribution.setSerpReqIdOverride(attribution.getSerpReqId());
        }
        if (attribution.getGclid() != null) {
            ocAttribution.setGclid(attribution.getGclid());
        }
        if (attribution.getYtpReferer() != null) {
            ocAttribution.setYtpReferer(attribution.getYtpReferer());
        }
        if (attribution.getYclid() != null) {
            ocAttribution.setYclid(attribution.getYclid());
        }
        if (attribution.getFbclid() != null) {
            ocAttribution.setFBclid(attribution.getFbclid());
        }
        if (attribution.getMetrikaClientId() != null) {
            ocAttribution.setMetrikaClientId(attribution.getMetrikaClientId());
        }
        if (attribution.getMetrikaUserId() != null) {
            ocAttribution.setMetrikaUserId(attribution.getMetrikaUserId());
        }
        if (attribution.getClid() != null) {
            ocAttribution.setClid(attribution.getClid());
        }
        if (attribution.getAffiliateClid() != null) {
            ocAttribution.setAffiliateClid(attribution.getAffiliateClid());
        }
        if (attribution.getAdmitadUid() != null) {
            ocAttribution.setAdmitadUid(attribution.getAdmitadUid());
        }
        if (attribution.getTravelpayoutsUid() != null) {
            ocAttribution.setTravelpayoutsUid(attribution.getTravelpayoutsUid());
        }
        if (attribution.getVid() != null) {
            ocAttribution.setVid(attribution.getVid());
        }
        if (attribution.getAffiliateVid() != null) {
            ocAttribution.setAffiliateVid(attribution.getAffiliateVid());
        }
        if (attribution.getReferralPartnerRequestId() != null) {
            ocAttribution.setReferralPartnerRequestId(attribution.getReferralPartnerRequestId());
        }
        if (isWhiteLabelActive(headers)) {
            ocAttribution.setReferralPartnerId(headers.getWhiteLabelPartnerIdStr());
        }
        if (userRegion != null) {
            ocAttribution.setUserRegionOverride(userRegion);
        }
        if (headers.getUserDevice() != null) {
            ocAttribution.setUserDevice(headers.getUserDevice());
        }
        if (headers.getAppPlatform() != null){
            ocAttribution.setAppPlatform(headers.getAppPlatform());
        }
        if (headers.getService() != null){
            ocAttribution.setService(headers.getService());
        }
        if (headers.getRequestId() != null) {
            ocAttribution.setYaTravelReqId(headers.getRequestId());
        }
        String testIds = headers.getTestIdsFromExpBoxes();
        if (testIds != null) {
            ocAttribution.setPortalTestIds(testIds);
        }
        String testBuckets = headers.getTestBucketsFromExpBoxes();
        if (testBuckets != null) {
            ocAttribution.setPortalTestBuckets(testBuckets);
        }
        if (headers.getUserIsStaffAsBool()) {
            ocAttribution.setIsStaffUser(true);
        }
        if (headers.getUserIsPlusAsBool()) {
            ocAttribution.setIsPlusUser(true);
        }
        return ocAttribution;
    }

    public static TReadReq.TAttribution.Builder prepareOfferCacheAttributionForGeoCounter(RequestAttributionProvider attribution,
                                                                                          CommonHttpHeaders headers,
                                                                                          Integer userRegion) {
        TReadReq.TAttribution.Builder attributionBuilder = prepareOfferCacheAttribution(attribution, headers, userRegion);

        String yandexUid = headers.getYandexUid();
        if (yandexUid != null) {
            attributionBuilder.setYandexUid(yandexUid);
        }
        String passportId = headers.getPassportId();
        if (passportId != null) {
            attributionBuilder.setPassportUid(passportId);
        }
        String iCookie = headers.getICookie();
        if (iCookie != null) {
            attributionBuilder.setICookie(iCookie);
        }
        return attributionBuilder;
    }

    public static TReadReq.Builder prepareOfferCacheRequestParams(RequestAttributionProvider attribution,
                                                                  CommonHttpHeaders headers,
                                                                  UserCredentials userCredentials,
                                                                  Integer userRegion,
                                                                  OfferSearchParamsProvider offerSearchParams,
                                                                  DebugOfferSearchParamsProvider debugOfferSearchParams,
                                                                  String labelHash,
                                                                  String searchPagePollingId,
                                                                  boolean withDates,
                                                                  Experiments experiments,
                                                                  UaasSearchExperiments uaasSearchExperiments,
                                                                  String debugId,
                                                                  boolean hasSearchParams,
                                                                  boolean sortOffersUsingPlus) {
        boolean robotRequest = isRobotRequest(headers);
        boolean useSearcher = !robotRequest && hasSearchParams;
        TReadReq.Builder ocReqBuilder;
        if (withDates) {
            ocReqBuilder = OfferCacheRequestParams.prepareBuilder(
                    offerSearchParams.getCheckinDate(), offerSearchParams.getCheckoutDate(),
                    offerSearchParams.getAdults(),
                    offerSearchParams.getChildrenAges(), useSearcher, debugId, hasSearchParams, sortOffersUsingPlus);
        } else {
            ocReqBuilder = OfferCacheRequestParams.prepareBuilder(
                    null, null,
                    offerSearchParams.getAdults(),
                    offerSearchParams.getChildrenAges(), useSearcher, debugId, hasSearchParams, sortOffersUsingPlus);
        }
        ocReqBuilder.setShowAllOperators(true);
        ocReqBuilder.setShowAllBadges(true);
        ocReqBuilder.setRobotRequest(robotRequest);
        ocReqBuilder.setHideMetaStrikethroughForBoyHotels(true);
        ocReqBuilder.setAllowMobileRates(isUserDeviceTouch(headers));
        ocReqBuilder.setWhiteLabelPartnerId(headers.getWhiteLabelPartnerId());
        if (debugOfferSearchParams != null && debugOfferSearchParams.getDebugPortalHost() != null) {
            ocReqBuilder.setDebugPortalHost(debugOfferSearchParams.getDebugPortalHost());
        }
        if (labelHash != null) {
            ocReqBuilder.setLabelHash(labelHash);
        }
        ocReqBuilder.setAllowRestrictedUserRates(userCredentials.isLoggedIn());
        if (experiments.isExp("125")) {
            ocReqBuilder.setOnlyRestrictedOffers(true);
        }
        if (experiments.isExp("725")) {
            ocReqBuilder.addExp(725); // Use non-trivial strings as permaroomIds
        }
        if (uaasSearchExperiments.isStrikeThroughPrices()) {
            ocReqBuilder.addExp(2713);
        }
        if (uaasSearchExperiments.isBookingLandSearchPage()) {
            ocReqBuilder.addStrExp("BookingAid=2192269");
        }
        if (uaasSearchExperiments.isBookingLandHotelPage()) {
            ocReqBuilder.addStrExp("BookingAid=2192270");
        }
        if (uaasSearchExperiments.isOstrovokClickoutTouchDisabled() && isUserDeviceTouch(headers)) {
            ocReqBuilder.addExp(5782);
        }
        if (uaasSearchExperiments.isOstrovokClickoutDisabled()) {
            ocReqBuilder.addBanOpId(EOperatorId.OI_OSTROVOK_VALUE);
        }
        if (isWhiteLabelActive(headers)) {
            ocReqBuilder.setOnlyBoYOffers(true);
        }
        ocReqBuilder.setOnlyBoYWhenBoYAvailable(true);

        ocReqBuilder.addAllKVExperiments(KVExperiments.toProto(headers.getExperiments()));

        var attributionBuilder = prepareOfferCacheAttribution(attribution, headers, userRegion);
        if (searchPagePollingId != null) {
            attributionBuilder.setSearchPagePollingId(searchPagePollingId);
        }
        ocReqBuilder.mergeAttribution(attributionBuilder.build());
        return ocReqBuilder;
    }

    public static GeoOriginEnum selectGeoOriginByDevice(CommonHttpHeaders headers, GeoOriginEnum originDesktop,
                                                        GeoOriginEnum originTouch) {
        if (isUserDeviceTouch(headers)) {
            return originTouch;
        } else {
            return originDesktop;
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    public static int calcSearchParamsHash(SearchHotelsReqV1 req, OfferSearchParamsProvider osp, Integer geoIdOverride) {
        List<Object> parts = new ArrayList<>();
        parts.add(osp.getCheckinDate());
        parts.add(osp.getCheckoutDate());
        if (osp.getAdults() != null) {
            parts.add(Ages.build(osp.getAdults(), osp.getChildrenAges()));
        } else {
            parts.add(null);
        }
        if (req.getFilterAtoms() != null) {
            parts.add(String.join(",", req.getFilterAtoms()));
        } else {
            parts.add(null);
        }
        parts.add(req.getFilterPriceFrom());
        parts.add(req.getFilterPriceTo());
        parts.add(geoIdOverride == null ? req.getGeoId() : geoIdOverride);
        parts.add(req.getPageHotelCount());
        parts.add(req.getPricedHotelLimit());
        parts.add(req.getTotalHotelLimit());
        parts.add(req.isDebugUseProdOffers());

        Hasher hasher = Hashing.murmur3_32().newHasher();
        for (Object part : parts) {
            hasher.putUnencodedChars(part == null ? "" : part.toString());
        }
        return hasher.hash().asInt();
    }

    public static String getHotelSeoTitle(LocalizationService localizationService, GeoBase geoBase,
                                          String hotelName, String categoryId, String categoryName,
                                          double lat, double lon, String locale, String domain, boolean hotelIsBoy) {
        // https://st.yandex-team.ru/TRAVELFRONT-2875#5e9f09d6c6d776477dedc29c
        String hotelNameLoweCase = hotelName.toLowerCase();
        String result = hotelName;
        if (!SEO_TITLE_CATEGORY_REGEXP.matcher(hotelNameLoweCase).find()) {
            String prefix;
            if ("hotels".equals(categoryId) && SEO_TITLE_MASCULINE_REGEXP.matcher(hotelName).find()) {
                prefix = localizationService.getLocalizedValue("SeoTitlePrefixHotel", locale);
            } else {
                prefix = categoryName;
            }
            result = prefix + " " + result;
        }
        int hotelGeoId = geoBase.getRegionIdByLocation(lat, lon);
        Integer cityGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId, GeoBaseHelpers.CITY_REGION_TYPE,
                domain);
        if (cityGeoId != null) {
            Linguistics linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, cityGeoId, locale);
            if (linguistics != null) {
                String[] cases = {linguistics.getNominativeCase(), linguistics.getGenitiveCase(),
                        linguistics.getDativeCase(), linguistics.getPrepositionalCase(), linguistics.getLocativeCase(),
                        linguistics.getDirectionalCase(), linguistics.getAblativeCase(),
                        linguistics.getAccusativeCase(), linguistics.getInstrumentalCase()
                };
                boolean endsWithCity = false;
                for (String case_ : cases) {
                    if (!Strings.isNullOrEmpty(case_) && hotelNameLoweCase.endsWith(case_.toLowerCase())) {
                        endsWithCity = true;
                        break;
                    }
                }
                if (!endsWithCity) {
                    result += String.format(" %s %s", linguistics.getPreposition(), linguistics.getPrepositionalCase());
                }
            }
        } else {
            Integer regionGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId,
                    GeoBaseHelpers.REGION_REGION_TYPE, domain);
            if (regionGeoId != null) {
                Linguistics linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, regionGeoId, locale);
                if (linguistics != null) {
                    result += String.format(", %s", linguistics.getNominativeCase());
                }
            }
        }
        Integer countryGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId,
                GeoBaseHelpers.COUNTRY_REGION_TYPE, domain);
        if (countryGeoId != null && countryGeoId != GeoBaseHelpers.RUSSIA_REGION) {
            Linguistics linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, countryGeoId, locale);
            if (linguistics != null) {
                result += String.format(", %s", linguistics.getNominativeCase());
            }
        }

        String postfixKey;
        if (hotelIsBoy) {
            postfixKey = "SeoTitlePostfixPlus";
        }
        else {
            postfixKey = "SeoTitlePostfix";
        }
        String postfix = localizationService.getLocalizedValueWithoutDefault(postfixKey, locale);
        if (postfix != null) {
            result += " " + postfix;
        }
        return result;
    }

    public static String getHotelSeoTitleExpMinPrice(LocalizationService localizationService, GeoBase geoBase,
                                                     String hotelName, String categoryId, String categoryName,
                                                     double lat, double lon, String locale, String domain,
                                                     OptionalInt hotelMinPrice) {
        // TRAVELORGANIC-240
        // Отель {Название} в {Город} от {мин цена} - Яндекс.Путешествия
        String hotelNameLoweCase = hotelName.toLowerCase();
        String result = hotelName;
        if (!SEO_TITLE_CATEGORY_REGEXP.matcher(hotelNameLoweCase).find()) {
            String prefix;
            if ("hotels".equals(categoryId) && SEO_TITLE_MASCULINE_REGEXP.matcher(hotelName).find()) {
                prefix = localizationService.getLocalizedValue("SeoTitlePrefixHotel", locale);
            } else {
                prefix = categoryName;
            }
            result = prefix + " " + result;
        }
        int hotelGeoId = geoBase.getRegionIdByLocation(lat, lon);
        Integer cityGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId, GeoBaseHelpers.CITY_REGION_TYPE,
                domain);
        if (cityGeoId != null) {
            Linguistics linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, cityGeoId, locale);
            if (linguistics != null) {
                String[] cases = {linguistics.getNominativeCase(), linguistics.getGenitiveCase(),
                        linguistics.getDativeCase(), linguistics.getPrepositionalCase(), linguistics.getLocativeCase(),
                        linguistics.getDirectionalCase(), linguistics.getAblativeCase(),
                        linguistics.getAccusativeCase(), linguistics.getInstrumentalCase()
                };
                boolean endsWithCity = false;
                for (String case_ : cases) {
                    if (!Strings.isNullOrEmpty(case_) && hotelNameLoweCase.endsWith(case_.toLowerCase())) {
                        endsWithCity = true;
                        break;
                    }
                }
                if (!endsWithCity) {
                    result += String.format(" %s %s", linguistics.getPreposition(), linguistics.getPrepositionalCase());
                }
            }
        } else {
            Integer regionGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId,
                    GeoBaseHelpers.REGION_REGION_TYPE, domain);
            if (regionGeoId != null) {
                Linguistics linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, regionGeoId, locale);
                if (linguistics != null) {
                    result += String.format(", %s", linguistics.getNominativeCase());
                }
            }
        }

        Integer countryGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId,
                GeoBaseHelpers.COUNTRY_REGION_TYPE, domain);
        if (countryGeoId != null && countryGeoId != GeoBaseHelpers.RUSSIA_REGION) {
            Linguistics linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, countryGeoId, locale);
            if (linguistics != null) {
                result += String.format(", %s", linguistics.getNominativeCase());
            }
        }

        if (hotelMinPrice.isPresent()) {
            String template = localizationService.getLocalizedValueWithoutDefault("SeoTitleMinPriceTemplate", locale);
            if (template != null) {
                result += String.format(template, hotelMinPrice.getAsInt());
            }
        }

        String postfix = localizationService.getLocalizedValueWithoutDefault("SeoTitlePostfix", locale);
        if (postfix != null) {
            result += " " + postfix;
        }
        return result;
    }

    public static String getHotelSeoDescription(
            LocalizationService localizationService,
            String hotelName,
            String locale,
            boolean hotelIsBoy,
            int plusDefaultPercentDiscount
    ) {
        if (hotelIsBoy) {
            String template = localizationService.getLocalizedValue("SeoHotelDescriptionTemplatePlus", locale);
            return String.format(template, hotelName, plusDefaultPercentDiscount);
        } else {
            String template = localizationService.getLocalizedValue("SeoHotelDescriptionTemplate", locale);
            return String.format(template, hotelName);
        }
    }

    public static <T> CompletableFuture<T> handleExceptions(String name, Supplier<CompletableFuture<T>> method) {
        try {
            return method.get();
        } catch (HotelNotFoundException e) {
            log.error("{}: Hotel not found: {}", name, e.getMessage());
            return CompletableFuture.failedFuture(e);
        } catch (LookupException e) {
            log.error("{}: Region not found: {}", name, e.getMessage());
            return CompletableFuture.failedFuture(new TravelApiBadRequestException("Invalid region id"));
        } catch (IllegalArgumentException e) {
            log.error("{}: IllegalArgumentException: {}", name, e.getMessage());
            return CompletableFuture.failedFuture(e);
        } catch (Exception e) {
            log.error("{}: Failed to complete", name, e);
            return CompletableFuture.failedFuture(e);
        }
    }

    public static void addGeoSearchFilters(GeoSearchReq geoReq, List<HotelFilterGroup> filtersGroups) {
        if (filtersGroups.stream().anyMatch(group -> group.getFilters().stream().map(HotelFilter::getFeatureId).collect(Collectors.toUnmodifiableSet()).size() > 1)) {
            throw new IllegalStateException("Found filter group with several feature ids");
        }
        if (filtersGroups.stream().anyMatch(group -> group.getFilters().stream().map(HotelFilter::getGeoSearchBusinessId).collect(Collectors.toUnmodifiableSet()).size() > 1)) {
            throw new IllegalStateException("Found filter group with several geosearch business ids");
        }
        Preconditions.checkState(filtersGroups.stream().allMatch(group -> group.getUniqueId().equals(group.getFilters().get(0).getFeatureId())),
                "Found filter group with business id != group id");
        geoReq.setFilterBoolOrEnumFeatures(filtersGroups.stream()
                .filter(group -> group.getFilters().stream().anyMatch(x -> x.getListValue() != null))
                .map(group -> {
                    if (group.getFilters().stream().anyMatch(x -> x.getListValue() == null)) {
                        throw new IllegalStateException(String.format("Found filter group with mix of filter types " +
                                "(groupId=%s)", group.getUniqueId()));
                    }
                    return new GeoSearchReq.BoolOrEnumFilter(HotelSearchUtils.getGeoSearchBusinessId(group.getFilters().get(0)),
                            group.getFilters().stream().flatMap(x -> x.getListValue().getValues().stream()).collect(Collectors.toUnmodifiableList()));
                })
                .collect(Collectors.toUnmodifiableList()));
        geoReq.setFilterNumericFeatures(filtersGroups.stream()
                .filter(group -> group.getFilters().stream().anyMatch(x -> x.getNumericValue() != null))
                .map(group -> {
                    if (group.getFilters().stream().anyMatch(x -> x.getNumericValue() == null)) {
                        throw new IllegalStateException(String.format("Found filter group with mix of filter types " +
                                "(groupId=%s)", group.getUniqueId()));
                    }
                    if (group.getFilters().size() > 1) {
                        throw new IllegalStateException(String.format("Found filter group with several numeric " +
                                "filters (groupId=%s)", group.getUniqueId()));
                    }
                    var filter = group.getFilters().get(0);
                    if (filter.getNumericValue().getMode() == HotelFilter.NumericValue.Mode.less) {
                        return new GeoSearchReq.NumericFilter(HotelSearchUtils.getGeoSearchBusinessId(filter), null,
                                filter.getNumericValue().getValue());
                    } else {
                        return new GeoSearchReq.NumericFilter(HotelSearchUtils.getGeoSearchBusinessId(filter),
                                filter.getNumericValue().getValue(), null);
                    }
                })
                .collect(Collectors.toUnmodifiableList()));
    }

    public static boolean isRobotRequest(CommonHttpHeaders headers) {
        return headers.getRealUserAgent() != null && BOTS_USER_AGENT_PATTERN.matcher(headers.getRealUserAgent()).find();
    }

    public static List<String> updateLegacyFilters(List<String> filterAtoms) {
        return filterAtoms
                .stream()
                .map(x -> LEGACY_FILTERS_REMAPPING.getOrDefault(x, x))
                .collect(Collectors.toUnmodifiableList());
    }

    public static int getPollingIterationsDelayMs(HotelsPortalProperties properties, Experiments experiments, int pollingIteration) {
        return getPollingIterationsDelayMs(properties.getPollingDelays(), experiments, pollingIteration);
    }

    public static int getPollingIterationsDelayMs(HotelsPortalProperties.PollingDelays prop, Experiments experiments, int pollingIteration) {
        if (prop.isExpDecay() || experiments.isExp("polling-delay-exp-decay")) {
            var base = prop.getBaseDelayMs();
            var k = Math.pow(prop.getLambda(), Math.min(pollingIteration, prop.getMaxIteration()));
            var delay = Math.min(k * base, prop.getMaxDelayMs());
            return (int)delay;
        } else {
            return getPollingIterationsDelayMsWithoutIteration(prop, experiments);
        }
    }

    public static int getPollingIterationsDelayMsWithoutIteration(HotelsPortalProperties properties, Experiments experiments) {
        return getPollingIterationsDelayMsWithoutIteration(properties.getPollingDelays(), experiments);
    }
    public static int getPollingIterationsDelayMsWithoutIteration(HotelsPortalProperties.PollingDelays prop, Experiments experiments) {
        return Objects.requireNonNullElse(experiments.getIntValue("polling_delay_ms"), prop.getFallbackDelayMs());
    }

    private static List<Business.Category> filterHotelCategoriesByIds(List<Business.Category> categories, List<String> ids) {
        Set<String> tags = ids.stream()
                .map(x -> String.format("id:%s", x))
                .collect(Collectors.toUnmodifiableSet());

        return categories.stream()
                .filter(x -> x.getTagList().stream().anyMatch(tags::contains))
                .collect(Collectors.toUnmodifiableList());
    }

    public static void incrementCounter(String name) {
        Counter.builder(name).register(Metrics.globalRegistry).increment();
    }

    public static void reportDistribution(String name, double value, Tags tags, long... sla) {
        DistributionSummary.builder(name)
                .serviceLevelObjectives(Arrays.stream(sla).asDoubleStream().toArray())
                .tags(tags)
                .register(Metrics.globalRegistry)
                .record(value);
    }

    public static int determineHotelGeoId(GeoBase geoBase, GeoHotel geoHotel, String domain, int defaultGeoId) {
        var hotelCoordinates = HotelsPortalUtils.extractHotelCoordinates(geoHotel);
        if (hotelCoordinates == null) {
            return defaultGeoId;
        }
        var hotelGeoId = GeoBaseHelpers.getRegionIdByLocationOrNull(geoBase, hotelCoordinates.getLat(), hotelCoordinates.getLon());
        if (hotelGeoId == null) {
            return defaultGeoId;
        }
        var preferredGeoId = GeoBaseHelpers.getPreferredGeoId(
                geoBase,
                hotelGeoId,
                Set.of(GeoBaseHelpers.COUNTRY_REGION_TYPE, GeoBaseHelpers.FOREIGN_TERRITORY_REGION_TYPE, GeoBaseHelpers.REGION_REGION_TYPE, GeoBaseHelpers.CITY_REGION_TYPE),
                domain
        );
        return Objects.requireNonNullElse(preferredGeoId, defaultGeoId);
    }

    public static boolean needLikes(CommonHttpHeaders headers, UserCredentials userCredentials) {
        return !HotelsPortalUtils.isRobotRequest(headers) &&
                userCredentials != null &&
                (userCredentials.getYandexUid() != null || userCredentials.isLoggedIn());
    }

    public static BoundingBox extendOrCreateBbox(BoundingBox bbox, Coordinates point) {
        if (bbox == null) {
            return BoundingBox.of(point.clone(), point.clone());
        } else {
            return bbox.extendByPoint(point);
        }
    }

    public static boolean isUserDeviceTouch(CommonHttpHeaders headers) {
        return USER_DEVICE_TOUCH.equalsIgnoreCase(headers.getUserDevice());
    }

    public static boolean isUserDeviceDesktop(CommonHttpHeaders headers) {
        return USER_DEVICE_DESKTOP.equalsIgnoreCase(headers.getUserDevice());
    }

    public static boolean isHotelsNearbyEnabled(CommonHttpHeaders headers) {
        return isUserDeviceTouch(headers);
    }

    public static void setAdditionalGeoSearchParams(GeoSearchReq geoReq, HotelsPortalProperties config,
                                                   String geoSearchParams) {
        Preconditions.checkState(geoReq.getAdditionalParams() == null || geoReq.getAdditionalParams().isEmpty());
        var params = config.getGeosearchParams();
        params.putAll(deserializeGeoSearchParams(geoSearchParams));
        geoReq.setAdditionalParams(params);
    }

    public static void setAdditionalGeoSearchParams(GeoSearchReq geoReq, HotelsPortalProperties config) {
        setAdditionalGeoSearchParams(geoReq, config, null);
    }

    private static Map<String, String> deserializeGeoSearchParams(String value) {
        if (Strings.isNullOrEmpty(value)) {
            return Collections.emptyMap();
        }
        ObjectMapper mapper = new ObjectMapper();
        MapLikeType type = mapper.getTypeFactory().constructMapLikeType(HashMap.class, String.class, String.class);
        try {
            return mapper.readerFor(type).readValue(value);
        } catch (IOException e) {
            log.warn("Unable to load map of geosearch params", e);
            return Collections.emptyMap();
        }
    }

    public static BoundingBox extendBboxABit(BoundingBox bbox) {
        return bbox.extendByRelativeValue(0.05, 0.003);
    }

    public static SortTypeRegistry.SortTypeLayout getSortTypeLayout(UaasSearchExperiments uaasSearchExperiments) {

        // https://st.yandex-team.ru/TRAVELBACK-2643
        if (uaasSearchExperiments.isLeftFilters() || uaasSearchExperiments.isSearchFormOldProduction() || uaasSearchExperiments.isSearchFormOldProductionRound()) {
            return SortTypeRegistry.SortTypeLayout.WITH_SHORT_NAMES;
        }

        return SortTypeRegistry.SortTypeLayout.DEFAULT;
    }

    public static boolean isCatRoomForAllPartners(UaasSearchExperiments uaasSearchExperiments, Experiments experiments) {
        return uaasSearchExperiments.isCatRoomForAll() || experiments.isExp("catroom-for-all-partners");
    }

    public static boolean hasForceOfferDiscount(Experiments experiments) {
        return experiments.getIntValue(FORCE_OFFER_DISCOUNT_EXP) != null;
    }

    public static List<Badge> toBadges(List<TGetHotelsResponse.THotelBadge> hotelBadges, HotelsPortalProperties config) {
        // TODO: (mpivko) move it to geocounter
        var hotelBadgePriorities = config.getHotelBadgePriorities();
        return hotelBadges.stream()
                .map(BadgeUtils::extractBadge)
                .filter(x -> hotelBadgePriorities.containsKey(x.getId()))
                .sorted(Comparator.comparing(x -> -hotelBadgePriorities.get(x.getId())))
                .limit(HotelsPortalUtils.MAX_BADGES_PER_HOTEL)
                .collect(Collectors.toUnmodifiableList());
    }

    public static String getExplorationRankingCategory(UaasSearchExperiments uaasSearchExperiments,
                                                      Experiments experiments) {
        var expValue = experiments.getValue("ranking-exploration");
        if (expValue != null) {
            return expValue;
        }
        return uaasSearchExperiments.getRankingExploration();
    }

    public static SortTypeRegistry.RealTimeRankingType getRealTimeRanking(UaasSearchExperiments uaasSearchExperiments,
                                                                          Experiments experiments,
                                                                          boolean enableFakeCatBoostRanking) {
        if (experiments.isExp("emergency-catboost-disable")) {
            return SortTypeRegistry.RealTimeRankingType.NONE;
        }
        var expValue = uaasSearchExperiments.getOnlineRanking();
        if (expValue != null) {
            if (expValue.equals("no_boy_boost")) {
                return SortTypeRegistry.RealTimeRankingType.NO_BOY_BOOST;
            } else if (expValue.equals("catboost")) {
                return SortTypeRegistry.RealTimeRankingType.CATBOOST;
            }
            throw new RuntimeException(String.format("Unknown exp value: %s", expValue));
        } else {
            if (experiments.isExp("realtime-catboost-ranking")) {
                return SortTypeRegistry.RealTimeRankingType.CATBOOST;
            } else if (experiments.isExp("noboy-realtime-catboost-ranking")) {
                return SortTypeRegistry.RealTimeRankingType.NO_BOY_BOOST;
            } else if (experiments.isExp("fake-realtime-catboost-ranking") || enableFakeCatBoostRanking) {
                return SortTypeRegistry.RealTimeRankingType.FAKE_CATBOOST;
            } else {
                return SortTypeRegistry.RealTimeRankingType.NONE;
            }
        }
    }

    public static boolean isFilterByGeoIdEnabled(UaasSearchExperiments uaasSearchExperiments, Integer requestedGeoId,
                                                 CommonHttpHeaders headers) {
        return uaasSearchExperiments.isMoskowAreaEnabled() && requestedGeoId != null
                && (requestedGeoId == GeoBaseHelpers.MOSCOW_DISTRICT || requestedGeoId == GeoBaseHelpers.LENINGRAD_DISTRICT)
                && isUserDeviceDesktop(headers);
    }

    public static boolean isWhiteLabelActive(CommonHttpHeaders headers) {
        return headers.getWhiteLabelPartnerId() != EWhiteLabelPartnerId.WL_UNKNOWN;
    }

    public static <T> String formatNumber(T number, String suffix) {
        var separator = ' ';
        return new DecimalFormat("#,###.#", new DecimalFormatSymbols() {{
            setDecimalSeparator(',');
            setGroupingSeparator(separator); // non-breaking space
        }}).format(number) + separator + suffix;
    }

    public static String formatDistanceMeters(long distanceMeters) {
        if (distanceMeters >= 1000000) { // >= 1000km
            return formatNumber(distanceMeters / 1000, "км");
        } else if (distanceMeters >= 1000) { // >= 1km
            return formatNumber(Math.round(distanceMeters / 100.0) / 10.0, "км");
        } else {
            return formatNumber((int)Math.round(distanceMeters / 50.0) * 50, "м");
        }
    }
}
