package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.servlet.http.HttpServletRequest;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.MetricUtils;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.GeoSearchPollingResult;
import ru.yandex.travel.api.models.hotels.HotelWithOffers;
import ru.yandex.travel.api.models.hotels.SearchFilterAndTextParams;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geocounter.GeoCounterService;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.promo.CachedActivePromosService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.tugc.TugcService;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.PartnerConfigService;
import ru.yandex.travel.hotels.common.promo.mir.MirUtils;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;


@Component
@Slf4j
@RequiredArgsConstructor
public class HotelsSearchService implements InitializingBean {

    private final HotelsPortalProperties config;
    private final GeoSearchService geoSearchService;
    private final GeoCounterService geoCounter;
    private final PartnerConfigService partnerConfigService;
    private final GeoBase geoBase;
    private final HotelsFilteringService hotelsFilteringService;
    private final AmenityService amenityService;
    private final HotelSlugService hotelSlugService;
    private final RegionsService regionsService;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final HotelImagesService hotelImagesService;
    private final TugcService tugcService;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;
    private final CachedActivePromosService cachedActivePromosService;

    @Override
    public void afterPropertiesSet() throws Exception {
    }

    public CompletableFuture<SearchHotelsRspV1> search(HttpServletRequest httpServletRequest, SearchHotelsReqV1 req,
                                                       CommonHttpHeaders headers,
                                                       UserCredentials userCredentials,
                                                       AdditionalSearchHotelsLogData additionalSearchHotelsLogData) {
        var exps = new Experiments(req, config);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        var newSearchExp = exps.isExp("search-via-geocounter") || uaasSearchExperiments.isHotelsSearchViaGeoCounter();
        var allowNewSearch = (req.getSelectedSortId() == null || !req.getSelectedSortId().equals(SortTypeRegistry.SORT_BY_DISTANCE_ID)
                || exps.isExp("sort-by-distance-via-geocounter")) && config.isAllowNewSearch();

        var usingNewSearch = newSearchExp && allowNewSearch;

        MetricUtils.addRequestMetricTag(httpServletRequest, "new_search", Boolean.toString(usingNewSearch), true);

        if (usingNewSearch) {
            if (req.getNavigationToken() != null) {
                try {
                    Integer.parseInt(req.getNavigationToken());
                } catch (NumberFormatException e) {
                    req.setNavigationToken(null);
                    req.setContext(null);
                }
            }
            if (req.getContext() != null && !req.getContext().contains("newsearch")) {
                req.setContext(null);
            }
            var searcher = new ViaGeocounterSearcher(config, geoCounter, geoSearchService, geoBase,
                    hotelsFilteringService, amenityService, hotelSlugService, regionsService, tugcService,
                    uaasExperimentDataProvider, hotelImagesService, req, headers, userCredentials,
                    imageWhitelistDataProvider, additionalSearchHotelsLogData, cachedActivePromosService);
            return searcher.run();
        } else {
            if (req.getNavigationToken() != null) {
                try {
                    Integer.parseInt(req.getNavigationToken());
                    req.setNavigationToken(null);
                    req.setContext(null);
                } catch (NumberFormatException ignored) {
                }
            }
            if (req.getContext() != null && req.getContext().contains("newsearch")) {
                req.setContext(null);
            }
        }

        ProcessingState pState = new ProcessingState(req, headers, additionalSearchHotelsLogData, uaasSearchExperiments,
                userCredentials, allowNewSearch);
        try {
            return pState.run();
        } catch (IllegalArgumentException exc) {
            log.error("{}: Request validation failed: {}", pState.logId, exc.getMessage());
            return CompletableFuture.failedFuture(exc);
        } catch (TravelApiBadRequestException exc) {
            log.error("{}: Request validation failed: {}", pState.logId, exc.getMessage());
            return CompletableFuture.failedFuture(exc);
        } catch (Exception exc) {
            log.error("{}: Unknown error", pState.logId, exc);
            return CompletableFuture.failedFuture(exc);
        }
    }

    private class ProcessingState {
        private final Instant started;
        private final SearchHotelsReqV1 req;
        private final AdditionalSearchHotelsLogData additionalSearchHotelsLogData;

        private final CompletableFuture<SearchHotelsRspV1> resultFuture;
        private final SortTypeRegistry sortTypeRegistry;
        private final Experiments experiments;
        private final UaasSearchExperiments uaasSearchExperiments;
        private String logId = null;
        private GeoSearchPoller poller;
        private final boolean newSearchPossible;
        private final CommonHttpHeaders headers;

        ProcessingState(SearchHotelsReqV1 req, CommonHttpHeaders headers,
                        AdditionalSearchHotelsLogData additionalSearchHotelsLogData,
                        UaasSearchExperiments uaasSearchExperiments, UserCredentials userCredentials,
                        boolean newSearchPossible) {
            this.started = Instant.now();
            this.req = req;
            this.additionalSearchHotelsLogData = additionalSearchHotelsLogData;
            this.resultFuture = new CompletableFuture<>();
            this.experiments = new Experiments(req, config);
            this.uaasSearchExperiments = uaasSearchExperiments;
            this.newSearchPossible = newSearchPossible;
            this.headers = headers;
            sortTypeRegistry = SortTypeRegistry.getSortTypeRegistry(
                    HotelsPortalUtils.isHotelsNearbyEnabled(headers),
                    HotelsPortalUtils.getSortTypeLayout(uaasSearchExperiments),
                    false, false, null, SortTypeRegistry.RealTimeRankingType.NONE); // GeoSearch doesn't support new ranking
            this.poller = new GeoSearchPoller(req, headers, config, geoSearchService, partnerConfigService,
                    hotelsFilteringService, geoBase, amenityService, hotelSlugService, hotelImagesService, experiments,
                    sortTypeRegistry, started, uaasSearchExperiments, tugcService, userCredentials,
                    imageWhitelistDataProvider, cachedActivePromosService);
        }

        private CompletableFuture<SearchHotelsRspV1> run() {
            try {
                poller.prepare();
            } finally {
                logId = poller.getLogId();
            }
            var geoSearchPollingResults = poller.run();
            CompletableFuture<CountHotelsRspV1> geoCounterFuture =
                    geoSearchPollingResults.getOfferSearchParamsWithBboxFuture()
                            .thenCompose(offerSearchParamsWithBbox -> HotelSearchUtils.doGeoCounterRequest(
                                    log, logId, started, req, hotelsFilteringService, uaasSearchExperiments,
                                    experiments, geoCounter, offerSearchParamsWithBbox, headers
                            ));
            CompletableFuture.allOf(geoSearchPollingResults.getPollingResultFuture(), geoCounterFuture)
                    .whenComplete((ignored, ignoredT) -> {
                CountHotelsRspV1 countRsp;
                try {
                    countRsp = geoCounterFuture.join();
                } catch (Exception exc) {
                    HotelsPortalUtils.incrementCounter("hotels.search.geoCounterError");
                    log.error("{}: Unable to complete geoCounter request", logId, exc);
                    countRsp = null;
                }

                GeoSearchPollingResult gsResult;
                try {
                    gsResult = geoSearchPollingResults.getPollingResultFuture().join();
                } catch (Exception exc) {
                    HotelsPortalUtils.incrementCounter("hotels.search.geoSearchError");
                    resultFuture.completeExceptionally(exc);
                    return;
                }

                SearchHotelsRspV1 searchRsp = prepareResponse(gsResult, countRsp);
                resultFuture.complete(searchRsp);

                reportHotelSearchStats(gsResult, searchRsp);
            });
            return resultFuture;
        }

        private void reportHotelSearchStats(GeoSearchPollingResult gsResult, SearchHotelsRspV1 searchRsp) {
            try {
                var hotelSearchStats = new HotelSearchStats();
                hotelSearchStats.setLogId(logId);
                hotelSearchStats.setPollingFinished(gsResult.isPollingFinished());
                hotelSearchStats.setHasTrueResultSize(gsResult.getTotalPricedHotelsCountAmongAllPages() != null);
                hotelSearchStats.setHasFilters(req.getFilterAtoms() != null && !req.getFilterAtoms().isEmpty());
                hotelSearchStats.setPollingFinishReason(gsResult.getPollingFinishReason());
                hotelSearchStats.setTotalPollingTime(gsResult.getTotalPollingTime());
                hotelSearchStats.setIterationTime(gsResult.getIterationTime());
                hotelSearchStats.setEpoch(req.getPollEpoch());
                hotelSearchStats.setIteration(req.getPollIteration());
                hotelSearchStats.setTotalRetries(gsResult.getTotalRetries());
                hotelSearchStats.setHeadHotelsCount(gsResult.getHeadHotels().size());
                hotelSearchStats.setTailHotelsCount(gsResult.getTailHotels().size());
                hotelSearchStats.setFoundHotelCount(searchRsp.getFoundHotelCount());
                hotelSearchStats.setTotalPricedHotelsOnAllGeoPages(gsResult.getTotalPricedHotelsCountAmongAllPages());
                hotelSearchStats.setTotalHotelsOnAllGeoPages(gsResult.getTotalHotelsCountAmongAllPages());
                hotelSearchStats.report();
            } catch (Exception e) {
                log.warn("Failed to report search stats: {}", e.getMessage());
            }
        }

        private SearchHotelsRspV1 prepareResponse(GeoSearchPollingResult gsResult, CountHotelsRspV1 countResponse) {
            List<HotelWithOffers> rspAllHotels = new ArrayList<>();
            rspAllHotels.addAll(gsResult.getHeadHotels());
            rspAllHotels.addAll(gsResult.getTailHotels());

            additionalSearchHotelsLogData.setHasOfferCacheMeta(!gsResult.getOfferCacheMeta().isFake());
            additionalSearchHotelsLogData.setPendingPartners(gsResult.getOfferCacheMeta().getProgress().getFinishedPartners());
            additionalSearchHotelsLogData.setFinishedPartners(gsResult.getOfferCacheMeta().getProgress().getPendingPartners());

            StringBuilder sb = new StringBuilder();
            for (HotelWithOffers hotel : rspAllHotels) {
                sb.append(hotel.getHotel().getPermalink());
                sb.append(" ");
                if (hotel.isSearchIsFinished()) {
                    sb.append("F(");
                    sb.append(hotel.getOffers().size());
                    sb.append(")");
                } else {
                    sb.append("I");
                }
                sb.append("; ");
            }
            log.info("{}: Sent hotels ({} priced): {}", logId, gsResult.getHeadHotels().size(), sb.toString());

            // HOTELS-5746 добавляем логику вывода если тип питания есть или только полную отменяемость для свойств оффера
            if (experiments.isExp("HOTELS-5746")) {
                rspAllHotels.forEach(hotel -> {
                    var offers = hotel.getOffers();
                    offers.forEach(offer -> {
                        HotelSearchUtils.setCancellationInfoNullIfNotFullyRefundable(offer);
                        HotelSearchUtils.setMealTypeNullIfNotMealTypeOrUnknown(offer);
                    });
                });
            }

            SearchHotelsRspV1 rsp = new SearchHotelsRspV1();
            rsp.setHotels(rspAllHotels);
            rsp.setPricedHotelCount(gsResult.getHeadHotels().size());
            rsp.setOfferSearchProgress(gsResult.getOfferCacheMeta().getProgress());
            rsp.getOfferSearchProgress().setFinished(gsResult.isPollingFinished()); //patch!
            if (!rsp.getOfferSearchProgress().isFinished()) {
                rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMs(config, experiments, req.getPollIteration()));
            }
            rsp.setOfferSearchParams(gsResult.getOfferCacheMeta().getOfferSearchParams());
            rsp.setOperatorById(gsResult.getOfferCacheMeta().getOperatorById());
            rsp.setPollEpoch(req.getPollEpoch());
            rsp.setPollIteration(req.getPollIteration());
            rsp.setSearchPagePollingId(gsResult.getPollingId());

            Boolean hasBoyOffers = gsResult.getHasBoyOffers();
            if (hasBoyOffers != null && gsResult.isPollingFinished()) {
                rsp.setHasBoyOffers(hasBoyOffers);
            }
            rsp.setBboxAsString(gsResult.getBbox());
            List<Coordinates> bboxAsList = new ArrayList<>();
            bboxAsList.add(gsResult.getBbox().getLeftDown());
            bboxAsList.add(gsResult.getBbox().getUpRight());

            rsp.setBboxAsStruct(bboxAsList);
            rsp.setRegion(HotelSearchUtils.fillRegion(regionsService, req.getDomain(), gsResult.getActualGeoId()));// Derpecated
            rsp.setActualRegion(HotelSearchUtils.fillShortRegion(geoBase, req.getDomain(), gsResult.getActualGeoId()));
            rsp.getRegion().getLinguistics().setNominativeCase(rsp.getActualRegion().getName());// Very bad and temporary workaround
            rsp.setSearchRegion(HotelSearchUtils.fillRegion(regionsService, req.getDomain(),
                    gsResult.getSearchGeoId()));
            rsp.setContext(gsResult.getContext());

            SearchHotelsRspV1.NavigationTokens navigationTokens = new SearchHotelsRspV1.NavigationTokens();
            navigationTokens.setPrevPage(gsResult.getPrevNavigToken());
            navigationTokens.setNextPage(gsResult.getNextNavigToken());
            navigationTokens.setCurrentPage(gsResult.getCurrentNavigToken());
            rsp.setNavigationTokens(navigationTokens);

            SearchFilterAndTextParams filterParams = new SearchFilterAndTextParams();
            filterParams.setFilterAtoms(req.getFilterAtoms());
            filterParams.setFilterPriceFrom(req.getFilterPriceFrom());
            filterParams.setFilterPriceTo(req.getFilterPriceTo());
            var filterInfo = hotelsFilteringService.composeDefaultFilterInfo(req, started,
                    HotelSearchUtils.getForceFilterLayout(req), gsResult.getSearchGeoId(), req.isOnlyCurrentGeoId(), headers);
            filterInfo.setParams(filterParams);
            rsp.setFilterInfo(filterInfo);
            rsp.setSearchControlInfo(hotelsFilteringService.composeControlInfo(req, started,
                    HotelSearchUtils.getForceFilterLayout(req), gsResult.getSearchGeoId(), headers));

            // Add geoCounter data
            if (countResponse != null) {
                log.info("{}: Found hotels (geosearch): {}, Found hotels (geocounter): {}", logId,
                        gsResult.getGeoRespFoundHotelsCount(), countResponse.getFoundHotelCount());
                if (gsResult.getTotalPricedHotelsCountAmongAllPages() != null) {
                    rsp.setFoundHotelCount(gsResult.getTotalPricedHotelsCountAmongAllPages());
                } else {
                    rsp.setFoundHotelCount(countResponse.getFoundHotelCount());
                }
                rsp.getFilterInfo().setQuickFilters(countResponse.getFilterInfo().getQuickFilters());
                rsp.getFilterInfo().setDetailedFilters(countResponse.getFilterInfo().getDetailedFilters());
                rsp.getFilterInfo().setDetailedFiltersBatches(countResponse.getFilterInfo().getDetailedFiltersBatches());
                rsp.getFilterInfo().setPriceFilter(countResponse.getFilterInfo().getPriceFilter());
                rsp.setSearchControlInfo(countResponse.getSearchControlInfo());
            } else {
                if (gsResult.getTotalPricedHotelsCountAmongAllPages() != null) {
                    rsp.setFoundHotelCount(gsResult.getTotalPricedHotelsCountAmongAllPages());
                } else {
                    rsp.setFoundHotelCount(gsResult.getGeoRespFoundHotelsCount());
                }
            }

            // Add sort info
            rsp.setSortInfo(HotelSearchUtils.buildSortInfo(sortTypeRegistry, gsResult.getSelectedSortId(),
                    gsResult.getSortOrigin()));

            if (MirUtils.isActive(cachedActivePromosService.getActivePromos())) {
                if (experiments.isExp("1349") && gsResult.isHasMirFilter()) {
                    rsp.setSearchBannerType(SearchHotelsRspV1.ESearchBannerType.MIR_ENABLED);
                } else {
                    rsp.setSearchBannerType(SearchHotelsRspV1.ESearchBannerType.MIR);
                }
            }

            var pageParams = ExtraVisitAndUserParamsUtils.initParamsMap();
            ExtraVisitAndUserParamsUtils.fillSearchParams(pageParams, rsp.getOfferSearchParams());
            ExtraVisitAndUserParamsUtils.fillGeoParams(pageParams, req.getDomain(), gsResult.getActualGeoId(), geoBase);
            pageParams.put("lastUsedSort", gsResult.getSelectedSortId());
            pageParams.put("topHotelSelected", Boolean.toString(gsResult.isTopHotelSelected()));
            pageParams.put("newSearchUsed", Boolean.toString(false));
            pageParams.put("newSearchPossible", Boolean.toString(newSearchPossible));

            var pollingId = gsResult.getPollingId();
            if (pollingId != null) {
                pageParams.put("pollingId", pollingId);
            }

            ExtraVisitAndUserParamsUtils.fillFilterParams(pageParams, filterParams);

            rsp.setExtraVisitAndUserParams(ExtraVisitAndUserParamsUtils.createForSearchPage(pageParams));

            rsp.setTimingInfo(new SearchHotelsRspV1.TimingInfo(Duration.between(started, Instant.now()).toMillis()));
            if (gsResult.isTopHotelSelected()) {
                rsp.setTopHotelSlug(req.getTopHotelSlug());
            }
            return rsp;
        }
    }
}
