package ru.yandex.travel.api.endpoints.hotels_portal;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import NUgc.NSchema.Moderation;
import NUgc.NSchema.NOrgs.OrgDigest;
import NUgc.NSchema.NOrgs.OrgPhoto;
import NUgc.NSchema.NOrgs.OrgReview;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.api.models.hotels.HotelImage;
import ru.yandex.travel.api.models.hotels.HotelReviewsInfo;
import ru.yandex.travel.api.models.hotels.ModerationStatus;
import ru.yandex.travel.api.models.hotels.ReviewImageSize;
import ru.yandex.travel.api.models.hotels.UserReactionType;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDigestRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcUploadImageRsp;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.Permalink;

@Slf4j
public class ReviewExtractor {
    public static HotelReviewsInfo.TextReview extractHotelReview(OrgReview.TOrgReview orgReview, Permalink permalink, String reqId,
                                                                boolean isCurrentUserReview, String keyPhraseFilter) {
        HotelReviewsInfo.TextReview textReview = new HotelReviewsInfo.TextReview();
        textReview.setId(orgReview.getReviewId());
        textReview.setText(orgReview.getText());
        if (orgReview.hasAuthor()) {
            textReview.setAuthor(extractHotelReviewAuthor(orgReview));
        }

        textReview.setUpdatedAt(ProtoUtils.toInstant(orgReview.getUpdatedTime()));
        textReview.setRating(orgReview.getRating());
        textReview.setTotalLikeCount(orgReview.getLikeCount());
        textReview.setTotalDislikeCount(orgReview.getDislikeCount());
        textReview.setUserReaction(extractHotelReviewUserReactionType(orgReview, permalink, reqId));
        textReview.setBusinessComment(null);

        textReview.setImages(extractHotelReviewImages(orgReview, isCurrentUserReview, permalink, reqId));
        textReview.setCommentCount(orgReview.getCommentCount());
        textReview.setModeration(extractHotelReviewModeration(orgReview, isCurrentUserReview, permalink, reqId));
        textReview.setKeyPhraseMatch(extractKeyPhraseMatch(orgReview, keyPhraseFilter));
        textReview.setSnippet(orgReview.getSnippet());

        return textReview;
    }

    public static HotelReviewsInfo extractHotelReviewsInfo(UgcDigestRsp ugcDigestRsp, Permalink permalink,
                                                           String reqId, int keyPhraseLimit,
                                                           int textReviewLimit, int textReviewOffset,
                                                           int maxTextReviewOffset, String keyPhraseFilter) {
        HotelReviewsInfo reviewsInfo = new HotelReviewsInfo();
        reviewsInfo.setTotalTextReviewCount(ugcDigestRsp.getDigest().getParams().getCount());
        List<HotelReviewsInfo.TextReview> textReviews = new LinkedList<>();
        reviewsInfo.setTextReviews(textReviews);
        for (OrgReview.TOrgReview orgReview : ugcDigestRsp.getDigest().getReviewsList()) {
            textReviews.add(extractHotelReview(orgReview, permalink, reqId, false, keyPhraseFilter));
        }
        //map key phrases
        reviewsInfo.setTotalKeyPhraseCount(ugcDigestRsp.getDigest().getKeyPhrasesCount());
        reviewsInfo.setKeyPhrases(extractHotelReviewKeyPhrases(ugcDigestRsp.getDigest(), keyPhraseLimit));

        if (ugcDigestRsp.getDigest().hasMyReview() && !ugcDigestRsp.getDigest().getMyReview().getReviewId().isEmpty()) {
            OrgReview.TOrgReview userTextReview = ugcDigestRsp.getDigest().getMyReview();
            reviewsInfo.setUserTextReview(extractHotelReview(userTextReview, permalink, reqId, true, keyPhraseFilter));
        }

        int nextOffset = textReviewOffset + textReviewLimit;
        reviewsInfo.setHasMore(nextOffset < maxTextReviewOffset && nextOffset < reviewsInfo.getTotalTextReviewCount());
        return reviewsInfo;
    }

    public static List<ReviewImageSize> extractUploadedHotelReviewImageSizes(List<UgcUploadImageRsp.Size> ugcSizes) {
        List<ReviewImageSize> sizes = new LinkedList<>();

        for (UgcUploadImageRsp.Size imageSize: ugcSizes) {
            ReviewImageSize reviewImageSize = new ReviewImageSize();
            reviewImageSize.setUrl(imageSize.getUrl());
            reviewImageSize.setSize(imageSize.getName());
            reviewImageSize.setHeight(imageSize.getHeight());
            reviewImageSize.setWidth(imageSize.getWidth());

            sizes.add(reviewImageSize);
        }

        return sizes;
    }

    private static HotelReviewsInfo.TextReview.Author extractHotelReviewAuthor(OrgReview.TOrgReview orgReview) {
        HotelReviewsInfo.TextReview.Author author = new HotelReviewsInfo.TextReview.Author();
        author.setAvatarUrl(orgReview.getAuthor().getAvatarUrl());
        author.setProfileUrl(orgReview.getAuthor().getProfileUrl());
        author.setName(orgReview.getAuthor().getName());
        author.setLevel(orgReview.getAuthor().getProfessionLevel());

        return author;
    }

    private static UserReactionType extractHotelReviewUserReactionType(OrgReview.TOrgReview orgReview, Permalink permalink, String reqId) {
        switch (orgReview.getUserReaction()) {
            case NONE:
                return UserReactionType.NONE;
            case LIKE:
                return UserReactionType.LIKE;
            case DISLIKE:
                return UserReactionType.DISLIKE;
            default:
                log.warn("Unknown user reaction for review id {} for permalink {}, reqId is {}",
                        orgReview.getReviewId(), permalink, reqId);
                return UserReactionType.NONE;
        }
    }

    private static List<HotelImage> extractHotelReviewImages(OrgReview.TOrgReview orgReview, boolean isCurrentUserReview, Permalink permalink, String reqId) {
        return orgReview.getPhotosList().stream().map((OrgPhoto.TOrgPhoto orgPhoto) -> {
            boolean isVisibleImage = isCurrentUserReview || (orgPhoto.hasModeration() && orgPhoto.getModeration().getStatus() == Moderation.TModerationStatus.EType.ACCEPTED);

            if (!isVisibleImage) {
                return null;
            }

            HotelImage image = new HotelImage();
            image.setId(orgPhoto.getPhotoId());
            image.setUrlTemplate(orgPhoto.getUrlTemplate());

            if (orgPhoto.hasModeration()) {
                HotelImage.HotelImageModeration moderation = new HotelImage.HotelImageModeration();
                String logMessage = "Unknown review image moderation status for image id " + orgPhoto.getPhotoId() + " for review id " + orgReview.getReviewId()
                        + " for permalink " + permalink + ", reqId is " + reqId;
                moderation.setStatus(convertUgcModerationStatusToModerationStatus(orgPhoto.getModeration().getStatus(), logMessage));
                image.setModeration(moderation);
            }

            return image;
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private static List<HotelReviewsInfo.KeyPhrase> extractHotelReviewKeyPhrases(OrgDigest.TOrgDigest orgDigest, int keyPhraseLimit) {
        List<HotelReviewsInfo.KeyPhrase> keyPhrases = new LinkedList<>();

        if (orgDigest.getParams().getCount() <= 0) {
            return keyPhrases;
        }

        for (OrgDigest.TOrgKeyPhrase ugcKeyPhrase : orgDigest.getKeyPhrasesList()) {
            if (keyPhrases.size() >= keyPhraseLimit) {
                break;
            }
            HotelReviewsInfo.KeyPhrase keyPhrase = new HotelReviewsInfo.KeyPhrase();
            keyPhrase.setName(ugcKeyPhrase.getKey());
            keyPhrase.setReviewCount(ugcKeyPhrase.getCount());
            keyPhrases.add(keyPhrase);
        }

        return keyPhrases.stream()
                .sorted((lKeyPhrase, rKeyPhrase) -> Integer.compare(rKeyPhrase.getReviewCount(), lKeyPhrase.getReviewCount()))
                .collect(Collectors.toList());
    }

    private static HotelReviewsInfo.TextReview.Moderation extractHotelReviewModeration(OrgReview.TOrgReview orgReview, boolean isCurrentUserReview,
                                                                                       Permalink permalink, String reqId) {
        if (!isCurrentUserReview) {
            return null;
        }

        HotelReviewsInfo.TextReview.Moderation moderation = new HotelReviewsInfo.TextReview.Moderation();

        if (orgReview.hasModeration() && orgReview.getModeration().getStatus() != null) {
            String logMessage = "Unknown text review moderation status for review id " + orgReview.getReviewId()
                    + " for permalink " + permalink + ", reqId is " + reqId;
            moderation.setStatus(convertUgcModerationStatusToModerationStatus(orgReview.getModeration().getStatus(), logMessage));
        }

        return moderation;
    }

    private static ModerationStatus convertUgcModerationStatusToModerationStatus(Moderation.TModerationStatus.EType status, String logMessage) {
        switch (status) {
            case ACCEPTED:
                return ModerationStatus.ACCEPTED;
            case UNRECOGNIZED:
            case IN_PROGRESS:
                return ModerationStatus.IN_PROGRESS;
            case DECLINED:
                return ModerationStatus.DECLINED;
            default:
                log.warn(logMessage);
                return ModerationStatus.IN_PROGRESS;
        }
    }

    private static HotelReviewsInfo.TextReview.KeyPhraseMatch extractKeyPhraseMatch(OrgReview.TOrgReview orgReview, String keyPhraseFilter) {
        if (keyPhraseFilter == null) {
            return null;
        }

        OrgReview.TKeyPhrase keyPhrase = orgReview.getKeyPhrasesList().stream()
                .filter((OrgReview.TKeyPhrase orgReviewKeyPhrase) -> orgReviewKeyPhrase.getKeyPhrase().equals(keyPhraseFilter))
                .findAny().orElse(null);

        if (keyPhrase == null) {
            return null;
        }

        HotelReviewsInfo.TextReview.KeyPhraseMatch keyPhraseMatch = new HotelReviewsInfo.TextReview.KeyPhraseMatch();

        keyPhraseMatch.setFragments(keyPhrase.getFragmentList().stream().map((OrgReview.TKeyPhrase.TFragment orgReviewFragment) -> {
            HotelReviewsInfo.TextReview.KeyPhraseMatch.Fragment fragment = new HotelReviewsInfo.TextReview.KeyPhraseMatch.Fragment();

            fragment.setSize(orgReviewFragment.getSize());
            fragment.setPosition(orgReviewFragment.getPosition());

            return fragment;
        }).collect(Collectors.toList()));

        return keyPhraseMatch;
    }
}
