package ru.yandex.travel.api.endpoints.hotels_portal

import NUgc.NSchema.Reaction.TReviewReaction.EType
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import org.springframework.web.multipart.MultipartFile
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException
import ru.yandex.travel.api.models.hotels.ETextReviewRanking
import ru.yandex.travel.api.models.hotels.ETextReviewRanking.byLikesCountDesc
import ru.yandex.travel.api.models.hotels.ETextReviewRanking.byRatingAsc
import ru.yandex.travel.api.models.hotels.ETextReviewRanking.byRatingDesc
import ru.yandex.travel.api.models.hotels.ETextReviewRanking.byRelevanceOrg
import ru.yandex.travel.api.models.hotels.ETextReviewRanking.byTime
import ru.yandex.travel.api.models.hotels.UserReactionType
import ru.yandex.travel.api.models.hotels.UserReactionType.DISLIKE
import ru.yandex.travel.api.models.hotels.UserReactionType.LIKE
import ru.yandex.travel.api.models.hotels.UserReactionType.NONE
import ru.yandex.travel.api.services.hotels.ugc.UgcService
import ru.yandex.travel.api.services.hotels.ugc.UgcServiceProperties
import ru.yandex.travel.api.services.hotels.ugc.model.UgcAddReviewReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcAttribution
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDeletePhotoReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDeleteReviewReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDigestReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDigestRsp
import ru.yandex.travel.api.services.hotels.ugc.model.UgcEditReviewReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcChangeReviewRsp
import ru.yandex.travel.api.services.hotels.ugc.model.UgcGetCurrentUserReviewReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcRanking
import ru.yandex.travel.api.services.hotels.ugc.model.UgcRanking.BY_LIKES_COUNT_DESC
import ru.yandex.travel.api.services.hotels.ugc.model.UgcRanking.BY_RATING_ASC
import ru.yandex.travel.api.services.hotels.ugc.model.UgcRanking.BY_RATING_DESC
import ru.yandex.travel.api.services.hotels.ugc.model.UgcRanking.BY_RELEVANCE_ORG
import ru.yandex.travel.api.services.hotels.ugc.model.UgcRanking.BY_TIME
import ru.yandex.travel.api.services.hotels.ugc.model.UgcSetReviewReactionReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcSetReviewReactionRsp
import ru.yandex.travel.api.services.hotels.ugc.model.UgcUploadImageReq
import ru.yandex.travel.api.services.hotels.ugc.model.UgcUploadImageRsp
import ru.yandex.travel.credentials.UserCredentials
import ru.yandex.travel.hotels.common.Permalink
import java.util.UUID
import java.util.concurrent.CompletableFuture

@Component
class ReviewService(
    @Autowired private val ugcService: UgcService,
    @Autowired private val config: UgcServiceProperties
) {
    private val log = LoggerFactory.getLogger(this.javaClass)

    fun uploadImage(image: MultipartFile, permalink: Permalink, reqId: String, isEnabledTestUgc: Boolean = false): CompletableFuture<UgcUploadImageRsp> {
        val req = UgcUploadImageReq(
            image = image,
            permalink = permalink,
            attribution = createUgcAttribution(reqId),
            isEnabledTestUgc = isEnabledTestUgc
        )

        checkAuthorization(req.attribution, permalink, "UploadImage")

        return ugcService.uploadPhoto(req)
    }

    fun deleteImages(imageIds: List<String>, permalink: Permalink, reqId: String, isEnabledTestUgc: Boolean = false): CompletableFuture<Void> {
        val reqs = imageIds.map { imageId -> UgcDeletePhotoReq(
            attribution = createUgcAttribution(reqId),
            imageId = imageId,
            permalink = permalink,
            isEnabledTestUgc = isEnabledTestUgc
        ) }

        checkAuthorization(reqs.first().attribution, permalink, "DeleteImages")

        val futures = reqs.map(ugcService::deletePhoto).toTypedArray()

        return CompletableFuture.allOf(*futures)
    }

    fun deleteReview(permalink: Permalink, reviewId: String, reqId: String, isEnabledTestUgc: Boolean = false): CompletableFuture<Void> {
        val req = UgcDeleteReviewReq(
            attribution = createUgcAttribution(reqId),
            permalink = permalink,
            reviewId = reviewId,
            isEnabledTestUgc = isEnabledTestUgc
        )

        checkAuthorization(req.attribution, permalink, "DeleteMyReview")

        return ugcService.deleteReview(req)
    }

    fun getDigest(permalink: Permalink, reqId: String, textReviewLimit: Int, textReviewOffset: Int, keyPhrase: String?,
            ranking: ETextReviewRanking, maxTextReviewOffset: Int, isEnabledTestUgc: Boolean = false): CompletableFuture<UgcDigestRsp> {
        if (textReviewLimit <= 0) {
            throw TravelApiBadRequestException("textReviewLimit should be > 0")
        }
        if (textReviewOffset % textReviewLimit != 0) {
            throw TravelApiBadRequestException("textReviewOffset must be multiply of textReviewLimit")
        }
        if (textReviewOffset >= maxTextReviewOffset) {
            throw TravelApiBadRequestException("textReviewOffset limit exceeded")
        }

        val ugcAttribution = createUgcAttribution(reqId)
        val ugcReq = UgcDigestReq(
            attribution = ugcAttribution,
            permalink = permalink,
            limit = textReviewLimit,
            offset = textReviewOffset,
            isIncludeMyReview = true,
            isIncludeMyReactions = true,
            isIncludeKeyPhrases = true,
            isIncludeSnippet = true,
            keyPhrase = keyPhrase,
            ranking = convertTextReviewRankingToUgcRanking(ranking),
            isEnabledTestUgc = isEnabledTestUgc
        )

        val digestFuture = ugcService.getDigest(ugcReq)

        if (!isEnabledTestUgc || !config.isEnabledTestUgc || ugcAttribution.passportUid == null) {
            return digestFuture
        }

        val ugcGetCurrentUserReviewReq = UgcGetCurrentUserReviewReq(
            attribution = createUgcAttribution(reqId),
            permalink = permalink,
            isEnabledTestUgc = isEnabledTestUgc
        )

        val digestRsp = digestFuture.join()

        return ugcService.getCurrentUserReview(ugcGetCurrentUserReviewReq).thenCompose {
            it?.let {
                digestRsp.digest = digestRsp.digest.toBuilder()
                    .setMyReview(it.review)
                    .build()
            }

            CompletableFuture.completedFuture(digestRsp)
        }
    }

    fun setReviewReaction(permalink: Permalink, reviewId: String, userReaction: UserReactionType, reqId: String, isEnabledTestUgc: Boolean = false): CompletableFuture<UgcSetReviewReactionRsp> {
        val methodName = "SetHotelReviewReaction"
        val req = UgcSetReviewReactionReq(
            attribution = createUgcAttribution(reqId),
            reviewId = reviewId,
            permalink = permalink,
            type = when (userReaction) {
                NONE -> EType.NONE
                LIKE -> EType.LIKE
                DISLIKE -> EType.DISLIKE
            },
            isEnabledTestUgc = isEnabledTestUgc
        )

        checkAuthorization(req.attribution, permalink, methodName)

        return ugcService.setReviewReaction(req)
    }

    fun changeReview(permalink: Permalink, reviewId: String?, reviewText: String?, reviewRating: Int?, imageIds: List<String>, reqId: String, isEnabledTestUgc: Boolean): CompletableFuture<UgcChangeReviewRsp> {
        return if (reviewId == null) {
            addReview(permalink, reviewText, reviewRating, imageIds, reqId, isEnabledTestUgc)
        } else {
            if (reviewText.isNullOrEmpty()) {
                throw IllegalStateException("Cannot change hotel review text to empty. ReqId: $reqId")
            }

            editReview(permalink, reviewId, reviewText, reviewRating, imageIds, reqId, isEnabledTestUgc)
        }
    }

    private fun addReview(permalink: Permalink, reviewText: String?, reviewRating: Int?, imageIds: List<String>, reqId: String, isEnabledTestUgc: Boolean = false): CompletableFuture<UgcChangeReviewRsp> {
        val req = UgcAddReviewReq(
            attribution = createUgcAttribution(reqId),
            permalink = permalink,
            text = reviewText,
            imageIds = imageIds,
            rating = reviewRating,
            isEnabledTestUgc = isEnabledTestUgc
        )

        checkAuthorization(req.attribution, permalink, "AddMyReview")

        return ugcService.addReview(req)
    }

    private fun editReview(permalink: Permalink, reviewId: String, reviewText: String, reviewRating: Int?, imageIds: List<String>, reqId: String, isEnabledTestUgc: Boolean): CompletableFuture<UgcChangeReviewRsp> {
        val req = UgcEditReviewReq(
            attribution = createUgcAttribution(reqId),
            permalink = permalink,
            text = reviewText,
            imageIds = imageIds,
            rating = reviewRating,
            reviewId = reviewId,
            isEnabledTestUgc = isEnabledTestUgc
        )

        checkAuthorization(req.attribution, permalink, "EditMyReview")

        return ugcService.editReview(req)
    }

    private fun convertTextReviewRankingToUgcRanking(textReviewRanking: ETextReviewRanking): UgcRanking {
        return when (textReviewRanking) {
            byTime -> BY_TIME
            byLikesCountDesc -> BY_LIKES_COUNT_DESC
            byRatingAsc -> BY_RATING_ASC
            byRatingDesc -> BY_RATING_DESC
            byRelevanceOrg -> BY_RELEVANCE_ORG
        }
    }

    private fun createUgcAttribution(reqId: String): UgcAttribution {
        val uc = UserCredentials.get()
        val attribution = UgcAttribution(
            reqId = generateReqId(reqId),
            yandexUid = uc?.yandexUid,
            passportUid = uc?.passportId,
            userIp = uc?.userIp
        )

        if (uc == null) {
            log.warn("UGC Req {}: UserCredentials not accessible", reqId)
        }

        return attribution
    }

    private fun generateReqId(parentReqId: String): String {
        return "$parentReqId/${UUID.randomUUID()}"
    }

    private fun checkAuthorization(attribution: UgcAttribution, permalink: Permalink, methodName: String) {
        if (attribution.passportUid == null) {
            throw TravelApiBadRequestException("Cannot execute $methodName for permalink $permalink, because passportId is not given")
        }
    }
}
