package ru.yandex.travel.api.endpoints.hotels_portal;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.proto.geocounter_service.ESortType;

public class SortTypeRegistry {

    public static final String SORT_BY_DISTANCE_ID = "near-first";

    private static final Map<SortTypeLayout, SortTypeNames> sortTypeNamesByLayout = Stream.of(
        new AbstractMap.SimpleImmutableEntry<>(SortTypeLayout.WITH_SHORT_NAMES, new SortTypeNames(
                "популярные", "дешевые", "дорогие", "с высоким рейтингом", "ближайшие")),
        new AbstractMap.SimpleImmutableEntry<>(SortTypeLayout.DEFAULT, new SortTypeNames(
                "Популярные", "Сначала дешевле", "Сначала дороже", "Высокий рейтинг", "Ближайшие"))
    ).collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));

    public enum SortTypeLayout {
        DEFAULT,
        WITH_SHORT_NAMES
    }

    public enum RealTimeRankingType {
        NONE,
        NO_BOY_BOOST,
        CATBOOST,
        FAKE_CATBOOST;
    }

    @RequiredArgsConstructor
    private static class SortTypeNames {
        @Getter
        private final String relevantFirst;

        @Getter
        private final String cheapFirst;

        @Getter
        private final String expensiveFirst;

        @Getter
        private final String highRatingFirst;

        @Getter
        private final String nearFirst;
    }

    @RequiredArgsConstructor
    public static class SortType {
        @Getter
        private final String id;

        @Getter
        private final String name;

        @Getter
        private final SearchHotelsRspV1.SortTypeHint hint;

        @Getter
        private final boolean byPrice;

        @Getter
        private final boolean byDistance;

        @Getter
        private final GeoSearchReq.SortType geoSortType;

        @Getter
        private final ESortType geoCounterSortType;

        private final Set<ESortType> alternateGeoCounterSortTypes;

        public boolean supportsGeoCounterSortType(ESortType st) {
            return geoCounterSortType == st || alternateGeoCounterSortTypes.contains(st);
        }
    }

    @RequiredArgsConstructor
    public static class SortTypeGroup {
        @Getter
        private final String id;

        @Getter
        private final String name;

        @Getter
        private final List<SortType> sortTypes;
    }

    @Data
    @EqualsAndHashCode
    @AllArgsConstructor
    private static class SortRegistryKey {
        boolean needHotelsNearby;
        SortTypeLayout layout;
        boolean useNewRanking;
        boolean usePersonalRanking;
        String exporationRankingCategory;
        RealTimeRankingType realTimeRankingType;
    }

    public SortType getDefaultSortType() {
        return defaultSortType;
    }

    public List<SortTypeGroup> getSortTypeGroups() {
        return sortTypeGroups;
    }

    public SortType getSortType(String selectedSortId) {
        Preconditions.checkState(selectedSortId != null);
        for (var sortType : sortTypes) {
            if (selectedSortId.equals(sortType.getId())) {
                return sortType;
            }
        }
        throw new TravelApiBadRequestException("Unknown sort id '" + selectedSortId + "'");
    }

    public SortType getSortTypeByGeoCounterType(ESortType sortTypeProto) {
        Preconditions.checkState(sortTypeProto != ESortType.ST_Unknown);
        for (var sortType : sortTypes) {
            if (sortType.supportsGeoCounterSortType(sortTypeProto)) {
                return sortType;
            }
        }
        throw new IllegalArgumentException("Unknown geocounter sort type '" + sortTypeProto + "'");
    }

    private SortTypeRegistry(SortType defaultSortType, List<SortType> sortTypes, List<SortTypeGroup> sortTypeGroups) {
        this.defaultSortType = defaultSortType;
        this.sortTypes = sortTypes;
        this.sortTypeGroups = sortTypeGroups;
    }

    public static SortTypeRegistry getSortTypeRegistry(boolean needHotelsNearby,
                                                       SortTypeLayout layout,
                                                       boolean useNewRanking,
                                                       boolean usePersonalRanking,
                                                       String exporationRankingCategory,
                                                       RealTimeRankingType realTimeRankingType) {
        return Registries.get(new SortRegistryKey(needHotelsNearby, layout, useNewRanking, usePersonalRanking, exporationRankingCategory, realTimeRankingType));
    }

    private static SortTypeRegistry BuildSortTypeRegistry(boolean needHotelsNearby,
                                                          SortTypeLayout layout,
                                                          boolean useNewRanking,
                                                          boolean usePersonalRanking,
                                                          String exporationRankingCategory,
                                                          RealTimeRankingType realTimeRankingType) {
        var relevaceSortType = ESortType.ST_ByRank;
        if (realTimeRankingType == RealTimeRankingType.CATBOOST) {
            relevaceSortType = ESortType.ST_ByRankWithCatBoost;
        } else if (realTimeRankingType == RealTimeRankingType.FAKE_CATBOOST) {
            relevaceSortType = ESortType.ST_ByRankWithCatBoostFake;
        } else if (realTimeRankingType == RealTimeRankingType.NO_BOY_BOOST) {
            relevaceSortType = ESortType.ST_ByRankExperimental;
        }
        if (useNewRanking) {
            relevaceSortType = ESortType.ST_ByRankExperimental;
        }
        if (usePersonalRanking) {
            relevaceSortType = ESortType.ST_ByRankWithPersonalization;
        }
        if (exporationRankingCategory != null) {
            var mapping = Map.<String, ESortType>of(
                    "base", ESortType.ST_ByRank,
                    "1", ESortType.ST_ByRankExploration1,
                    "2", ESortType.ST_ByRankExploration2,
                    "3", ESortType.ST_ByRankExploration3,
                    "4", ESortType.ST_ByRankExploration4,
                    "5", ESortType.ST_ByRankExploration5
            );
            Preconditions.checkState(mapping.containsKey(exporationRankingCategory), "Unknown ranking category " + exporationRankingCategory);
            relevaceSortType = mapping.get(exporationRankingCategory);
        }
        SortTypeNames sortTypeNames = sortTypeNamesByLayout.get(layout);

        var relevantFirst = new SortType(
                "relevant-first",
                sortTypeNames.getRelevantFirst(),
                SearchHotelsRspV1.SortTypeHint.NONE,
                false,
                false,
                null,
                relevaceSortType,
                Set.of(ESortType.ST_ByRankWithPersonalization, ESortType.ST_ByRankExperimental, ESortType.ST_ByRank,
                        ESortType.ST_ByRankExploration1, ESortType.ST_ByRankExploration2,
                        ESortType.ST_ByRankExploration3, ESortType.ST_ByRankExploration4,
                        ESortType.ST_ByRankExploration5, ESortType.ST_ByRankWithCatBoost)
        );
        var cheapFirst = new SortType(
                "cheap-first",
                sortTypeNames.getCheapFirst(),
                SearchHotelsRspV1.SortTypeHint.NONE,
                true,
                false,
                GeoSearchReq.SortType.CHEAP_FIRST,
                ESortType.ST_ByPriceAsc,
                Collections.emptySet()
        );
        var expensiveFirst = new SortType(
                "expensive-first",
                sortTypeNames.getExpensiveFirst(),
                SearchHotelsRspV1.SortTypeHint.NONE,
                true,
                false,
                GeoSearchReq.SortType.EXPENSIVE_FIRST,
                ESortType.ST_ByPriceDesc,
                Collections.emptySet()
        );
        var highRatingFirst = new SortType(
                "high-rating-first",
                sortTypeNames.getHighRatingFirst(),
                SearchHotelsRspV1.SortTypeHint.NONE,
                false,
                false,
                GeoSearchReq.SortType.HIGH_RATING_FIRST,
                ESortType.ST_ByRating,
                Collections.emptySet()
        );
        var nearFirst = new SortType(
                SORT_BY_DISTANCE_ID,
                sortTypeNames.getNearFirst(),
                SearchHotelsRspV1.SortTypeHint.NONE,
                false,
                true,
                null,
                ESortType.ST_ByDistance,
                Collections.emptySet()
        );

        var sortTypes = new ArrayList<>(List.of(relevantFirst, cheapFirst, expensiveFirst, highRatingFirst));
        var sortTypesGroups = new ArrayList<>(List.of(
                new SortTypeGroup("relevant-first-group", sortTypeNames.getRelevantFirst(), List.of(relevantFirst)),
                new SortTypeGroup("cheap-first-group", sortTypeNames.getCheapFirst(), List.of(cheapFirst)),
                new SortTypeGroup("expensive-first-group", sortTypeNames.getExpensiveFirst(), List.of(expensiveFirst)),
                new SortTypeGroup("high-rating-first-group", sortTypeNames.getHighRatingFirst(), List.of(highRatingFirst))
        ));

        if (needHotelsNearby) {
            sortTypes.add(nearFirst);
            sortTypesGroups.add(new SortTypeGroup("near-first-group", sortTypeNames.getNearFirst(), List.of(nearFirst)));
        }

        return new SortTypeRegistry(relevantFirst, sortTypes, sortTypesGroups);
    }

    private static void doTwice(Consumer<Boolean> c) {
        c.accept(true);
        c.accept(false);
    }

    public static final Map<SortRegistryKey, SortTypeRegistry> Registries = new HashMap<>() {{
        doTwice(useNewRanking ->
            doTwice(needHotelsNearby ->
                doTwice(usePersonalRanking ->
                    Arrays.stream(RealTimeRankingType.values()).forEach(realTimeRankingType ->
                        Arrays.stream(SortTypeLayout.values()).forEach(layout ->
                                Stream.of(null, "base", "1", "2", "3", "4", "5").forEach(exporationRankingCategory ->
                                        put(new SortRegistryKey(needHotelsNearby, layout, useNewRanking, usePersonalRanking, exporationRankingCategory, realTimeRankingType),
                                                BuildSortTypeRegistry(needHotelsNearby, layout, useNewRanking, usePersonalRanking, exporationRankingCategory, realTimeRankingType))
                                )
                        )
                    )
                )
            )
        );
    }};

    private final SortType defaultSortType;
    private final List<SortType> sortTypes;
    private final List<SortTypeGroup> sortTypeGroups;
}
