package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import lombok.AllArgsConstructor;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestRspV1;
import ru.yandex.travel.api.models.hotels.OfferSearchParams;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.avia.personalization.personal_search.v2.EntryType;
import ru.yandex.travel.avia.personalization.personal_search.v2.THotelEntry;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.proto.suggest.Suggest;

@Component
@AllArgsConstructor
public class SuggestItemMapper {

    private final LocalizationService localizationService;
    private final RegionsService regionsService;
    private final GeoBase geoBase;
    private final HotelSlugService hotelSlugService;
    public final static Map<EntryType, String> PERSONALIZED_SUGGEST_TYPE_TO_NAME = new HashMap<>() {{
        put(EntryType.ENTRY_TYPE_ORDER, "order");
        put(EntryType.ENTRY_TYPE_SEARCH, "search");
    }};

    public SuggestRspV1.SuggestItem toSuggestItem(SuggestReqV1 request, Suggest.THotelSuggest hotelSuggest) {
        final Permalink permalink = Permalink.of(hotelSuggest.getPermalink());

        final SuggestRspV1.SuggestItem item = new SuggestRspV1.SuggestItem(
                hotelSuggest.getId(),
                hotelSuggest.getName()
        );
        item.setDescription(hotelSuggest.getDescription());
        item.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.HOTEL));
        item.getRedirectParams().setPermalink(permalink);
        item.getRedirectParams().setHotelSlug(hotelSlugService.findMainSlugByPermalink(permalink));

        return item;
    }

    public SuggestRspV1.SuggestItem toSuggestItem(SuggestReqV1 request, Suggest.TRegionSuggest regionSuggest,
                                                  UaasSearchExperiments experiments, boolean isTouch) {
        final int geoId = regionSuggest.getGeoId();
        String regionSuggestName;
        if (experiments.isMoskowAreaEnabled() && !isTouch) {
            switch (geoId) {
                case GeoBaseHelpers.MOSCOW_DISTRICT:
                    regionSuggestName = "Московская область";
                    break;
                case GeoBaseHelpers.LENINGRAD_DISTRICT:
                    regionSuggestName = "Ленинградская область";
                    break;
                default: regionSuggestName = regionSuggest.getName();
            }
        } else {
            regionSuggestName = regionSuggest.getName();
        }

        final SuggestRspV1.SuggestItem item = new SuggestRspV1.SuggestItem(regionSuggest.getId(),
                regionSuggestName);
        item.setDescription(GeoBaseHelpers.getRegionDescription(
                geoBase,
                request.getDomain(),
                geoId,
                true,
                false,
                GeoBaseHelpers.DEFAULT_ALLOWED_REGION_TYPES,
                new HashSet<Integer>(),
                false));
        item.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.REGION));
        item.getRedirectParams().setGeoId(geoId);

        return item;
    }

    public SuggestRspV1.SuggestItem toSuggestItem(SuggestReqV1 request, THotelEntry entry) {
        final int geoId = (int) entry.getPointTo().getGeoId();
        if (geoId == 0) {
            return null;
        }

        final String name = regionsService.getRegionName(geoId, request.getDomain());
        var entryType = PERSONALIZED_SUGGEST_TYPE_TO_NAME.getOrDefault(entry.getType(), "unknown");
        var suggestId = String.format("personalized-%s-%d-%s", entryType, geoId,
                request.getLanguage().getShortLanguage());

        final SuggestRspV1.SuggestItem item = new SuggestRspV1.SuggestItem(suggestId, name);
        item.setDescription(GeoBaseHelpers.getRegionDescription(
                geoBase,
                request.getDomain(),
                geoId,
                true,
                false,
                GeoBaseHelpers.DEFAULT_ALLOWED_REGION_TYPES,
                new HashSet<Integer>(),
                false));
        item.setRedirectParams(new SuggestRspV1.RedirectParams(getPersonalizedSuggestType(entry.getType())));
        item.getRedirectParams().setGeoId(geoId);
        item.getRedirectParams().setOfferSearchParams(new OfferSearchParams());
        item.getRedirectParams().getOfferSearchParams().setAdults((int) entry.getTravelers().getAdults());
        item.getRedirectParams().getOfferSearchParams().setChildrenAges(entry.getTravelers().getChildrenAgesList());
        item.getRedirectParams().getOfferSearchParams().setCheckinDate(LocalDate.parse(entry.getCheckInDate()));
        item.getRedirectParams().getOfferSearchParams().setCheckoutDate(LocalDate.parse(entry.getCheckOutDate()));

        return item;
    }

    private SuggestRspV1.SuggestType getPersonalizedSuggestType(EntryType type) {
        if (type == EntryType.ENTRY_TYPE_ORDER) {
            return SuggestRspV1.SuggestType.CROSS_SALE;
        }
        return SuggestRspV1.SuggestType.HISTORY;
    }
}
