package ru.yandex.travel.api.endpoints.hotels_portal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.geobase6.LookupException;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.LogSuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.LogSuggestRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestRspV1;
import ru.yandex.travel.api.infrastucture.ProtobufJsonLogger;
import ru.yandex.travel.api.proto.hotels_portal.TCommonHttpHeaders;
import ru.yandex.travel.api.proto.hotels_portal.THotelsSuggestChoiceLogRecord;
import ru.yandex.travel.api.proto.hotels_portal.THotelsSuggestLogRecord;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;
import ru.yandex.travel.api.services.hotels.suggest.HotelSuggestService;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.api.services.personalization.PersonalizationClientFactory;
import ru.yandex.travel.api.services.portal_recipes.PortalRecipesService;
import ru.yandex.travel.avia.personalization.personal_search.v2.EntryType;
import ru.yandex.travel.avia.personalization.personal_search.v2.TGetHotelsSuggestRequestV2;
import ru.yandex.travel.avia.personalization.personal_search.v2.TGetPersonalSearchResponseV2;
import ru.yandex.travel.avia.personalization.personal_search.v2.THotelEntry;
import ru.yandex.travel.avia.personalization.personal_search.v2.TPersonalSearchEntryV2;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.common.LanguageType;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;

import static org.apache.logging.log4j.util.Strings.isBlank;

@Component
@RequiredArgsConstructor
@Slf4j
public class SuggestService {
    public enum SuggestType {
        REGION("Region"),
        HOTEL("Hotel"),
        @Deprecated
        SEARCH("Search"),
        HOTELS_NEARBY("HotelsNearby"),
        HISTORY("SearchHistory"),
        CROSSSALE("CrossSale"),
        ;

        final String typeName;

        SuggestType(String typeName) {
            this.typeName = typeName;
        }

        public String getId() {
            return this.name().toLowerCase();
        }

        public int getIndex() {
            return this.ordinal();
        }

        public static SuggestType parse(String value) {
            return SuggestType.valueOf(value.toUpperCase());
        }
    }

    @Data
    @AllArgsConstructor
    private static class SuggestGroups {
        final List<SuggestRspV1.SuggestItem> regionGroup;
        final List<SuggestRspV1.SuggestItem> hotelGroup;
        final List<SuggestRspV1.SuggestItem> searchGroup;
    }

    private final ProtobufJsonLogger suggestLogger = new ProtobufJsonLogger("ru.yandex.travel.api.endpoints" +
            ".hotels_portal.SuggestLogger");
    private final ProtobufJsonLogger suggestChoiceLogger = new ProtobufJsonLogger("ru.yandex.travel.api.endpoints" +
            ".hotels_portal.SuggestChoiceLogger");

    private final RegionsService regionsService;
    private final GeoSearchService geoSearchService;
    private final GeoBase geoBase;
    private final LocalizationService localizationService;
    private final HotelSuggestService hotelSuggestService;
    private final SuggestItemMapper mapper;
    private final HotelSlugService hotelSlugService;
    private final RegionSlugService regionSlugService;
    private final PortalRecipesService portalRecipesService;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final PersonalizationClientFactory personalizationClientFactory;
    private final HotelsPortalProperties hotelsPortalProperties;

    public CompletableFuture<SuggestRspV1> processSuggestRequest(SuggestReqV1 req, CommonHttpHeaders headers) {
        return prepareSuggestResponse(req, headers).thenApply(rsp -> {
            logSuggestReqRsp(req, headers, rsp);
            return rsp;
        });
    }

    public CompletableFuture<SuggestRspV1> prepareSuggestResponse(SuggestReqV1 req, CommonHttpHeaders headers) {
        UaasSearchExperiments uaasSearchExperiments =
                uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        var exps = new Experiments(null, hotelsPortalProperties);
        boolean hotelsNearbyEnabled = HotelsPortalUtils.isHotelsNearbyEnabled(headers);
        String hotelsNearbyQuery = localizationService.getLocalizedValue("HotelsNearby",
                req.getLanguage().getShortLanguage()).toLowerCase(Locale.ROOT);
        boolean userNeedsHotelsNearby =
                isBlank(req.getQuery()) || hotelsNearbyQuery.startsWith(req.getQuery().toLowerCase(Locale.ROOT));
        // Force show hotels nearby when coordinates are present to avoid cases when this option suddenly disappears
        // after geolocation request
        var showHotelsNearby = userNeedsHotelsNearby && (hotelsNearbyEnabled || req.getUserCoordinates() != null);
        int limit = req.getLimit() - (showHotelsNearby ? 1 : 0);

        CompletableFuture<List<SuggestRspV1.SuggestGroup>> groups;
        if (!isBlank(req.getQuery())) {
            groups = getHotelsSuggest(req, limit, uaasSearchExperiments, HotelsPortalUtils.isUserDeviceTouch(headers))
                    .thenApply(items -> convertAndStripToLimit(items, req, limit));
        } else if (req.isRegionIdentifierPresent()) {
            groups = CompletableFuture.completedFuture(getSuggestListByRegionIdentifier(req, uaasSearchExperiments, HotelsPortalUtils.isUserDeviceTouch(headers)));
        } else if (req.isHotelIdentifierPresent()) {
            groups = getSuggestListByHotelIdentifier(req);
        } else {
            groups = CompletableFuture.completedFuture(getRecipesSuggestList(req, limit));
        }
        if (showHotelsNearby) {
            groups = groups.thenApply(items -> Stream
                    .concat(Stream.of(getHotelsNearbyGroup(req)), items.stream())
                    .collect(Collectors.toUnmodifiableList()));
        }
        var userNeedsPersonalizedSuggests = (
                isBlank(req.getQuery()) && isBlank(req.getHotelSlug()) &&
                        !req.isRegionIdentifierPresent()
        );
        if (userNeedsPersonalizedSuggests) {
            var personalizedSuggestFuture = getHotelsPersonalizedSuggests(req, headers);
            groups = groups.thenCompose(items -> addPersonalizedSuggest(personalizedSuggestFuture, items));
        }
        return groups.thenApply(SuggestRspV1::new);
    }

    private CompletableFuture<List<SuggestRspV1.SuggestGroup>> addPersonalizedSuggest(CompletableFuture<List<SuggestRspV1.SuggestGroup>> personalizedSuggestFuture, List<SuggestRspV1.SuggestGroup> items) {
        return personalizedSuggestFuture.thenApply(g ->
                Stream.concat(g.stream(), items.stream()).collect(Collectors.toUnmodifiableList())
        );
    }

    private CompletableFuture<List<SuggestRspV1.SuggestGroup>> getHotelsPersonalizedSuggests(SuggestReqV1 req, CommonHttpHeaders headers) {
        try {
            var requestProtoBuilder = TGetHotelsSuggestRequestV2.newBuilder()
                    .setYandexUid(headers.getYandexUid())
                    .setSearchesLimit(3)
                    .setOrdersLimit(1);
            if (headers.getPassportId() != null) {
                requestProtoBuilder.setPassportId(headers.getPassportId());
            }
            var requestProto = requestProtoBuilder.build();
            var client = personalizationClientFactory.createRoundRobinStub();
            return FutureUtils.buildCompletableFuture(client.getHotelsSuggest(requestProto)).thenApply(rsp -> mapPersonalizedHotelsSuggest(req, rsp));
        } catch (Exception e) {
            log.error("Error getting personalized suggests: ", e);
            return CompletableFuture.completedFuture(new ArrayList<>());
        }
    }

    private List<SuggestRspV1.SuggestGroup> mapPersonalizedHotelsSuggest(SuggestReqV1 req,
                                                                         TGetPersonalSearchResponseV2 rsp) {
        return rsp.getEntriesList().stream()
                .map(TPersonalSearchEntryV2::getHotel)
                .collect(Collectors.groupingBy(THotelEntry::getType)).values().stream()
                .sorted(Comparator.comparingInt(e -> -e.get(0).getType().getNumber()))
                .map(g -> new SuggestRspV1.SuggestGroup(
                        localizationService.getLocalizedValue(mapPersonalSuggestType(g.get(0).getType()),
                                req.getLanguage().getShortLanguage()),
                        g.stream()
                                .map(e -> mapper.toSuggestItem(req, e))
                                .filter(Objects::nonNull)
                                .collect(Collectors.toUnmodifiableList())
                ))
                .filter(g -> g.getItems().size() > 0)
                .collect(Collectors.toUnmodifiableList());
    }

    private String mapPersonalSuggestType(EntryType type) {
        return type == EntryType.ENTRY_TYPE_ORDER ? SuggestType.CROSSSALE.typeName : SuggestType.HISTORY.typeName;
    }

    private SuggestRspV1.SuggestGroup getHotelsNearbyGroup(SuggestReqV1 req) {
        var suggestItem = new SuggestRspV1.SuggestItem(
                String.format("hotels-nearby-%s", req.getLanguage().getShortLanguage()),
                localizationService.getLocalizedValue("HotelsNearby", req.getLanguage().getShortLanguage()));
        suggestItem.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.HOTELS_NEARBY));
        suggestItem.getRedirectParams().setSelectedSortId(SortTypeRegistry.SORT_BY_DISTANCE_ID);
        suggestItem.getRedirectParams().setSortOrigin(req.getUserCoordinates() != null ?
                req.getUserCoordinates().toString() : null);
        return new SuggestRspV1.SuggestGroup(
                localizationService.getLocalizedValue(SuggestType.HOTELS_NEARBY.typeName,
                        req.getLanguage().getShortLanguage()),
                List.of(suggestItem)
        );
    }

    private List<SuggestRspV1.SuggestGroup> convertAndStripToLimit(SuggestGroups suggestGroups, SuggestReqV1 req,
                                                                   int limit) {
        final int groupsCount = 3;
        int suggestIndex = 0;

        int takeRegions = 0;
        int takeHotels = 0;
        int takeSearch = 0;

        while (0 < limit) {
            int currentIndex = suggestIndex++ % groupsCount;
            if (currentIndex == 0) {
                if (takeRegions < suggestGroups.regionGroup.size()) {
                    takeRegions++;
                    limit--;
                }
            } else if (currentIndex == 1) {
                if (takeHotels < suggestGroups.hotelGroup.size()) {
                    takeHotels++;
                    limit--;
                }
            } else if (currentIndex == 2) {
                if (takeSearch < suggestGroups.searchGroup.size()) {
                    takeSearch++;
                    limit--;
                }
            }
            if (takeRegions == suggestGroups.regionGroup.size() && takeHotels == suggestGroups.hotelGroup.size()
                    && takeSearch == suggestGroups.searchGroup.size()) {
                break;
            }
        }
        final List<SuggestRspV1.SuggestGroup> result = new ArrayList<>(groupsCount);
        if (takeRegions != 0) {
            result.add(new SuggestRspV1.SuggestGroup(
                    localizationService.getLocalizedValue(SuggestType.REGION.typeName,
                            req.getLanguage().getShortLanguage()),
                    suggestGroups.regionGroup.subList(0, takeRegions)
            ));
        }
        if (takeHotels != 0) {
            result.add(new SuggestRspV1.SuggestGroup(
                    localizationService.getLocalizedValue(SuggestType.HOTEL.typeName,
                            req.getLanguage().getShortLanguage()),
                    suggestGroups.hotelGroup.subList(0, takeHotels)
            ));
        }
        if (takeSearch != 0) {
            result.add(new SuggestRspV1.SuggestGroup(
                    localizationService.getLocalizedValue(SuggestType.SEARCH.typeName,
                            req.getLanguage().getShortLanguage()),
                    suggestGroups.searchGroup.subList(0, takeSearch)
            ));
        }
        return result;
    }

    public CompletableFuture<LogSuggestRspV1> processLogSuggestRequest(LogSuggestReqV1 req, CommonHttpHeaders headers) {
        logUserSuggestChoice(req, headers);
        return CompletableFuture.completedFuture(new LogSuggestRspV1());
    }

    private CompletableFuture<SuggestGroups> getHotelsSuggest(SuggestReqV1 request, int limit,
                                                              UaasSearchExperiments experiments, boolean isTouch) {
        final String query = request.getQuery().trim();

        return hotelSuggestService
                .getSuggest("", query, limit)
                .thenApply(suggest -> {
                    var regionSuggestItems =
                            suggest
                                    .getRegionSuggests()
                                    .stream()
                                    .map(regionSuggest ->
                                            mapper.toSuggestItem(request, regionSuggest, experiments, isTouch))
                                    .collect(Collectors.toList());
                    if (hotelsPortalProperties.isRenameDoubleSuggestDescription()){
                        regionSuggestItems = GeoBaseHelpers.findAndRenameDoubleSuggestDescription(regionSuggestItems, geoBase,
                                request.getDomain());
                    }

                    return new SuggestGroups(
                            regionSuggestItems,
                            suggest.getHotelSuggests().stream().map(hotelSuggest -> mapper.toSuggestItem(request,
                                    hotelSuggest)).collect(Collectors.toList()),
                        /* Temporary disabled for TRAVELBACK-592
                        appendUserRequestIfMissing(
                                suggest.getSearchSuggests().stream().map(searchSuggest -> mapper.toSuggestItem
                                (request, searchSuggest)),
                                mapper.makeItemFromRequest(request)
                        )*/
                            List.of());
                });
    }

    private static Stream<SuggestRspV1.SuggestItem> appendUserRequestIfMissing(Stream<SuggestRspV1.SuggestItem> suggests, SuggestRspV1.SuggestItem userRequest) {
        final List<SuggestRspV1.SuggestItem> search = suggests.collect(Collectors.toList());
        if (search.stream().noneMatch(item -> item.getName().toLowerCase().equals(userRequest.getName().toLowerCase()))) {
            return Stream.concat(search.stream(), Stream.of(userRequest));
        } else {
            return search.stream();
        }
    }

    private List<SuggestRspV1.SuggestGroup> getRecipesSuggestList(SuggestReqV1 req, int limit) {
        return Collections.singletonList(new SuggestRspV1.SuggestGroup(
                localizationService.getLocalizedValue(SuggestType.REGION.typeName,
                        req.getLanguage().getShortLanguage()),
                portalRecipesService.get(PortalRecipesService.HOTELS_TYPE)
                        .stream()
                        .limit(limit)
                        .map(recipe -> {
                            final String name = regionsService.getRegionName(recipe.getToGeoId(), req.getDomain());
                            final SuggestRspV1.SuggestItem resp = new SuggestRspV1.SuggestItem(
                                    String.format("region-%d-%s", recipe.getToGeoId(),
                                            req.getLanguage().getShortLanguage()),
                                    name
                            );
                            resp.setDescription(GeoBaseHelpers.getRegionDescription(
                                    geoBase,
                                    req.getDomain(),
                                    recipe.getToGeoId(),
                                    false,
                                    false,
                                    GeoBaseHelpers.DEFAULT_ALLOWED_REGION_TYPES,
                                    new HashSet<Integer>(),
                                    false));
                            resp.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.REGION));
                            resp.getRedirectParams().setGeoId(recipe.getToGeoId());
                            return resp;
                        }).collect(Collectors.toList())
        ));
    }

    private SuggestRspV1.SuggestItem makeSuggestRspForRegion(String domain, int geoId, LanguageType lang,
                                                             UaasSearchExperiments experiments, boolean isTouch) throws IllegalArgumentException {
        String suggestItemName;

        if (experiments.isMoskowAreaEnabled() && !isTouch) {
            switch (geoId) {
                case GeoBaseHelpers.MOSCOW_DISTRICT:
                    suggestItemName = "Московская область";
                    break;
                case GeoBaseHelpers.LENINGRAD_DISTRICT:
                    suggestItemName = "Ленинградская область";
                    break;
                default: suggestItemName = GeoBaseHelpers.getRegionName(geoBase, domain, geoId);
            }
        } else {
            suggestItemName = GeoBaseHelpers.getRegionName(geoBase, domain, geoId);
        }

        final SuggestRspV1.SuggestItem r = new SuggestRspV1.SuggestItem(
                String.format("region-%d-%s", geoId, lang.getShortLanguage()),
                suggestItemName
        );
        r.setDescription(GeoBaseHelpers.getRegionDescription(
                geoBase,
                domain,
                geoId,
                true,
                false,
                GeoBaseHelpers.DEFAULT_ALLOWED_REGION_TYPES,
                new HashSet<>(),
                false));
        r.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.REGION));
        r.getRedirectParams().setGeoId(geoId);
        return r;
    }

    private SuggestRspV1.SuggestItem makeSuggestRspForInvalidRegion(int geoId, LanguageType lang) {
        final String unknownRegion = "Неизвестный регион"; // TODO: Remove this hack or make it locale-dependent

        final SuggestRspV1.SuggestItem r = new SuggestRspV1.SuggestItem("region-invalid", unknownRegion);
        r.setDescription(unknownRegion);
        r.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.REGION));
        r.getRedirectParams().setGeoId(0);

        return r;
    }

    private SuggestRspV1.SuggestItem makeSuggestRspForHotel(Permalink permalink, String name, String descr,
                                                            LanguageType lang) {
        final SuggestRspV1.SuggestItem r = new SuggestRspV1.SuggestItem(
                String.format("hotel-%d-%s", permalink.asLong(), lang.getShortLanguage()),
                name
        );
        r.setDescription(descr);
        r.setRedirectParams(new SuggestRspV1.RedirectParams(SuggestRspV1.SuggestType.HOTEL));
        r.getRedirectParams().setPermalink(permalink);
        r.getRedirectParams().setHotelSlug(hotelSlugService.findMainSlugByPermalink(permalink));

        return r;
    }

    private List<SuggestRspV1.SuggestGroup> getSuggestListByRegionIdentifier(SuggestReqV1 req, UaasSearchExperiments experiments,
                                                                             boolean isTouch) {
        int geoId;
        if (Objects.nonNull(req.getRegionSlug())) {
            geoId = regionSlugService.getGeoIdBySlug(req.getRegionSlug());
        } else {
            geoId = req.getGeoId();
        }
        final String domain = req.getDomain();
        final LanguageType language = req.getLanguage();

        boolean validRegion = true;
        try {
            var regionHash = geoBase.getRegionById(geoId, domain);
            var regionType = regionHash.getAttr("type").getInteger();
            if (regionType == GeoBaseHelpers.HIDDEN_REGION_TYPE || regionType == GeoBaseHelpers.OTHER_REGION_TYPE) {
                log.info("Banned region (geoId={}, domain={})", geoId, domain);
                validRegion = false;
            }
        } catch (LookupException e) {
            log.info("Unknown geoId " + geoId);
            validRegion = false;
        }

        return Collections.singletonList(new SuggestRspV1.SuggestGroup(
                localizationService.getLocalizedValue(SuggestType.REGION.typeName, language.getShortLanguage()),
                List.of(validRegion
                        ? makeSuggestRspForRegion(domain, geoId, language, experiments, isTouch)
                        : makeSuggestRspForInvalidRegion(geoId, language)
                )));
    }

    private CompletableFuture<List<SuggestRspV1.SuggestGroup>> getSuggestListByHotelIdentifier(SuggestReqV1 req) {
        return HotelsPortalUtils.handleExceptions("getSuggestListByHotelIdentifier", () -> {
            Permalink permalink = hotelSlugService.findPermalinkByHotelIdentifier(req);
            GeoSearchReq geoSearchReq = GeoSearchReq.byPermalink(GeoOriginEnum.SUGGEST, permalink, null);
            geoSearchReq.setLimit(1);
            geoSearchReq.setLanguage(req.getLanguage());
            return geoSearchService.querySingleHotel(geoSearchReq)
                    .thenApply(geoSearchRsp -> {
                        GeoHotel hotel = geoSearchRsp.getHotel();
                        if (hotel.getGeoObjectMetadata() == null) {
                            log.error("GeoSearch returned hotel with no geoObjectMetadata for permalink {}",
                                    permalink);
                            throw new IllegalStateException("Permalink has no geoObjectMetadata: " + permalink);
                        }
                        return Collections.singletonList(new SuggestRspV1.SuggestGroup(
                                localizationService.getLocalizedValue(SuggestType.HOTEL.typeName,
                                        req.getLanguage().getShortLanguage()),
                                List.of(makeSuggestRspForHotel(
                                        permalink,
                                        hotel.getGeoObjectMetadata().getName(),
                                        hotel.getGeoObjectMetadata().getAddress().getFormattedAddress(),
                                        req.getLanguage()
                                ))
                        ));
                    });
        });
    }

    private TCommonHttpHeaders convertCommonHeadersToLog(CommonHttpHeaders headers) {
        TCommonHttpHeaders.Builder builder = TCommonHttpHeaders.newBuilder();
        if (headers.getUserGid() != null) {
            builder.setUserGid(headers.getUserGid());
        }
        if (headers.getUserDevice() != null) {
            builder.setUserDevice(headers.getUserDevice());
        }
        if (headers.getUserLogin() != null) {
            builder.setUserLogin(headers.getUserLogin());
        }
        if (headers.getUserIP() != null) {
            builder.setUserIP(headers.getUserIP());
        }
        builder.setUserIsStaff(headers.getUserIsStaffAsBool());
        builder.setUserIsPlus(headers.getUserIsPlusAsBool());
        if (headers.getRequestId() != null) {
            builder.setRequestId(headers.getRequestId());
        }
        if (headers.getYandexUid() != null) {
            builder.setYandexUid(headers.getYandexUid());
        }
        if (headers.getRealUserAgent() != null) {
            builder.setUserAgent(headers.getRealUserAgent());
        }
        return builder.build();
    }

    private void logSuggestReqRsp(SuggestReqV1 req, CommonHttpHeaders headers, SuggestRspV1 rsp) {
        THotelsSuggestLogRecord.TRequest.Builder reqBuilder = THotelsSuggestLogRecord.TRequest.newBuilder();
        if (req.getQuery() != null) {
            reqBuilder.setQuery(req.getQuery());
        }
        if (req.getGeoId() != null) {
            reqBuilder.setGeoId(req.getGeoId());
        }
        if (req.getRegionSlug() != null) {
            reqBuilder.setRegionSlug(req.getRegionSlug());
        }
        if (req.getPermalink() != null) {
            reqBuilder.setPermalink(req.getPermalink().asLong());
        }
        if (req.getHotelSlug() != null) {
            reqBuilder.setHotelSlug(req.getHotelSlug());
        }
        reqBuilder.setLimit(req.getLimit());
        reqBuilder.setLanguage(req.getLanguage().getShortLanguage());
        reqBuilder.setDomain(req.getDomain());
        if (req.getSessionId() != null) {
            reqBuilder.setSessionId(req.getSessionId());
        }
        if (req.getRequestIndex() != null) {
            reqBuilder.setRequestIndex(req.getRequestIndex());
        }
        if (req.getPathname() != null) {
            reqBuilder.setPathname(req.getPathname());
        }
        if (req.getGeoLocationStatus() != null) {
            reqBuilder.setGeoLocationStatus(req.getGeoLocationStatus().toString());
        }
        if (req.getUserCoordinates() != null) {
            reqBuilder.setUserCoordinateLon(req.getUserCoordinates().getLon());
            reqBuilder.setUserCoordinateLat(req.getUserCoordinates().getLat());
        }

        THotelsSuggestLogRecord.TResponse.Builder rspBuilder = THotelsSuggestLogRecord.TResponse.newBuilder();
        for (SuggestRspV1.SuggestGroup group : rsp.getGroups()) {
            THotelsSuggestLogRecord.TResponse.TGroup.Builder grpBuilder =
                    THotelsSuggestLogRecord.TResponse.TGroup.newBuilder();
            grpBuilder.setName(group.getName());
            for (SuggestRspV1.SuggestItem item : group.getItems()) {
                THotelsSuggestLogRecord.TResponse.TGroup.TItem.Builder itemBuilder =
                        THotelsSuggestLogRecord.TResponse.TGroup.TItem.newBuilder();
                itemBuilder.setId(item.getId());
                itemBuilder.setName(item.getName());
                if (item.getDescription() != null) {
                    itemBuilder.setDescription(item.getDescription());
                }
                itemBuilder.setType(item.getRedirectParams().getType().name());
                if (item.getRedirectParams().getGeoId() != null) {
                    itemBuilder.setGeoId(item.getRedirectParams().getGeoId());
                }
                if (item.getRedirectParams().getPermalink() != null) {
                    itemBuilder.setPermalink(item.getRedirectParams().getPermalink().asLong());
                }
                if (item.getRedirectParams().getHotelSlug() != null) {
                    itemBuilder.setHotelSlug(item.getRedirectParams().getHotelSlug());
                }
                grpBuilder.addItems(itemBuilder);
            }
            rspBuilder.addGroups(grpBuilder);
        }

        THotelsSuggestLogRecord.Builder builder = THotelsSuggestLogRecord.newBuilder();
        builder.setReq(reqBuilder);
        builder.setReqHeaders(convertCommonHeadersToLog(headers));
        builder.setResp(rspBuilder);
        builder.setRequestTimestamp(System.currentTimeMillis());
        suggestLogger.log(builder.build());
    }

    private void logUserSuggestChoice(LogSuggestReqV1 req, CommonHttpHeaders headers) {
        THotelsSuggestChoiceLogRecord.TRequest.Builder reqBuilder = THotelsSuggestChoiceLogRecord.TRequest.newBuilder();
        if (req.getSelectedId() != null) {
            reqBuilder.setSelectedId(req.getSelectedId());
        }
        if (req.getSessionId() != null) {
            reqBuilder.setSessionId(req.getSessionId());
        }
        if (req.getRequestIndex() != null) {
            reqBuilder.setRequestIndex(req.getRequestIndex());
        }
        if (req.getIsManualClick() != null) {
            reqBuilder.setIsManualClick(req.getIsManualClick());
        }
        if (req.getIsTrustedUser() != null) {
            reqBuilder.setIsTrustedUser(req.getIsTrustedUser());
        }
        THotelsSuggestChoiceLogRecord.Builder builder = THotelsSuggestChoiceLogRecord.newBuilder();
        builder.setReq(reqBuilder);
        builder.setReqHeaders(convertCommonHeadersToLog(headers));
        builder.setRequestTimestamp(System.currentTimeMillis());
        suggestChoiceLogger.log(builder.build());
    }
}
