package ru.yandex.travel.api.endpoints.hotels_portal;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.CountHotelsRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.HotelOffer;
import ru.yandex.travel.api.models.hotels.HotelWithOffers;
import ru.yandex.travel.api.models.hotels.OfferSearchParams;
import ru.yandex.travel.api.models.hotels.OfferSearchParamsWithBbox;
import ru.yandex.travel.api.models.hotels.OfferSearchProgress;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.SearchFilterAndTextParams;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.geocounter.GeoCounterService;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterGetHotelsReq;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterGetHotelsRsp;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.promo.CachedActivePromosService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels.tugc.TugcService;
import ru.yandex.travel.api.services.hotels.tugc.model.HotelFavoriteInfosRsp;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.common.promo.mir.MirUtils;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.ERefundType;
import ru.yandex.travel.hotels.proto.geocounter_service.EUserSegmentType;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetHotelsResponse;

@Slf4j
public class ViaGeocounterSearcher {
    private final HotelsPortalProperties hotelsPortalProperties;
    private final GeoCounterService geoCounter;
    private final GeoSearchService geoSearch;
    private final GeoBase geoBase;
    private final HotelsFilteringService hotelsFilteringService;
    private final AmenityService amenityService;
    private final HotelSlugService hotelSlugService;
    private final RegionsService regionsService;
    private final TugcService tugcService;
    private final CommonHttpHeaders commonHttpHeaders;
    private final UserCredentials userCredentials;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;
    private final Instant started;
    private final Experiments experiments;
    private final String reqId;
    private final String logId;
    private final SortTypeRegistry sortTypeRegistry;
    private final SearchHotelsReqV1 req;
    private final UaasSearchExperiments uaasSearchExperiments;
    private final HotelImagesService hotelImagesService;
    private final AdditionalSearchHotelsLogData additionalSearchHotelsLogData;
    private final CachedActivePromosService cachedActivePromosService;

    public ViaGeocounterSearcher(HotelsPortalProperties hotelsPortalProperties,
                                 GeoCounterService geoCounter,
                                 GeoSearchService geoSearch,
                                 GeoBase geoBase,
                                 HotelsFilteringService hotelsFilteringService,
                                 AmenityService amenityService,
                                 HotelSlugService hotelSlugService,
                                 RegionsService regionsService,
                                 TugcService tugcService,
                                 ExperimentDataProvider uaasExperimentDataProvider,
                                 HotelImagesService hotelImagesService,
                                 SearchHotelsReqV1 req,
                                 CommonHttpHeaders headers,
                                 UserCredentials userCredentials,
                                 ImageWhitelistDataProvider imageWhitelistDataProvider,
                                 AdditionalSearchHotelsLogData additionalSearchHotelsLogData,
                                 CachedActivePromosService cachedActivePromosService) {
        this.hotelsPortalProperties = hotelsPortalProperties;
        this.geoCounter = geoCounter;
        this.geoSearch = geoSearch;
        this.geoBase = geoBase;
        this.hotelsFilteringService = hotelsFilteringService;
        this.amenityService = amenityService;
        this.hotelSlugService = hotelSlugService;
        this.regionsService = regionsService;
        this.tugcService = tugcService;
        this.commonHttpHeaders = headers;
        this.userCredentials = userCredentials;
        this.imageWhitelistDataProvider = imageWhitelistDataProvider;
        this.additionalSearchHotelsLogData = additionalSearchHotelsLogData;
        this.cachedActivePromosService = cachedActivePromosService;

        uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        this.hotelImagesService = hotelImagesService;
        this.req = req;
        started = Instant.now();
        experiments = new Experiments(req, hotelsPortalProperties);
        reqId = UUID.randomUUID().toString().replace("-", "");
        logId = "SearchHotelsNew(" + reqId + ")";
        sortTypeRegistry = SortTypeRegistry.getSortTypeRegistry(
                HotelsPortalUtils.isHotelsNearbyEnabled(headers),
                HotelsPortalUtils.getSortTypeLayout(uaasSearchExperiments),
                uaasSearchExperiments.isNewRanking() || experiments.isExp("old-new-ranking"),
                uaasSearchExperiments.isPersonalRanking() || experiments.isExp("personal-ranking"),
                HotelsPortalUtils.getExplorationRankingCategory(uaasSearchExperiments, experiments),
                HotelsPortalUtils.getRealTimeRanking(uaasSearchExperiments, experiments, hotelsPortalProperties.isFakeCatBoostRanking()));
    }

    public CompletableFuture<SearchHotelsRspV1> run() {
        var allowTopHotel = true;
        if (!List.of(
                SearchHotelsReqV1.EStartSearchReasonType.mount,
                SearchHotelsReqV1.EStartSearchReasonType.mapBounds,
                SearchHotelsReqV1.EStartSearchReasonType.queryByLocation,
                SearchHotelsReqV1.EStartSearchReasonType.queryWithSameGeo,
                SearchHotelsReqV1.EStartSearchReasonType.navigationToken).contains(req.getStartSearchReason())) {
            allowTopHotel = false;
        }
        var topHotelSelected = allowTopHotel && req.getTopHotelSlug() != null;
        Permalink topHotelPermalink = null;

        TravelPreconditions.checkRequestArgument(req.getGeoId() == null || req.getGeoId() != 0, "GeoId shoudn't be zero");
        Integer travelGeoId;
        if (topHotelSelected) {
            travelGeoId = req.getGeoId();
            topHotelPermalink = HotelSearchUtils.getTopHotelPermalink(hotelSlugService, req);
        } else {
            var userRegion = HotelsPortalUtils.determineUserRegion(geoBase, this.commonHttpHeaders, req);
            travelGeoId = Objects.requireNonNullElse(req.getGeoId(), Objects.requireNonNullElse(userRegion, 213));
        }

        if (!Strings.isNullOrEmpty(req.getContext())) {
            var prefix = req.getContext().split("-", 1);
            if (prefix.length > 0) {
                log.info("{}: context prefix: {}", logId, prefix[0]);
            }
        }

        patchRequestParams();

        var currentPageSkip = Strings.isNullOrEmpty(req.getNavigationToken()) ? 0 : Integer.parseInt(req.getNavigationToken());
        TravelPreconditions.checkRequestArgument(currentPageSkip + req.getTotalHotelLimit() <= 1000, "Navigation token or hotel limit is too big");

        var getHotelsFuture = doGeoCounterGetHotelsRequest(travelGeoId, topHotelPermalink, currentPageSkip, req.getTotalHotelLimit());

        var getCountsFuture = getHotelsFuture.thenCompose(hotelsRsp -> {
            var offerSearchParamsWithBbox = new OfferSearchParamsWithBbox(
                    new OfferSearchParams(hotelsRsp.getCheckInDate(), hotelsRsp.getCheckOutDate(),
                            hotelsRsp.getAges().getAdults(), hotelsRsp.getAges().getChildrenAges()),
                    hotelsRsp.getBbox());
            return HotelSearchUtils.doGeoCounterRequest(
                    log, logId, started, req, hotelsFilteringService, uaasSearchExperiments,
                    experiments, geoCounter, offerSearchParamsWithBbox, commonHttpHeaders
            );
        });

        var geoSearchFuture = getHotelsFuture.thenCompose(hotelsRsp -> {
            if (experiments.isExp("no-geosearch")) {
                return null;
            }

            var permalinks = hotelsRsp.getHotels().stream()
                    .map(x -> Permalink.of(x.getPermalink()))
                    .collect(Collectors.toUnmodifiableList());
            var origin = HotelsPortalUtils.selectGeoOriginByDevice(commonHttpHeaders,
                    GeoOriginEnum.SEARCH_HOTELS_PAGE_DESKTOP, GeoOriginEnum.SEARCH_HOTELS_PAGE_TOUCH);

            var geoReq = GeoSearchReq.byPermalinks(origin, permalinks, commonHttpHeaders);
            HotelsPortalUtils.setAdditionalGeoSearchParams(geoReq, hotelsPortalProperties);
            geoReq.setLimit(permalinks.size());
            geoReq.setIncludeOfferCache(false);
            geoReq.setIncludeSpravPhotos(true);
            geoReq.setIncludePhotos(true);
            geoReq.setIncludeRating(true);
            geoReq.setIncludeNearByStops(true);
            geoReq.setRequestLogId(logId);

            return geoSearch.query(geoReq).thenApply(geoRsp -> {
                String reqId = geoRsp.getResponseInfo() != null ? geoRsp.getResponseInfo().getReqid() : "---";
                log.info("{}: GeoSearch request finished in {} + {} ms, ReqId: {}", logId,
                        geoRsp.getResponseTime().toMillis(),
                        geoRsp.getParseTime().toMillis(), reqId);
                return geoRsp;
            });
        });

        CompletableFuture<HotelFavoriteInfosRsp> tugcFuture;

        if (HotelsPortalUtils.needLikes(commonHttpHeaders, userCredentials)) {
            tugcFuture = getHotelsFuture.thenCompose(hotelsRsp -> {
                var permalinks = hotelsRsp.getHotels().stream()
                        .map(x -> Permalink.of(x.getPermalink()))
                        .collect(Collectors.toUnmodifiableList());
                return tugcService.getHotelFavoriteInfos(logId, userCredentials, permalinks);
            });
        } else {
            tugcFuture = CompletableFuture.completedFuture(null);
        }

        return CompletableFuture.allOf(getHotelsFuture, getCountsFuture, geoSearchFuture, tugcFuture)
                .handle((ignored, ignoredT) -> {
            GeoCounterGetHotelsRsp hotelsRsp;
            CountHotelsRspV1 countRsp;
            GeoSearchRsp geoSearchRsp;
            HotelFavoriteInfosRsp tugcRsp;

            try {
                countRsp = getCountsFuture.join();
            } catch (Exception exc) {
                log.error("{}: Unable to complete geoCounter counts request", logId, exc);
                countRsp = null;
            }

            try {
                hotelsRsp = getHotelsFuture.join();
            } catch (Exception exc) {
                if (exc.getCause() instanceof GeoCounterService.InvalidPollingTokenError) {
                    log.info("{}: Unable to complete geoCounter hotels request, invalid token", logId, exc);
                    return CompletableFuture.<SearchHotelsRspV1>failedFuture(new TravelApiBadRequestException("Invalid context"));
                }
                log.error("{}: Unable to complete geoCounter hotels request", logId, exc);
                return CompletableFuture.<SearchHotelsRspV1>failedFuture(exc);
            }

            try {
                geoSearchRsp = geoSearchFuture.join();
            } catch (Exception exc) {
                log.error("{}: Unable to complete geoSearch request", logId, exc);
                geoSearchRsp = null;
            }

            try {
                tugcRsp = tugcFuture.join();
            } catch (Exception exc) {
                log.error("{}: Unable to complete TUGC request", logId, exc);
                tugcRsp = null;
            }

            fillAdditionalInfo(hotelsRsp);

            return CompletableFuture.completedFuture(prepareResponse(hotelsRsp, geoSearchRsp, countRsp, tugcRsp,
                    currentPageSkip, topHotelSelected));
        }).thenCompose(x -> x);
    }

    private void patchRequestParams() {
        if (req.getPollIteration() == 0) {
            HotelSearchUtils.updateLegacyFilters(req);
            if (req.getStartSearchReason() == SearchHotelsReqV1.EStartSearchReasonType.queryByLocation) {
                HotelSearchUtils.clearNonPersistentFilters(req, hotelsFilteringService, started, commonHttpHeaders, cachedActivePromosService);
            }
            if (req.getStartSearchReason() == SearchHotelsReqV1.EStartSearchReasonType.mount) {
                HotelSearchUtils.clearInvalidFilters(req, hotelsFilteringService, started, commonHttpHeaders, cachedActivePromosService);
            }

            if (List.of(
                    SearchHotelsReqV1.EStartSearchReasonType.mapBounds,
                    SearchHotelsReqV1.EStartSearchReasonType.filters,
                    SearchHotelsReqV1.EStartSearchReasonType.sort).contains(req.getStartSearchReason())) {
                req.setNavigationToken(null);
            }
        }
    }

    private SearchHotelsRspV1 prepareResponse(GeoCounterGetHotelsRsp getHotelsResult,
                                              GeoSearchRsp geoSearchRsp,
                                              CountHotelsRspV1 countResponse,
                                              HotelFavoriteInfosRsp tugcRsp,
                                              int currentPageSkip,
                                              boolean topHotelSelected) {
        List<HotelWithOffers> rspAllHotels = new ArrayList<>();

        var geoHotels = geoSearchRsp == null
                ? Map.<Permalink, GeoHotel>of()
                : geoSearchRsp.getHotels().stream().collect(Collectors.toUnmodifiableMap(x -> x.getPermalink(), x -> x));

        for (var hotel : getHotelsResult.getHotels()) {
            var geoHotel = geoHotels.get(Permalink.of(hotel.getPermalink()));

            var hotelWithOffers = new HotelWithOffers();

            var rspHotel = HotelSearchUtils.extractHotel(
                    hotel, geoHotel, tugcRsp, req, imageWhitelistDataProvider,
                    hotelSlugService, amenityService, hotelImagesService, hotelsPortalProperties,
                    experiments, uaasSearchExperiments);

            if (experiments.isExp("location-on-search-snippet") && hotel.hasDisplayedLocationGeoId()) {
                rspHotel.setLocation(GeoBaseHelpers.getRegionName(geoBase, req.getDomain(), hotel.getDisplayedLocationGeoId()));
            }

            hotelWithOffers.setHotel(rspHotel);
            hotelWithOffers.setSearchIsFinished(hotel.getPollingFinished());
            hotelWithOffers.setSearchedByUser(hotel.getIsTopHotel());

            if (!hotel.hasOfferInfo() && !hotel.getIsTopHotel()) {
                log.error("{}: No offer info for hotel {}", logId, hotel.getPermalink());
                continue;
            }

            if (!hotel.hasOfferInfo()) {
                hotelWithOffers.setOffers(List.of());
            } else {
                var offer = new HotelOffer();

                offer.setPrice(new Price(hotel.getOfferInfo().getPrice(), Price.Currency.RUB));

                if (experiments.isExp("HOTELS-5746")) {
                    if (hotel.getOfferInfo().getCancellationInfo().getRefundType() == ERefundType.RT_FULLY_REFUNDABLE) {
                        offer.setCancellationInfo(prepareCancelationInfo(hotel.getOfferInfo().getCancellationInfo()));
                    }
                    var pansionInfo = hotel.getOfferInfo().getPansionInfo();
                    if (pansionInfo.getPansionType() != EPansionType.PT_RO && pansionInfo.getPansionType() != EPansionType.PT_UNKNOWN) {
                        offer.setMealType(prepareMealType(pansionInfo));
                    }
                } else {
                    offer.setCancellationInfo(prepareCancelationInfo(hotel.getOfferInfo().getCancellationInfo()));
                    offer.setMealType(prepareMealType(hotel.getOfferInfo().getPansionInfo()));
                }
                if (hotel.getOfferInfo().hasStrikethroughPrice() || HotelsPortalUtils.hasForceOfferDiscount(experiments)) {
                    offer.setDiscountInfo(HotelsPortalUtils.extractOfferDiscountInfo(Price.Currency.RUB,
                            hotel.getOfferInfo().getPrice(), hotel.getOfferInfo().getStrikethroughPrice(), experiments));
                }
                if (hotel.getOfferInfo().hasYandexPlusInfo()) {
                    offer.setOfferYandexPlusInfo(new HotelOffer.OfferYandexPlusInfo(hotel.getOfferInfo().getYandexPlusInfo().getPoints(), hotel.getOfferInfo().getYandexPlusInfo().getEligible()));
                }
                hotelWithOffers.setOffers(List.of(offer));
            }

            hotelWithOffers.setBadges(HotelsPortalUtils.toBadges(hotel.getBadgesList(), hotelsPortalProperties));
            if (!hotelWithOffers.getOffers().isEmpty() && experiments.isExp("hotel-badges-in-offer")) {
                hotelWithOffers.getOffers().get(0).setBadges(hotelWithOffers.getBadges());
            }

            rspAllHotels.add(hotelWithOffers);
        }

        SearchHotelsRspV1 rsp = new SearchHotelsRspV1();
        rsp.setHotels(rspAllHotels);
        rsp.setPricedHotelCount(rspAllHotels.size());
        rsp.setOfferSearchProgress(new OfferSearchProgress());
        rsp.getOfferSearchProgress().setFinished(getHotelsResult.isPollingFinished());
        if (!rsp.getOfferSearchProgress().isFinished()) {
            rsp.setNextPollingRequestDelayMs(HotelsPortalUtils.getPollingIterationsDelayMs(hotelsPortalProperties, experiments, req.getPollIteration()));
        }
        var offerSearchParams = new OfferSearchParams();
        offerSearchParams.setCheckinDate(getHotelsResult.getCheckInDate());
        offerSearchParams.setCheckoutDate(getHotelsResult.getCheckOutDate());
        offerSearchParams.setAdults(getHotelsResult.getAges().getAdults());
        offerSearchParams.setChildrenAges(getHotelsResult.getAges().getChildrenAges());
        rsp.setOfferSearchParams(offerSearchParams);
        rsp.setOperatorById(Collections.emptyMap());
        rsp.setPollEpoch(req.getPollEpoch());
        rsp.setPollIteration(req.getPollIteration());
        rsp.setSearchPagePollingId(getHotelsResult.getPollingId());
        Boolean hasBoyOffers = getHotelsResult.getHasBoyOffers();
        if (hasBoyOffers != null && getHotelsResult.isPollingFinished()){
            rsp.setHasBoyOffers(hasBoyOffers);
        }
        var searchGeoId = getHotelsResult.getGeoId();
        var actualGeoId = getHotelsResult.getGeoId();

        rsp.setBboxAsString(getHotelsResult.getBbox());
        List<Coordinates> bboxAsList = new ArrayList<>();
        bboxAsList.add(rsp.getBboxAsString().getLeftDown());
        bboxAsList.add(rsp.getBboxAsString().getUpRight());
        rsp.setBboxAsStruct(bboxAsList);
        rsp.setActualRegion(HotelSearchUtils.fillShortRegion(geoBase, req.getDomain(), actualGeoId));
        rsp.setSearchRegion(HotelSearchUtils.fillRegion(regionsService, req.getDomain(), searchGeoId));
        rsp.setContext(getHotelsResult.getPollingContext());

        SearchHotelsRspV1.NavigationTokens navigationTokens = new SearchHotelsRspV1.NavigationTokens();
        navigationTokens.setPrevPage(currentPageSkip == 0 ? "" : Integer.toString(currentPageSkip - req.getPageHotelCount()));
        var haveNextPage = getHotelsResult.getHotelResultCounts().getHotelsOnCurrentPageCount() > req.getPageHotelCount() ||
                (getHotelsResult.getHotelResultCounts().getHotelsOnCurrentPageCount() == req.getPageHotelCount() && getHotelsResult.getHotelResultCounts().isHaveMoreHotels());
        navigationTokens.setNextPage(haveNextPage ? Integer.toString(currentPageSkip + req.getPageHotelCount()) : "");
        navigationTokens.setCurrentPage(Integer.toString(currentPageSkip));
        rsp.setNavigationTokens(navigationTokens);

        SearchFilterAndTextParams filterParams = new SearchFilterAndTextParams();
        filterParams.setFilterAtoms(req.getFilterAtoms());
        filterParams.setFilterPriceFrom(req.getFilterPriceFrom());
        filterParams.setFilterPriceTo(req.getFilterPriceTo());

        var filterInfo = hotelsFilteringService.composeDefaultFilterInfo(req, started,
                HotelSearchUtils.getForceFilterLayout(req), searchGeoId, req.isOnlyCurrentGeoId(), commonHttpHeaders);
        filterInfo.setParams(filterParams);
        rsp.setFilterInfo(filterInfo);
        rsp.setSearchControlInfo(hotelsFilteringService.composeDefaultControlInfo(req, started,
                HotelSearchUtils.getForceFilterLayout(req), searchGeoId, commonHttpHeaders));

        // Пустая выдача
        if (getHotelsResult.isPollingFinished() && getHotelsResult.getHotelResultCounts().getTotalHotelsCount() == 0) {
            SearchHotelsRspV1.ResetFilterInfo resetFilterInfo = hotelsFilteringService.composeResetFilterInfo(req, started,
                    HotelSearchUtils.getForceFilterLayout(req), searchGeoId, commonHttpHeaders);
            filterInfo.setResetFilterInfo(resetFilterInfo);
        }

        // Add geoCounter data
        if (countResponse != null) {
            rsp.setFoundHotelCount(countResponse.getFoundHotelCount());
            rsp.getFilterInfo().setQuickFilters(countResponse.getFilterInfo().getQuickFilters());
            rsp.getFilterInfo().setDetailedFilters(countResponse.getFilterInfo().getDetailedFilters());
            rsp.getFilterInfo().setDetailedFiltersBatches(countResponse.getFilterInfo().getDetailedFiltersBatches());
            rsp.getFilterInfo().setPriceFilter(countResponse.getFilterInfo().getPriceFilter());
            rsp.setSearchControlInfo(countResponse.getSearchControlInfo());
        } else {
            rsp.setFoundHotelCount(rsp.getHotels().size());
        }

        var selectedSort = sortTypeRegistry.getSortTypeByGeoCounterType(getHotelsResult.getSortType()).getId();
        var sortOrigin = getHotelsResult.getSortOrigin();

        // Add sort info
        rsp.setSortInfo(HotelSearchUtils.buildSortInfo(sortTypeRegistry, selectedSort, sortOrigin));

        if (MirUtils.isActive(cachedActivePromosService.getActivePromos())) {
            rsp.setSearchBannerType(SearchHotelsRspV1.ESearchBannerType.MIR);
        }

        if (topHotelSelected) {
            rsp.setTopHotelSlug(req.getTopHotelSlug());
        }

        var pageParams = ExtraVisitAndUserParamsUtils.initParamsMap();
        ExtraVisitAndUserParamsUtils.fillSearchParams(pageParams, rsp.getOfferSearchParams());
        ExtraVisitAndUserParamsUtils.fillGeoParams(pageParams, req.getDomain(), actualGeoId, geoBase);
        pageParams.put("lastUsedSort", selectedSort);
        pageParams.put("topHotelSelected", Boolean.toString(topHotelSelected));
        pageParams.put("newSearchUsed", Boolean.toString(true));
        pageParams.put("newSearchPossible", Boolean.toString(true));
        pageParams.put("pollingId", getHotelsResult.getPollingId());
        pageParams.put("cryptaDataAvailable", Boolean.toString(getHotelsResult.isCryptaDataAvailable()));

        ExtraVisitAndUserParamsUtils.fillFilterParams(pageParams, filterParams);

        rsp.setExtraVisitAndUserParams(ExtraVisitAndUserParamsUtils.createForSearchPage(pageParams));
        rsp.setTimingInfo(new SearchHotelsRspV1.TimingInfo(Duration.between(started, Instant.now()).toMillis()));
        return rsp;
    }

    private void fillAdditionalInfo(GeoCounterGetHotelsRsp hotelsRsp) {
        if (hotelsRsp.getLogData() != null) {
            if (hotelsRsp.getLogData().hasUserSegmentsInfo()) {
                var segments = hotelsRsp.getLogData().getUserSegmentsInfo().getSegmentsList().stream()
                        .map(pbSegment -> new AdditionalSearchHotelsLogData.UserSegment(
                                pbSegment.getKeywordId(),
                                pbSegment.getKeywordName(),
                                pbSegment.getSegmentId(),
                                pbSegment.getSegmentName(),
                                mapUserSegmentType(pbSegment.getSegmentType()),
                                pbSegment.getSegmentType() == EUserSegmentType.UST_Weighted ? pbSegment.getWeight() : null))
                        .collect(Collectors.toUnmodifiableList());
                additionalSearchHotelsLogData.setUserSegmentsInfo(new AdditionalSearchHotelsLogData.UserSegmentsInfo(segments));
            }
            if (hotelsRsp.getLogData().hasRealTimeRankingInfo()) {
                var hotelRankingInfos = hotelsRsp.getLogData().getRealTimeRankingInfo().getHotelInfoList().stream()
                        .map(pbRealTimeRankingInfo -> new AdditionalSearchHotelsLogData.HotelRealTimeRankingInfo(
                                pbRealTimeRankingInfo.getPermalink(),
                                pbRealTimeRankingInfo.getCatBoostUsed(),
                                pbRealTimeRankingInfo.hasCatBoostPrediction() ? (double)pbRealTimeRankingInfo.getCatBoostPrediction() : null
                        ))
                        .collect(Collectors.toUnmodifiableList());
                additionalSearchHotelsLogData.setRealTimeRankingInfo(new AdditionalSearchHotelsLogData.RealTimeRankingInfo(hotelRankingInfos));
            }
        }
    }

    private AdditionalSearchHotelsLogData.UserSegmentType mapUserSegmentType(EUserSegmentType segmentType) {
        if (segmentType == EUserSegmentType.UST_NoValue) {
            return AdditionalSearchHotelsLogData.UserSegmentType.NO_VALUE;
        } else if (segmentType == EUserSegmentType.UST_Weighted) {
            return AdditionalSearchHotelsLogData.UserSegmentType.WEIGHTED;
        }
        return AdditionalSearchHotelsLogData.UserSegmentType.UNKNOWN;
    }

    private SortTypeRegistry.SortType detectSelectedSort() {
        SortTypeRegistry.SortType selectedSortType;
        if (req.getSelectedSortId() != null) {
            selectedSortType = sortTypeRegistry.getSortType(req.getSelectedSortId());
        } else {
            selectedSortType = sortTypeRegistry.getDefaultSortType();
        }
        return selectedSortType;
    }

    private Coordinates detectSortOrigin(SortTypeRegistry.SortType sortType, int skip) {
        if (!sortType.isByDistance()) {
            return null;
        }
        if (req.getPollIteration() == 0 && skip == 0) {
            TravelPreconditions.checkRequestArgument(req.getSortOrigin() != null || req.getUserCoordinates() != null,
                    "sortOrigin or userCoordinates required for selected_sort_id=%s on new search", req.getSelectedSortId());
            return Objects.requireNonNullElse(req.getUserCoordinates(), req.getSortOrigin());
        } else {
            TravelPreconditions.checkRequestArgument(req.getSortOrigin() != null,
                    "sortOrigin required for selected_sort_id=%s on search continuation", req.getSelectedSortId());
            return req.getSortOrigin();
        }
    }

    private HotelOffer.MealType prepareMealType(TGetHotelsResponse.TPansionInfo pansionInfo) {
        HotelOffer.MealType mt = new HotelOffer.MealType();
        mt.setId(pansionInfo.getPansionType().name());
        mt.setName(pansionInfo.getPansionName());
        return mt;
    }

    private HotelOffer.CancellationInfo prepareCancelationInfo(TGetHotelsResponse.TCancellationInfo cancellationInfo) {
        var ci = new HotelOffer.CancellationInfo();
        ci.setHasFreeCancellation(cancellationInfo.getHasFreeCancellation());
        ci.setRefundType(HotelsPortalUtils.mapRefundProtoToPojo(cancellationInfo.getRefundType()));
        ci.setRefundRules(HotelsPortalUtils.mapProtoRefundRulesToPojo(cancellationInfo.getRefundRulesList(), Price.Currency.RUB));
        return ci;
    }

    private CompletableFuture<GeoCounterGetHotelsRsp> doGeoCounterGetHotelsRequest(Integer travelGeoId, Permalink topHotelPermalink, int skip, int limit) {
        log.info("{}: Doing getHotels request to geoCounter", logId);
        GeoOriginEnum origin = HotelsPortalUtils.selectGeoOriginByDevice(commonHttpHeaders,
                GeoOriginEnum.SEARCH_HOTELS_PAGE_DESKTOP, GeoOriginEnum.SEARCH_HOTELS_PAGE_TOUCH);
        var selectedSort = detectSelectedSort();
        var sortOrigin = detectSortOrigin(selectedSort, skip);
        var getHotelsReqProto = HotelSearchUtils.prepareGeoCounterGetHotelsReq(travelGeoId, req, req, req.getBbox(), started,
                HotelSearchUtils.getForceFilterLayout(req), selectedSort.getGeoCounterSortType(), sortOrigin, req.getContext(),
                true, req.getPollIteration(), skip, limit, reqId, !HotelsPortalUtils.isRobotRequest(commonHttpHeaders),
                origin, topHotelPermalink, uaasSearchExperiments, req, commonHttpHeaders, geoBase,
                userCredentials.isLoggedIn(), hotelsPortalProperties.isSortOffersUsingPlus(),
                hotelsFilteringService, parseForceCryptaSegments(req.getForceCryptaSegment()), false);
        return geoCounter.getHotels(new GeoCounterGetHotelsReq(getHotelsReqProto, logId, req.isDebugUseProdOffers())).whenComplete((gcRsp, t) -> {
            if (t == null) {
                log.info("{}: Processed getHotels response from geoCounter in {} ms", logId,
                        gcRsp.getResponseTime().toMillis());
            } else {
                log.info("{}: Finished failed getHotels request to geoCounter in {} ms", logId,
                        gcRsp.getResponseTime().toMillis());
            }
            log.info("{}: GeoCounter reqId: {}, sessionId: {}", logId, gcRsp.getReqId(), gcRsp.getSessionId());
        });
    }

    private Map<Integer, Integer> parseForceCryptaSegments(String forceCryptaSegments) {
        if (Strings.isNullOrEmpty(forceCryptaSegments)) {
            return null;
        }
        if (forceCryptaSegments.equals("empty")) {
            return Map.of();
        }
        return Arrays.stream(forceCryptaSegments.split("\\|"))
                .map(x -> {
                    var parts = x.split("~");
                    TravelPreconditions.checkRequestArgument(parts.length == 2, "Invalid crypta segments format");
                    return parts;
                })
                .map(x -> {
                    try {
                        var key = Integer.parseInt(x[0]);
                        var value = Integer.parseInt(x[1]);
                        return List.of(key, value);
                    } catch (NumberFormatException e) {
                        throw new TravelApiBadRequestException(String.format("Invalid key (%s) or value (%s) - expected integer", x[0], x[1]));
                    }
                })
                .collect(Collectors.toUnmodifiableMap(x -> x.get(0), x -> x.get(1)));
    }
}
