package ru.yandex.travel.api.endpoints.hotels_portal.breadcrumbs;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.geobase6.RegionHash;
import ru.yandex.travel.api.models.Region;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.breadcrumbs.Breadcrumb;
import ru.yandex.travel.api.models.hotels.breadcrumbs.Breadcrumbs;
import ru.yandex.travel.api.models.hotels.breadcrumbs.FilterBreadcrumb;
import ru.yandex.travel.api.models.hotels.breadcrumbs.GeoRegionBreadcrumb;
import ru.yandex.travel.api.models.hotels.breadcrumbs.HotelBreadcrumb;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.static_pages.RegionPagesStorage;

@Component
@EnableConfigurationProperties(ru.yandex.travel.api.endpoints.hotels_portal.breadcrumbs.BreadcrumbsServiceProperties.class)
@Slf4j
@RequiredArgsConstructor
public class BreadcrumbsService {

    private final BreadcrumbsServiceProperties config;
    private final GeoBase geoBase;
    private final RegionsService regionsService;
    private final RegionPagesStorage regionPagesStorage;

    public Breadcrumbs getHotelBreadcrumbs(String domain, Coordinates coordinates) {
        // Current logic is to use only last breadcrumb, see HOTELS-4822
        try {
            int geoId = geoBase.getRegionIdByLocation(coordinates.getLat(), coordinates.getLon());
            List<Integer> geoIds = getHotelGeoIds(geoId, domain, config.getAllowedHotelRegionTypes());
            if (!geoIds.isEmpty()) {
                geoIds = Collections.singletonList(geoIds.get(geoIds.size() - 1));
            }
            return getBreadcrumbsFromGeoIds(geoIds, domain, false);
        } catch (Exception e) {// For example it may happen if geobase is disabled
            log.warn("Failed to get hotel breadcrumbs for {}, cause: {}", coordinates, e.toString());
            return null;
        }
    }

    public Breadcrumbs getHotelSeoBreadcrumbs(String domain, Coordinates coordinates, String hotelName, String hotelSlug) {
        try {
            int geoId = geoBase.getRegionIdByLocation(coordinates.getLat(), coordinates.getLon());
            Breadcrumbs breadcrumbs = getBreadcrumbsFromGeoIds(getGeoIdsChainReversed(geoId, domain), domain, true);
            HotelBreadcrumb hotelBreadcrumb = HotelBreadcrumb.builder()
                    .name(hotelName)
                    .slug(hotelSlug)
                    .build();
            breadcrumbs.getItems().add(hotelBreadcrumb);
            breadcrumbs.setGeoRegions(null);
            return breadcrumbs;
        } catch (Exception e) {// For example it may happen if geobase is disabled
            log.warn("Failed to get hotel seo breadcrumbs for {}, cause: {}", coordinates, e.toString());
            return null;
        }
    }

    public Breadcrumbs getStaticPageBreadcrumbs(String domain, int geoId, String filterName, String filterSlug) {
        try {
            Breadcrumbs breadcrumbs = getBreadcrumbsFromGeoIds(getGeoIdsChainReversed(geoId, domain), domain, true);
            if (!Strings.isNullOrEmpty(filterName)) {
                FilterBreadcrumb filterBreadcrumb = FilterBreadcrumb.builder()
                        .name(filterName)
                        .slug(filterSlug)
                        .build();
                breadcrumbs.getItems().add(filterBreadcrumb);
            }
            return breadcrumbs;
        } catch (Exception e) {// For example it may happen if geobase is disabled
            log.warn("Failed to get static page breadcrumbs for geoId {}, cause: {}", geoId, e.toString());
            return null;
        }
    }

    private List<Integer> getHotelGeoIds(int geoId, String domain, List<Integer> allowedRegionTypes) {
        Map<Integer, Integer> type2GeoId = new HashMap<>();
        Integer firstGeoId = null;
        while (geoId != GeoBaseHelpers.WORLD_REGION) {
            RegionHash hash = geoBase.getRegionById(geoId, domain);
            Integer type = hash.getAttr("type").getInteger();
            type2GeoId.put(type, geoId);
            if (firstGeoId == null) {
                firstGeoId = geoId;
            }
            Integer parentGeoId = config.getParentReplaces().get(geoId);
            if (parentGeoId == null) {
                parentGeoId = geoBase.getParentId(geoId, domain);
            }
            geoId = parentGeoId;
        }
        List<Integer> geoIds = new ArrayList<>();
        Integer lastType = null;
        for (Integer type : allowedRegionTypes) {
            Integer partGeoId = type2GeoId.get(type);
            if (partGeoId != null) {
                geoIds.add(partGeoId);
            }
            lastType = type;
        }
        if (lastType != null && !type2GeoId.containsKey(lastType)) {
            // для загородных отелей
            geoIds.add(firstGeoId);
        }
        return geoIds;
    }

    private List<Integer> getGeoIdsChainReversed(int geoId,  String domain) {
        List<Integer> geoIds = new ArrayList<>();
        for (GeoBaseHelpers.RegionInfo region : GeoBaseHelpers.getRegionChain(geoBase, geoId, domain)) {
            geoIds.add(region.getGeoId());
        }
        Collections.reverse(geoIds);
        return geoIds;
    }

    private Breadcrumbs getBreadcrumbsFromGeoIds(List<Integer> geoIds, String domain, boolean onlyRegionsWithSeoPages) {
        List<Region> regionList = new ArrayList<>();
        List<Breadcrumb> breadcrumbList = new ArrayList<>();
        for (Integer regionGeoId: geoIds) {
            if (onlyRegionsWithSeoPages && regionPagesStorage.tryGetRegionPage(regionGeoId).isEmpty()) {
                continue;
            }
            Region region = regionsService.getRegion(regionGeoId, domain, "ru");
            regionList.add(region);
            breadcrumbList.add(new GeoRegionBreadcrumb(region));
        }
        Breadcrumbs breadcrumbs = new Breadcrumbs();
        breadcrumbs.setGeoRegions(regionList);
        breadcrumbs.setItems(breadcrumbList);
        return breadcrumbs;
    }
}
