package ru.yandex.travel.api.endpoints.idm

import com.fasterxml.jackson.databind.ObjectMapper
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.*
import org.springframework.web.context.request.async.DeferredResult
import ru.yandex.travel.api.endpoints.idm.req_rsp.BaseIdmResponseV2
import ru.yandex.travel.api.endpoints.idm.req_rsp.GetAllRolesResponseV2
import ru.yandex.travel.api.endpoints.idm.req_rsp.InfoResponseV2
import ru.yandex.travel.api.endpoints.idm.req_rsp.ModifyRoleResponseV2
import ru.yandex.travel.api.infrastucture.ResponseProcessor
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers
import ru.yandex.travel.api.services.idm.IdmServiceGateway
import ru.yandex.travel.api.services.idm.model.errors.IdmInputException
import ru.yandex.travel.orders.client.HAGrpcException
import java.io.IOException

@RestController
@RequestMapping(value = ["/api/idm/v2"])
class IdmControllerV2(private val serviceGateway: IdmServiceGateway, private val responseProcessor: ResponseProcessor) {

    private val mapper = ObjectMapper()

    @GetMapping(value = ["/info/"], produces = [MediaType.APPLICATION_JSON_VALUE])
    fun info(): DeferredResult<InfoResponseV2> {
        return responseProcessor.replyWithFutureRetrying(
            "IdmInfo",
            {
                serviceGateway.roles().thenApply { InfoResponseV2(roles = it) }
            },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
        )
    }

    /*
    https://wiki.yandex-team.ru/intranet/idm/API#add-role
     */
    @PostMapping(
        value = ["/add-role/"],
        consumes = [MediaType.APPLICATION_FORM_URLENCODED_VALUE],
        produces = [MediaType.APPLICATION_JSON_VALUE]
    )
    fun addRole(
        @RequestParam(value = "login") login: String,
        @RequestParam(value = "role") role: String,
        @RequestParam(value = "uid", required = false) passportUid: String?,  // yandex-team passport uid
        @RequestParam(value = "path", required = false) path: String?,
        @RequestParam(value = "fields", required = false) fields: String?,
    ): DeferredResult<ModifyRoleResponseV2> {
        try {
            val roleMap = mapper.readerForMapOf(String::class.java).readValue<Map<String, String>>(role)
            val fieldsMap =
                if (fields != null) mapper.readerForMapOf(Any::class.java).readValue<Map<String, Any>>(fields) else null
            return responseProcessor.replyWithFuture(
                "IdmAddRole"
            ) {
                serviceGateway.addRole(login, roleMap, passportUid, fieldsMap).thenApply {
                    ModifyRoleResponseV2(
                        code = it.code,
                        warning = it.warning,
                        error = it.error,
                        fatal = it.fatal,
                        data = fieldsMap
                    )
                }
            }
        } catch (e: IOException) {
            throw IllegalArgumentException(e)
        }
    }

    @PostMapping(
        value = ["/remove-role/"],
        consumes = [MediaType.APPLICATION_FORM_URLENCODED_VALUE],
        produces = [MediaType.APPLICATION_JSON_VALUE],
    )
    fun removeRole(
        login: String,
        role: String,
        path: String?,
        fields: String?,
    ): DeferredResult<ModifyRoleResponseV2> {
        try {
            val roleMap = mapper.readerForMapOf(String::class.java).readValue<Map<String, String>>(role)
            return responseProcessor.replyWithFuture(
                "IdmRemoveRole"
            ) {
                serviceGateway.removeRole(login, roleMap).thenApply {
                    ModifyRoleResponseV2(
                        code = it.code,
                        warning = it.warning,
                        error = it.error,
                        fatal = it.fatal,
                    )
                }
            }
        } catch (e: IOException) {
            throw IllegalArgumentException(e)
        }
    }

    @GetMapping(
        value = ["/get-all-roles/"],
        produces = [MediaType.APPLICATION_JSON_VALUE],
    )
    fun getAllRoles(): DeferredResult<GetAllRolesResponseV2> {
        return responseProcessor.replyWithFuture("GetAllRoles") {
            serviceGateway.getAllRoles().thenApply { resp ->
                GetAllRolesResponseV2(users = resp.map { user ->
                    GetAllRolesResponseV2.User(
                        login = user.login,
                        roles = user.roles.map {
                            listOf(it.slugs, it.fields)
                        })
                })
            }
        }
    }

    @ExceptionHandler(HAGrpcException::class)
    fun handleHAGrpcException(e: HAGrpcException): ResponseEntity<BaseIdmResponseV2> {
        return ResponseEntity(BaseIdmResponseV2(code = 2, error = e.message), HttpStatus.OK)
    }

    @ExceptionHandler(IdmInputException::class)
    fun handleInputErrors(ex: IdmInputException): ResponseEntity<BaseIdmResponseV2> {
        return ResponseEntity(BaseIdmResponseV2(code = 3, fatal = ex.message), HttpStatus.OK)
    }
}
