package ru.yandex.travel.api.endpoints.komod.cross_search;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.komod.cross_search.req_rsp.TransformSearchContextReqV1;
import ru.yandex.travel.api.endpoints.komod.cross_search.req_rsp.TransformSearchContextRspV1;
import ru.yandex.travel.api.endpoints.komod.cross_search.req_rsp.TravelVertical;
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station_to_settlement.TrainStationToSettlementDataProvider;
import ru.yandex.travel.api.services.geo.CrossSearchPointProvider;
import ru.yandex.travel.api.services.geo.model.CrossSearchPointInfo;
import ru.yandex.travel.api.services.geo.model.PointId;
import ru.yandex.travel.dicts.rasp.proto.TStation;
import ru.yandex.travel.dicts.rasp.proto.TTransport;

@Component
@RequiredArgsConstructor
public class CrossSearchControllerImpl {
    private final CrossSearchPointProvider pointProvider;
    private final TrainStationDataProvider stationDataProvider;
    private final TrainStationToSettlementDataProvider stationToSettlementDataProvider;

    private static final String locale = "ru";
    private static final String domain = "ru";
    private static final Set<TStation.EType> busStationTypes = Set.of(TStation.EType.TYPE_BUS_STATION, TStation.EType.TYPE_BUS_STOP);

    public CompletableFuture<TransformSearchContextRspV1> transformSearchContext(TransformSearchContextReqV1 req) {
        var from = PointId.builder().pointKey(req.getFromPointKey()).geoId(req.getFromGeoId()).build();
        var to = PointId.builder().pointKey(req.getToPointKey()).geoId(req.getToGeoId()).build();
        var response = TransformSearchContextRspV1.builder()
                .from(applyVerticalFilter(getPoint(from), req.getVertical()))
                .to(applyVerticalFilter(getPoint(to), req.getVertical()))
                .build();
        return CompletableFuture.completedFuture(response);
    }

    private CrossSearchPointInfo getPoint(PointId pointId) {
        if (pointId == null) {
            return null;
        }
        if (!Strings.isNullOrEmpty(pointId.getPointKey())) {
            return pointProvider.getByPointKey(pointId, locale, domain);
        }
        if (pointId.getGeoId() != null) {
            return pointProvider.getByGeoId(pointId.getGeoId(), domain, locale);
        }
        // TODO: the branch that transforms hotels context will be implemented later
        return null;
    }

    private CrossSearchPointInfo applyVerticalFilter(CrossSearchPointInfo pointInfo, TravelVertical vertical) {
        var isTransport = vertical == TravelVertical.AVIA || vertical == TravelVertical.TRAINS || vertical == TravelVertical.BUSES;
        if (!isTransport) {
            return pointInfo;
        }
        if (pointInfo == null || Strings.isNullOrEmpty(pointInfo.getPointKey())) {
            return null;
        }
        var pointId = Integer.parseInt(pointInfo.getPointKey().substring(1));
        if (vertical == TravelVertical.AVIA && hasAirport(pointId)) {
            return pointInfo;
        }
        if (vertical == TravelVertical.TRAINS && hasRailwayStation(pointId)) {
            return pointInfo;
        }
        if (vertical == TravelVertical.BUSES && hasBusStation(pointId)) {
            return pointInfo;
        }
        return null;
    }

    private boolean hasAirport(int settlementId) {
        return getStations(settlementId).stream().anyMatch(s -> s.getType() == TStation.EType.TYPE_AIRPORT);
    }

    private boolean hasRailwayStation(int settlementId) {
        return getStations(settlementId).stream().anyMatch(s ->
                s.getType() == TStation.EType.TYPE_TRAIN_STATION ||
                        s.getType() == TStation.EType.TYPE_STATION && s.getTransportType() == TTransport.EType.TYPE_TRAIN
        );
    }

    private boolean hasBusStation(int settlementId) {
        return getStations(settlementId).stream().anyMatch(s ->
                busStationTypes.contains(s.getType()) ||
                        s.getType() == TStation.EType.TYPE_STATION && s.getTransportType() == TTransport.EType.TYPE_BUS
        );
    }

    private List<TStation> getStations(int settlementId) {
        var stations = stationDataProvider.findBySettlementId(settlementId);
        var stationsToSettlementStream =
                stationToSettlementDataProvider.getStationIds(settlementId).stream().map(id -> {
                    try {
                        return stationDataProvider.getById(id);
                    } catch (Exception e) {
                        return null;
                    }
                }).filter(Objects::nonNull).filter(s -> !s.getIsHidden());
        return Stream.concat(stations.stream(), stationsToSettlementStream).collect(Collectors.toUnmodifiableList());
    }
}
