package ru.yandex.travel.api.endpoints.personalization;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.personalization.req_rsp.AviaHistoryReqV1;
import ru.yandex.travel.api.endpoints.personalization.req_rsp.HotelsSuggestReqV1;
import ru.yandex.travel.api.endpoints.personalization.req_rsp.PersonalizationRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.commons.http.CommonHttpHeaders;

@RestController
@RequestMapping(value = "/api/personalization")
@RequiredArgsConstructor
public class PersonalizationController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final PersonalizationControllerImpl personalizationControllerImpl;

    @RequestMapping(value = "/v1/hotels/suggest", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<PersonalizationRspV1> getHotelsSuggest(@BindFromQuery HotelsSuggestReqV1 req) {
        return responseProcessor.replyWithFutureRetrying("PersonalizationGetHotelsSuggest",
                () -> personalizationControllerImpl.getHotelsSuggest(req, CommonHttpHeaders.get()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/avia/history", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<PersonalizationRspV1> getAviaHistory(@BindFromQuery AviaHistoryReqV1 req) {
        return responseProcessor.replyWithFutureRetrying("PersonalizationGetAviaHistory",
                () -> personalizationControllerImpl.getAviaHistory(req, CommonHttpHeaders.get()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
