package ru.yandex.travel.api.endpoints.personalization;

import java.time.LocalDate;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.personalization.req_rsp.AviaHistoryReqV1;
import ru.yandex.travel.api.endpoints.personalization.req_rsp.HotelsSuggestReqV1;
import ru.yandex.travel.api.endpoints.personalization.req_rsp.PersonalizationRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.Region;
import ru.yandex.travel.api.models.avia.Passengers;
import ru.yandex.travel.api.models.avia.ServiceClass;
import ru.yandex.travel.api.models.hotels.OfferSearchParams;
import ru.yandex.travel.api.models.personalization.AviaItem;
import ru.yandex.travel.api.models.personalization.AviaPoint;
import ru.yandex.travel.api.models.personalization.HotelsItem;
import ru.yandex.travel.api.models.personalization.PersonalizationItem;
import ru.yandex.travel.api.models.personalization.PersonalizationItemType;
import ru.yandex.travel.api.models.personalization.PointType;
import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.personalization.PersonalizationClientFactory;
import ru.yandex.travel.avia.personalization.personal_search.v2.EntryType;
import ru.yandex.travel.avia.personalization.personal_search.v2.TAviaEntry;
import ru.yandex.travel.avia.personalization.personal_search.v2.TGeoPoint;
import ru.yandex.travel.avia.personalization.personal_search.v2.TGetAviaHistoryRequestV2;
import ru.yandex.travel.avia.personalization.personal_search.v2.TGetHotelsSuggestRequestV2;
import ru.yandex.travel.avia.personalization.personal_search.v2.TGetPersonalSearchResponseV2;
import ru.yandex.travel.avia.personalization.personal_search.v2.THotelEntry;
import ru.yandex.travel.avia.personalization.personal_search.v2.TPersonalSearchEntryV2;
import ru.yandex.travel.avia.personalization.personal_search.v2.TTravelers;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;

@Component
@RequiredArgsConstructor
@Slf4j
public class PersonalizationControllerImpl {
    private final PersonalizationClientFactory personalizationClientFactory;
    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final GeoBase geoBase;

    private static final String locale = "ru";

    public CompletableFuture<PersonalizationRspV1> getHotelsSuggest(HotelsSuggestReqV1 req, CommonHttpHeaders headers) {
        if (Strings.isNullOrEmpty(headers.getYandexUid())) {
            return CompletableFuture.completedFuture(PersonalizationRspV1.builder().items(List.of()).build());
        }
        var requestProtoBuilder = TGetHotelsSuggestRequestV2.newBuilder()
                .setYandexUid(headers.getYandexUid())
                .setOrdersLimit(req.getOrdersLimit())
                .setSearchesLimit(req.getSearchesLimit());
        if (headers.getPassportId() != null) {
            requestProtoBuilder.setPassportId(headers.getPassportId());
        }
        var requestProto = requestProtoBuilder.build();
        var personalizationClient = personalizationClientFactory.createRoundRobinStub();
        return FutureUtils.buildCompletableFuture(personalizationClient.getHotelsSuggest(requestProto))
                .thenApply(this::mapResponse);
    }

    public CompletableFuture<PersonalizationRspV1> getAviaHistory(AviaHistoryReqV1 req, CommonHttpHeaders headers) {
        if (Strings.isNullOrEmpty(headers.getYandexUid())) {
            return CompletableFuture.completedFuture(PersonalizationRspV1.builder().items(List.of()).build());
        }
        var requestProtoBuilder = TGetAviaHistoryRequestV2.newBuilder()
                .setYandexUid(headers.getYandexUid())
                .setLimit(req.getLimit());
        if (headers.getPassportId() != null) {
            requestProtoBuilder.setPassportId(headers.getPassportId());
        }
        var requestProto = requestProtoBuilder.build();
        var personalizationClient = personalizationClientFactory.createRoundRobinStub();
        return FutureUtils.buildCompletableFuture(personalizationClient.getAviaHistory(requestProto))
                .thenApply(this::mapResponse);
    }

    private PersonalizationRspV1 mapResponse(TGetPersonalSearchResponseV2 rsp) {
        return PersonalizationRspV1.builder().items(mapItems(rsp.getEntriesList())).build();
    }

    private List<PersonalizationItem> mapItems(List<TPersonalSearchEntryV2> entries) {
        return entries.stream().map(this::mapItem).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private PersonalizationItem mapItem(TPersonalSearchEntryV2 entry) {
        try {
            if (entry.hasHotel()) {
                return mapHotelItem(entry.getHotel());
            }
            if (entry.hasAvia()) {
                return mapAviaItem(entry.getAvia());
            }
        } catch (Exception e) {
            log.error("Failed to map personalization entry", e);
        }
        return null;
    }

    private PersonalizationItem mapAviaItem(TAviaEntry entry) {
        return AviaItem.builder()
                .from(mapAviaPoint(entry.getPointFrom()))
                .to(mapAviaPoint(entry.getPointTo()))
                .passengers(mapPassengers(entry.getTravelers()))
                .when(entry.getWhen())
                .returnDate(entry.getReturnDate())
                .serviceClass(mapServiceClass(entry.getAviaClass()))
                .build();
    }

    private AviaPoint mapAviaPoint(TGeoPoint point) {
        var settlement = getSettlementByPointCode(point.getPointCode());
        if (settlement == null) {
            log.warn("no settlement with point code: {}", point.getPointCode());
            return null;
        }
        var settlementLinguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, settlement.getGeoId(), locale);
        var settlementTitle = settlementLinguistics.getNominativeCase();
        var pointBuilder = AviaPoint.builder()
                .type(PointType.SETTLEMENT)
                .cityTitle(settlementTitle)
                .title(settlementTitle)
                .pointCode(trainSettlementDataProvider.getSettlementCode(settlement.getId()))
                .pointKey(TrainSettlementDataProvider.Companion.prepareRaspSettlementId(settlement.getId()));
        var regionGeoId = GeoBaseHelpers.getRegionRoundTo(
                geoBase, settlement.getGeoId(), GeoBaseHelpers.REGION_REGION_TYPE, locale
        );
        if (regionGeoId != null) {
            var regionTitle = GeoBaseHelpers.getRegionLinguistics(geoBase, regionGeoId, locale).getNominativeCase();
            pointBuilder = pointBuilder.regionTitle(regionTitle);
        }
        var countryGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, settlement.getGeoId(),
                GeoBaseHelpers.COUNTRY_REGION_TYPE, locale);
        if (countryGeoId != null) {
            var countryTitle = GeoBaseHelpers.getRegionLinguistics(geoBase, countryGeoId, locale).getNominativeCase();
            pointBuilder = pointBuilder.countryTitle(countryTitle);
        }
        return pointBuilder.build();
    }

    private ServiceClass mapServiceClass(String aviaClass) {
        return ServiceClass.fromString(aviaClass.toLowerCase());
    }

    private Passengers mapPassengers(TTravelers travelers) {
        return Passengers.builder()
                .adults((int) travelers.getAdults())
                .children((int) travelers.getChildren())
                .infants((int) travelers.getInfants())
                .build();
    }

    private HotelsItem mapHotelItem(THotelEntry entry) {
        if (Strings.isNullOrEmpty(entry.getPointTo().getPointCode())) {
            return null;
        }
        return HotelsItem.builder()
                .offerSearchParams(mapOfferSearchParams(entry))
                .region(mapSettlement(entry.getPointTo().getPointCode()))
                .settlementId(entry.getPointTo().getPointCode())
                .type(mapType(entry.getType()))
                .build();
    }

    private PersonalizationItemType mapType(EntryType type) {
        switch (type) {
            case ENTRY_TYPE_SEARCH:
                return PersonalizationItemType.historical;
            case ENTRY_TYPE_ORDER:
                return PersonalizationItemType.crossSale;
            default:
                return PersonalizationItemType.other;
        }
    }

    private TSettlement getSettlementByPointCode(String settlementPointCode) {
        var settlementId = parseSettlementId(settlementPointCode);
        try {
            return trainSettlementDataProvider.getById(settlementId);
        } catch (NoSuchElementException e) {
            throw new TravelApiBadRequestException(String.format("Unknown point code: %s", settlementPointCode));
        }
    }

    private Region mapSettlement(String settlementPointCode) {
        var regionBuilder = Region.builder();
        var settlement = getSettlementByPointCode(settlementPointCode);
        var linguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, settlement.getGeoId(), locale);
        return regionBuilder.geoId(settlement.getGeoId()).slug(settlement.getSlug()).linguistics(linguistics).build();
    }

    public int parseSettlementId(String pointCode) {
        TravelPreconditions.checkRequestArgument(pointCode.startsWith("c"), "Invalid settlement_id");
        try {
            return Integer.parseInt(pointCode.substring(1));
        } catch (NumberFormatException | StringIndexOutOfBoundsException e) {
            throw new TravelApiBadRequestException("Invalid settlement_id");
        }
    }

    private OfferSearchParams mapOfferSearchParams(THotelEntry entry) {
        var params = new OfferSearchParams();
        params.setAdults((int) entry.getTravelers().getAdults());
        params.setCheckinDate(LocalDate.parse(entry.getCheckInDate()));
        params.setChildrenAges(Collections.emptyList());
        if (!Strings.isNullOrEmpty(entry.getCheckOutDate())) {
            params.setCheckoutDate(LocalDate.parse(entry.getCheckOutDate()));
        }
        return params;
    }
}
