package ru.yandex.travel.api.endpoints.portal_recipes;

import java.util.ArrayList;
import java.util.List;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.portal_recipes.req_rsp.PortalRecipesReqV1;
import ru.yandex.travel.api.endpoints.portal_recipes.req_rsp.PortalRecipesRspV1;
import ru.yandex.travel.api.models.common.Price;
import ru.yandex.travel.api.proto.portal_recipes.PortalRecipesRaw;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.region_images.RegionImagesService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;
import ru.yandex.travel.api.services.portal_recipes.PortalRecipesService;

@Component
@EnableConfigurationProperties(PortalRecipesImplProperties.class)
@Slf4j
@RequiredArgsConstructor
public class PortalRecipesImpl {
    private final static String LANG = "ru";

    private final PortalRecipesImplProperties params;
    private final PortalRecipesService portalRecipesService;
    private final RegionSlugService regionSlugService;
    private final RegionImagesService regionImagesService;
    private final GeoBase geoBase;

    public PortalRecipesRspV1 get(PortalRecipesReqV1 req) {
        var recipesBlock = new PortalRecipesRspV1.RecipesBlock();
        recipesBlock.setDirections(new ArrayList<>());
        List<PortalRecipesRspV1.CrossLinksBlock> crossLinksBlocks = new ArrayList<>();
        var crossLinksBlock = new PortalRecipesRspV1.CrossLinksBlock();
        crossLinksBlock.setDirections(new ArrayList<>());
        switch (req.getType()) {
            case PortalRecipesService.AVIA_TYPE:
                crossLinksBlock.setTitle(params.getAviaCrossLinksTitle());
                break;
            case PortalRecipesService.BUSES_TYPE:
                crossLinksBlock.setTitle(params.getBusesCrossLinksTitle());
                break;
            case PortalRecipesService.HOTELS_TYPE:
                crossLinksBlock.setTitle(params.getHotelsRuCrossLinksTitle());
                break;
            case PortalRecipesService.TRAINS_TYPE:
                crossLinksBlock.setTitle(params.getTrainsCrossLinksTitle());
        }
        crossLinksBlocks.add(crossLinksBlock);
        if (req.getType().equals(PortalRecipesService.HOTELS_TYPE)) {
            var hotelsForeignCrossLinksBlock = new PortalRecipesRspV1.CrossLinksBlock();
            hotelsForeignCrossLinksBlock.setTitle(params.getHotelsForeignCrossLinksTitle());
            hotelsForeignCrossLinksBlock.setDirections(new ArrayList<>());
            crossLinksBlocks.add(hotelsForeignCrossLinksBlock);
        }

        PortalRecipesRspV1 rsp = new PortalRecipesRspV1();
        rsp.setRecipes(recipesBlock);
        rsp.setCrossLinks(crossLinksBlocks);
        for (PortalRecipesRaw raw: portalRecipesService.get(req.getType())) {
            appendRsp(raw, req.getType(), rsp);
        }

        return rsp;
    }

    private void appendRsp(PortalRecipesRaw raw, String type, PortalRecipesRspV1 rsp) {
        PortalRecipesRspV1.DirectionDestination directionFrom = null;
        if (raw.hasFromGeoId()) {
            directionFrom = new PortalRecipesRspV1.DirectionDestination();
            directionFrom.setSlug(raw.getFromSlug());
            directionFrom.setLinguistics(GeoBaseHelpers.getRegionLinguistics(geoBase, raw.getFromGeoId(), LANG));
        }
        var directionTo = new PortalRecipesRspV1.DirectionDestination();
        if (type.equals(PortalRecipesService.HOTELS_TYPE)) {
            directionTo.setSlug(regionSlugService.getSlugByGeoId(raw.getToGeoId()));
        } else {
            directionTo.setSlug(raw.getToSlug());
        }
        directionTo.setLinguistics(GeoBaseHelpers.getRegionLinguistics(geoBase, raw.getToGeoId(), LANG));
        var direction = new PortalRecipesRspV1.Direction();
        direction.setFrom(directionFrom);
        direction.setTo(directionTo);
        direction.setImageUrl(regionImagesService.getImageUsingTree(raw.getToGeoId()));
        var price = new Price(raw.getMinPrice(), Price.Currency.RUB);
        direction.setMinPrice(price);

        if (type.equals(PortalRecipesService.HOTELS_TYPE) && !raw.getHotelsRu()) {
            var directions = rsp.getCrossLinks().get(1).getDirections();
            if (directions.size() < params.getCrossLinksCount()) {
                directions.add(direction);
            }
            return;
        }
        if (rsp.getRecipes().getDirections().size() < params.getRecipesCount()) {
            rsp.getRecipes().getDirections().add(direction);
            return;
        }
        if (rsp.getCrossLinks().get(0).getDirections().size() < params.getCrossLinksCount()) {
            rsp.getCrossLinks().get(0).getDirections().add(direction);
        }
    }
}
