package ru.yandex.travel.api.endpoints.pretrip.eventsblock;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.orders.happy_page.AfishaResponseMapper;
import ru.yandex.travel.api.services.orders.happy_page.IziTravelResponseMapper;
import ru.yandex.travel.api.services.orders.happy_page.afisha.AfishaService;
import ru.yandex.travel.api.services.orders.happy_page.afisha.AfishaServiceProperties;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaImageSize;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.IziTravelProperties;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.IziTravelService;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelCitiesChildrenResponse;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelHelper;
import ru.yandex.travel.api.services.orders.happy_page.model.AfishaCrossSalePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.IziTravelCrossSalePayload;

@RestController
@RequestMapping(value = "/api/events_block")
@RequiredArgsConstructor
@EnableConfigurationProperties({AfishaServiceProperties.class, IziTravelProperties.class})
public class EventsBlockController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final AfishaService afishaService;
    private final AfishaServiceProperties afishaProperties;
    private final AfishaResponseMapper afishaResponseMapper;
    private final IziTravelService iziTravelService;
    private final IziTravelProperties iziTravelProperties;
    private final IziTravelResponseMapper iziTravelResponseMapper;
    private final RegionsService regionsService;

    @RequestMapping(value = "/v1/get_afisha_block", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<AfishaCrossSalePayload> getAfishaBlock(
        @RequestParam(value = "startDate") String startDate,
        @RequestParam(value = "resultsLimit") Integer resultsLimit,
        // rawItemsLimit - запрашиваем заданное число событий, потом отфильтровываем события без картинок
        @RequestParam(value = "rawItemsLimit", defaultValue="0") Integer rawItemsLimit,
        @RequestParam(value = "period") Integer period,
        @RequestParam(value = "geoId") Integer geoId) {
        LocalDate reqDate = LocalDate.parse(startDate, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        return responseProcessor.replyWithFutureRetrying(
                "EventsBlockFlowV1GetAfishaBlock",
                () -> afishaService.getClosestCityInfo(geoId).thenCompose(cityInfo ->
                        afishaService.getActualEvents(
                                cityInfo.getGeoid(),
                                Math.max(resultsLimit, rawItemsLimit),
                                period,
                                reqDate,
                                AfishaImageSize.SMALL
                        ).thenApply(response -> {
                            AfishaCrossSalePayload afishaPayload = new AfishaCrossSalePayload();
                            afishaPayload.setEvents(
                                    afishaResponseMapper.getEventsFromResponse(
                                            afishaProperties.getLinkBaseUrl(),
                                            reqDate.format(DateTimeFormatter.ISO_DATE),
                                            response)
                                            .stream()
                                            .filter(e -> e.getImageUrl() != null && e.getImageUrl().length() > 0)
                                            .limit(resultsLimit)
                                            .collect(Collectors.toUnmodifiableList()));
                            afishaPayload.setRegionUrl(
                                    afishaProperties.getLinkBaseUrl() + cityInfo.getUrl() + "/events");
                            afishaPayload.setCoordinates(cityInfo.getCoordinates());
                            return afishaPayload;
                        })),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/get_izi_travel_block", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<IziTravelCrossSalePayload> getIziTravelBlock(@RequestParam(value = "resultsLimit") Integer resultsLimit,
                                                                       @RequestParam(value = "geoId") Integer geoId) {
        return responseProcessor.replyWithFutureRetrying(
            "EventsBlockFlowV1GetIziTravelBlock",
            () -> {
                UUID cityUuid = getCityUuid(geoId);
                if (cityUuid != null) {
                    List<UUID> toursIds = iziTravelService.getCityChildren(cityUuid, null)
                            .thenApply(response -> response.stream()
                                    .filter(IziTravelHelper::checkCityChildIsAllowed)
                                    .map(IziTravelCitiesChildrenResponse::getUuid)
                                    .collect(Collectors.toList()))
                            .join();
                    if (toursIds.size() > resultsLimit) {
                        toursIds = toursIds.subList(0, resultsLimit);
                    }
                    IziTravelCrossSalePayload iziPayload = new IziTravelCrossSalePayload();
                    iziPayload.setDirectUrl(iziTravelProperties.getLinkBaseUrl() + "/city/" + cityUuid.toString());
                    if (toursIds.size() == 0) {
                        iziPayload.setTours(new ArrayList<>());
                        return CompletableFuture.completedFuture(iziPayload);
                    }
                    return iziTravelService.getBatchOfObjects(toursIds)
                        .thenApply(response -> {
                            iziPayload.setTours(iziTravelResponseMapper.getToursFromResponse(iziTravelProperties.getLinkBaseUrl(), response));
                            return iziPayload;
                        });
                } else {
                    return CompletableFuture.failedFuture(new IziTravelCityNotFoundException("City not found in IZI.TRAVEL"));
                }
            },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private UUID getCityUuid(Integer geoId) {
        try {
            var destRegion = regionsService.getRegion(geoId, "en", "en");
            return iziTravelService.searchCityByNameAndCoords(destRegion.getLinguistics().getNominativeCase(), destRegion.getCoordinates()).join();
        } catch (RuntimeException ignored) {
            return null;
        }
    }

    @ExceptionHandler(IziTravelCityNotFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<String> handleIziTravelCityNotFoundException(IziTravelCityNotFoundException e) {
        return ResponseEntity.status(HttpStatus.NOT_FOUND).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }
}
