package ru.yandex.travel.api.endpoints.pretrip.orderinfoblock;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.pretrip.orderinfoblock.req_rsp.GetOrderInfoReqV1;
import ru.yandex.travel.api.endpoints.pretrip.orderinfoblock.req_rsp.GetOrderInfoRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.orders.happy_page.PretripOrderInfoService;
import ru.yandex.travel.api.services.orders.notifier.model.OrderInfoPayload;

@RestController
@RequestMapping(value = "/api/pretrip_order_info")
@RequiredArgsConstructor
public class OrderInfoBlockController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final PretripOrderInfoService pretripOrderInfoService;

    @RequestMapping(value = "/v1/get_order_info", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Получение информации о заказе для pretrip notifier")
    public DeferredResult<GetOrderInfoRspV1> getOrderInfo(@Validated @BindFromQuery GetOrderInfoReqV1 req) {
        return responseProcessor.replyWithFuture(
                "PretripGetOrderInfo",
                () -> pretripOrderInfoService.getOrderInfo(req.getOrderId())
                        .thenApply(this::mapOrderInfoPayloadToResponse));
    }

    private GetOrderInfoRspV1 mapOrderInfoPayloadToResponse(OrderInfoPayload orderInfoPayload) {
        return GetOrderInfoRspV1.builder().payload(orderInfoPayload).build();
    }
}
