package ru.yandex.travel.api.endpoints.subscriptions;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscribeOnBetterPriceReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscribeReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionPromoConfigReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionPromoConfigRspV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionStatusReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionStatusRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.subscriptions.NotifierSubscriptionService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;

@RestController
@RequestMapping(value = "/api/subscriptions")
@RequiredArgsConstructor
public class SubscriptionsController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final NotifierSubscriptionService notifierSubscriptionService;

    @RequestMapping(value = "/v1/subscribe", method = RequestMethod.POST, produces = "application/json")
    public DeferredResult<Void> subscribe(@RequestBody @Valid SubscribeReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("SubscriptionsSubscribe",
                () -> notifierSubscriptionService.subscribe(request, CommonHttpHeaders.get()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/unsubscribe", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<Void> unsubscribe(@RequestParam("id") String id) {
        return responseProcessor.replyWithFutureRetrying("SubscriptionsUnsubscribe",
                () -> notifierSubscriptionService.unsubscribe(id),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/get_status", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<SubscriptionStatusRspV1> getStatus(@RequestParam("email") String email) {
        var request = SubscriptionStatusReqV1.builder().email(email).build();
        return responseProcessor.replyWithFutureRetrying("SubscriptionsGetStatus",
                () -> notifierSubscriptionService.getStatus(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/get_promo_config", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<SubscriptionPromoConfigRspV1> getPromoConfig(@Validated @BindFromQuery @Valid SubscriptionPromoConfigReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("SubscriptionsGetPromoConfig",
                () -> notifierSubscriptionService.getPromoConfig(request, CommonHttpHeaders.get()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/subscribe_on_better_price", method = RequestMethod.POST, produces = "application/json")
    public DeferredResult<Void> subscribeOnBetterPrice(@RequestBody @Valid SubscribeOnBetterPriceReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("SubscriptionsSubscribeOnBetterPrice",
                () -> notifierSubscriptionService.subscribeOnBetterPrice(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
