package ru.yandex.travel.api.endpoints.takeout;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutDeleteReqV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutDeleteRspV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutResultReqV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutResultRspV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutStartReqV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutStartRspV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutStatusRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.orders.TakeoutOrdersService;

/**
 * Provides API as defined in https://wiki.yandex-team.ru/users/yakushevsky/takeoutrequirements/
 */
@RestController
@RequestMapping(value = {"/api/takeout", "/1/takeout"})
@RequiredArgsConstructor
@Slf4j
public class TakeoutOrdersController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final TakeoutOrdersService takeoutOrdersService;

    @RequestMapping(value = {"/v1/start", "/start"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation("Создание запроса информации о заказах пользователя")
    public DeferredResult<TakeoutStartRspV1> start(@Valid TakeoutStartReqV1 request) {
        return responseProcessor.replyWithFuture("TakeoutStart", () -> takeoutOrdersService.start(request));
    }

    @RequestMapping(value = {"/v1/result", "/result"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation("Получение результата запроса информации о заказах пользователя")
    public DeferredResult<TakeoutResultRspV1> result(@Valid TakeoutResultReqV1 request) {
        return responseProcessor.replyWithFuture("TakeoutResult", () -> takeoutOrdersService.result(request));
    }

    @RequestMapping(value = {"/v1/status", "/status"}, method = RequestMethod.GET, produces = "application/json")
    @ApiOperation("Получение статуса возможности удаления заказов пользователя")
    public DeferredResult<TakeoutStatusRspV1> status(@RequestParam("request_id") String requestId) {
        return responseProcessor.replyWithFuture("TakeoutStatus", () -> takeoutOrdersService.status(requestId));
    }

    @RequestMapping(value = {"/v1/delete", "/delete"}, method = RequestMethod.POST, produces = "application/json")
    @ApiOperation("Запрос на удаление заказов пользователя")
    public DeferredResult<TakeoutDeleteRspV1> delete(@RequestParam("request_id") String requestId,
                                                     @Valid TakeoutDeleteReqV1 request) {
        return responseProcessor.replyWithFuture("TakeoutDelete",
                () -> takeoutOrdersService.delete(requestId, request));
    }
}
