package ru.yandex.travel.api.endpoints.tours_whitelabel;

import java.time.LocalDate;
import java.util.Base64;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalProperties;
import ru.yandex.travel.api.endpoints.tours_whitelabel.req_rsp.BindTourReqV1;
import ru.yandex.travel.api.endpoints.tours_whitelabel.req_rsp.BindTourRspV1;
import ru.yandex.travel.api.endpoints.tours_whitelabel.req_rsp.GetWidgetInfoReqV1;
import ru.yandex.travel.api.endpoints.tours_whitelabel.req_rsp.GetWidgetInfoRspV1;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.services.hotels.promogranter.PromoGranterService;
import ru.yandex.travel.api.services.hotels.redir.HotelsRedirService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.cpa.tours.EEnvironment;
import ru.yandex.travel.cpa.tours.TToursLabel;
import ru.yandex.travel.hotels.proto.promogranter_service.TCreateAdditionalBindingReq;

@Component
@RequiredArgsConstructor
@EnableConfigurationProperties(HotelsPortalProperties.class)
@Slf4j
public class ToursWhitelabelImpl {
    private final HotelsRedirService hotelsRedirService;
    private final PromoGranterService promoGranterService;
    private final HotelsPortalProperties hotelsPortalProperties;

    public CompletableFuture<GetWidgetInfoRspV1> getWidgetInfo(GetWidgetInfoReqV1 req,
                                                               CommonHttpHeaders headers) {
        var reqId = UUID.randomUUID().toString();
        var builder = TToursLabel.newBuilder();

        fillLabelFromAttribution(req, builder);
        fillLabelFromHeaders(headers, builder);
        builder.setLabelGenerationDate(LocalDate.now().toString());
        builder.setEnvironment(hotelsPortalProperties.isProdToursLabelEnvironment() ? EEnvironment.E_Prod : EEnvironment.E_Testing);

        var label = Base64.getUrlEncoder().encodeToString(builder.build().toByteArray());
        return hotelsRedirService.getToursLabelHash(reqId, label).thenApply(labelHash -> {
            var resp = new GetWidgetInfoRspV1();
            resp.setLabel(labelHash);
            return resp;
        });
    }

    private void fillLabelFromAttribution(GetWidgetInfoReqV1 req, TToursLabel.Builder builder) {
        if (req.getUtmSource() != null) {
            builder.setUtmSource(req.getUtmSource());
        }
        if (req.getUtmMedium() != null) {
            builder.setUtmMedium(req.getUtmMedium());
        }
        if (req.getUtmCampaign() != null) {
            builder.setUtmCampaign(req.getUtmCampaign());
        }
        if (req.getUtmTerm() != null) {
            builder.setUtmTerm(req.getUtmTerm());
        }
        if (req.getUtmContent() != null) {
            builder.setUtmContent(req.getUtmContent());
        }
        if (req.getSerpReqId() != null) {
            builder.setSerpReqId(req.getSerpReqId());
        }
        if (req.getGclid() != null) {
            builder.setGclid(req.getGclid());
        }
        if (req.getUserRegion() != null) {
            builder.setUserRegion(req.getUserRegion());
        }
        if (req.getYtpReferer() != null) {
            builder.setYtpReferer(req.getYtpReferer());
        }
        if (req.getYclid() != null) {
            builder.setYclid(req.getYclid());
        }
        if (req.getFbclid() != null) {
            builder.setFBclid(req.getFbclid());
        }
        if (req.getMetrikaClientId() != null) {
            builder.setMetrikaClientId(req.getMetrikaClientId());
        }
        if (req.getMetrikaUserId() != null) {
            builder.setMetrikaUserId(req.getMetrikaUserId());
        }
        if (req.getClid() != null) {
            builder.setClid(req.getClid());
        }
        if (req.getAffiliateClid() != null) {
            builder.setAffiliateClid(req.getAffiliateClid());
        }
        if (req.getAdmitadUid() != null) {
            builder.setAdmitadUid(req.getAdmitadUid());
        }
        if (req.getTravelpayoutsUid() != null) {
            builder.setTravelpayoutsUid(req.getTravelpayoutsUid());
        }
        if (req.getVid() != null) {
            builder.setVid(req.getVid());
        }
        if (req.getAffiliateVid() != null) {
            builder.setAffiliateVid(req.getAffiliateVid());
        }
    }

    private void fillLabelFromHeaders(CommonHttpHeaders headers, TToursLabel.Builder builder) {
        if (headers.getUserDevice() != null) {
            builder.setUserDevice(headers.getUserDevice());
        }
        builder.setIsStaffUser(headers.getUserIsStaffAsBool());
        builder.setIsPlusUser(headers.getUserIsPlusAsBool());
        try {
            if (headers.getIntegerTestIdsFromExpBoxes() != null) {
                builder.addAllIntPortalTestIds(headers.getIntegerTestIdsFromExpBoxes());
            }
            if (headers.getIntegerTestBucketsFromExpBoxes() != null) {
                builder.addAllIntPortalTestBuckets(headers.getIntegerTestBucketsFromExpBoxes());
            }
        } catch (NumberFormatException e) {
            log.warn("Invalid test buckets in request: {}", headers.getExpBoxes(), e);
        }
        if (headers.getRequestId() != null) {
            builder.setYaTravelReqId(headers.getRequestId());
        }
        if (headers.getICookieDecrypted() != null) {
            builder.setICookie(headers.getICookieDecrypted());
        }
        if (headers.getYandexUid() != null) {
            builder.setYandexUid(headers.getYandexUid());
        }
        if (headers.getPassportId() != null) {
            builder.setPassportUid(headers.getPassportId());
        }
        if (headers.getUserIP() != null) {
            builder.setUserIp(headers.getUserIP());
        }
    }

    public CompletableFuture<BindTourRspV1> bindTour(BindTourReqV1 req,
                                                     CommonHttpHeaders headers) {
        TravelPreconditions.checkRequestArgument(headers.getPassportId() != null, "Can bind tour only to logged in user");
        TravelPreconditions.checkRequestArgument(headers.getUserIsPlusAsBool(), "Can bind tour only to plus user");
        var builder = TCreateAdditionalBindingReq.newBuilder()
                .setTourId(req.getTourId())
                .setCheckInDate(req.getCheckinDate().toString())
                .setCheckOutDate(req.getCheckoutDate().toString());
        if (headers.getUserIP() != null) {
            builder.setUserIp(headers.getUserIP());
        }
        if (headers.getPassportId() != null) {
            builder.setPassportId(headers.getPassportId());
        }
        var reqId = UUID.randomUUID().toString();
        return promoGranterService.createAdditionalBinding(builder.build(), reqId)
                .thenApply(protoRsp -> {
                    var rsp = new BindTourRspV1();
                    switch (protoRsp.getCreationStatus()) {
                        case BCS_ORDER_NOT_FOUND:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.ORDER_NOT_FOUND);
                            break;
                        case BCS_ALREADY_BOUND_TO_CURRENT_USER:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.ALREADY_BOUND_TO_CURRENT_USER);
                            break;
                        case BCS_ALREADY_BOUND_TO_OTHER_USER:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.ALREADY_BOUND_TO_OTHER_USER);
                            break;
                        case BCS_NO_LABEL:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.NO_LABEL);
                            break;
                        case BCS_NON_PLUS_ON_PURCHASE:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.NON_PLUS_ON_PURCHASE);
                            break;
                        case BCS_ORDER_IS_TOO_OLD:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.ORDER_IS_TOO_OLD);
                            break;
                        case BCS_CREATED:
                            rsp.setCreationStatus(BindTourRspV1.EBindingCreationStatus.CREATED);
                            break;
                        default:
                            var msg = String.format("Unknown (or error) binding creation status: %s (reqId=%s)", protoRsp.getCreationStatus(), reqId);
                            log.error(msg);
                            throw new RuntimeException(msg);
                    }
                    if (protoRsp.hasExpectedTopupDate()) {
                        rsp.setExpectedTopupDate(LocalDate.parse(protoRsp.getExpectedTopupDate()));
                    }
                    return rsp;
                });
    }
}
