package ru.yandex.travel.api.endpoints.trains_booking_flow;


import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.AddInsuranceReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CalculateRefundAmountReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CalculateRefundAmountRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CancelOrderReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.ChangeRegistrationStatusReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CreateOrderReqV2;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CreateOrderRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.DownloadBlankReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderInfoRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderStatusRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.RefundReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.StartPaymentReqV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.orders.TrainOrdersService;
import ru.yandex.travel.api.services.orders.TrainOrdersServiceProperties;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;

@RestController
@RequestMapping(value = "/api/trains_booking_flow")
@RequiredArgsConstructor
@EnableConfigurationProperties(TrainOrdersServiceProperties.class)
@Slf4j
public class TrainsBookingFlowController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final TrainOrdersService ordersService;
    private final TrainOrdersServiceProperties trainOrdersServiceProperties;

    private final ExperimentDataProvider experimentDataProvider;

    @RequestMapping(value = "/v1/add_insurance", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation("Добавление страховки в существующую бронь")
    public DeferredResult<Void> addInsurance(@RequestBody @Valid AddInsuranceReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowAddInsurance",
                () -> ordersService.addInsurance(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v2/create_order", method = RequestMethod.POST, consumes = "application/json",
            produces = "application/json")
    @ApiOperation("Создание заказа и начало бронирование")
    public DeferredResult<CreateOrderRspV1> createOrder(@RequestBody @Valid CreateOrderReqV2 request) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowCreateOrderV2",
                () -> ordersService.createOrder(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
                null, request.getDeduplicationKey()
        );
    }

    @RequestMapping(value = "/v1/cancel_order", method = RequestMethod.POST, consumes = "application/json")
    @ApiOperation("Запуск пользовательской отмены")
    public DeferredResult<Void> cancelOrder(@RequestBody @Valid CancelOrderReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowCancelOrder",
                () -> ordersService.cancelOrder(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/calculate_refund_amount", method = RequestMethod.POST, consumes = "application/json",
            produces = "application/json")
    @ApiOperation("Получение суммы к возврату")
    public DeferredResult<CalculateRefundAmountRspV1> calculateRefundAmount(@RequestBody @Valid CalculateRefundAmountReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowCalculateRefundAmount",
                () -> ordersService.calculateRefundAmount(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/start_refund", method = RequestMethod.POST, produces = "application/json", consumes =
            "application/json")
    public DeferredResult<Void> startRefund(@RequestBody @Valid RefundReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowStartRefund",
                () -> ordersService.startRefund(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    // TODO(ganintsev): remove this method when TRAVELFRONT-5517 deployed to prod
    @RequestMapping(value = "/v1/download_blank", method = RequestMethod.GET)
    @ApiOperation("Получение pdf билета")
    public DeferredResult<ResponseEntity<byte[]>> downloadBlank(@BindFromQuery DownloadBlankReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowDownloadBlank",
                () -> ordersService.downloadBlank(request).thenApply(b -> ResponseEntity
                        .status(HttpStatus.OK)
                        .contentType(MediaType.APPLICATION_PDF)
                        .body(b)
                ),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
                trainOrdersServiceProperties.getDownloadBlankDuration()
        );
    }

    @RequestMapping(value = "/v1/get_actualized_order_info", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation("Получение обновленной в Экспрессе информации о заказе")
    public DeferredResult<OrderInfoRspV1> getActualizedOrderInfo(@RequestParam("id") String id) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowGetActualizedOrderInfo",
                () -> ordersService.getOrderInfo(id, true),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
                trainOrdersServiceProperties.getGetActualizedOrderInfoDuration()
        );
    }

    @RequestMapping(value = "/v1/get_order_info", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation("Получение информации о заказе")
    public DeferredResult<OrderInfoRspV1> getOrderInfo(@RequestParam("id") String id) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowGetOrderInfo",
                () -> ordersService.getOrderInfo(id, false),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/get_order_status", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation("Получение статуса заказа (используется для поллинга)")
    public DeferredResult<OrderStatusRspV1> getOrderStatus(@RequestParam("id") String id) {
        return responseProcessor.replyWithFutureRetrying("TrainsBookingFlowGetOrderStatus",
                () -> ordersService.getOrderStatus(id),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/start_payment", method = RequestMethod.POST, consumes = "application/json",
            produces = "application/json")
    @ApiOperation("Начало оплаты")
    public DeferredResult<Void> startPayment(@RequestBody @Valid StartPaymentReqV1 request) {
        var newCommonTrustWebForm = experimentDataProvider
                .getInstance(OrderExperiments.class, CommonHttpHeaders.get())
                .isNewCommonPaymentWebForm();
        return responseProcessor.replyWithFutureRetrying(
                "TrainsBookingFlowStartPayment",
                () -> ordersService.payOrder(request.getOrderId(), request.getReturnUrl(),
                        request.getCustomerSource(), newCommonTrustWebForm),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/change_registration_status", method = RequestMethod.POST,
            consumes = "application/json", produces = "application/json")
    @ApiOperation("Смена статуса электронной регистрации")
    public DeferredResult<Void> changeRegistrationStatus(@RequestBody @Valid ChangeRegistrationStatusReqV1 request) {
        return responseProcessor.replyWithFutureRetrying(
                "TrainsBookingFlowChangeRegistrationStatus",
                () -> ordersService.changeRegistrationStatus(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
