package ru.yandex.travel.api.endpoints.trains_search;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.trains_search.exceptions.CanonicalNotFoundException;
import ru.yandex.travel.api.endpoints.trains_search.req_rsp.DirectionReq;
import ru.yandex.travel.api.endpoints.trains_search.req_rsp.DirectionRsp;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.spec.trains.canonical.HandleCanonicalRequest;
import ru.yandex.travel.api.spec.trains.canonical.HandleCanonicalResponse;
import ru.yandex.travel.api.spec.trains.crosslinks.HandleCrossLinksRequest;
import ru.yandex.travel.api.spec.trains.crosslinks.HandleCrossLinksResponse;
import ru.yandex.travel.api.spec.trains.price_calendar.HandlePriceCalendarRequest;
import ru.yandex.travel.api.spec.trains.price_calendar.HandlePriceCalendarResponse;
import ru.yandex.travel.api.spec.trains.search.HandleSearchRequest;
import ru.yandex.travel.api.spec.trains.search.HandleSearchResponse;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;

@RestController
@RequestMapping(path = "/api/trains")
@RequiredArgsConstructor
@Slf4j
public class TrainsSearchController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }


    @ExceptionHandler(CanonicalNotFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<String> handleCanonicalNotFoundException(CanonicalNotFoundException e) {
        return ResponseEntity.status(HttpStatus.NOT_FOUND).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    private final ResponseProcessor responseProcessor;
    private final TrainsSearchImpl trainsSearch;

    @GetMapping(path = "/v1/search", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<HandleSearchResponse> search(HandleSearchRequest.Builder requestBuilder) {
        var request = requestBuilder.build();
        return responseProcessor.replyWithFuture(
                "TrainsSearch",
                () -> trainsSearch.search(request, CommonHttpHeaders.get())
        );
    }

    @GetMapping(path = "/v1/direction", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<DirectionRsp> direction(@Valid DirectionReq request) {
        return responseProcessor.replyWithFuture(
                "TrainsDirection",
                () -> trainsSearch.direction(request, CommonHttpHeaders.get(), UserCredentials.get())
        );
    }

    @GetMapping(path = "/v1/crosslinks", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<HandleCrossLinksResponse> crosslinks(HandleCrossLinksRequest.Builder requestBuilder) {
        var request = requestBuilder.build();
        return responseProcessor.replyWithFuture(
                "TrainsCrossLinks",
                () -> trainsSearch.crosslinks(request)
        );
    }

    @GetMapping(path = "/v1/canonical", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<HandleCanonicalResponse> canonical(HandleCanonicalRequest.Builder requestBuilder) {
        var request = requestBuilder.build();
        return responseProcessor.replyWithFuture(
                "TrainsCanonical",
                () -> trainsSearch.canonical(request)
        );
    }

    @GetMapping(path = "/v1/price-calendar", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<HandlePriceCalendarResponse> priceCalendar(HandlePriceCalendarRequest.Builder requestBuilder) {
        var request = requestBuilder.build();
        return responseProcessor.replyWithFuture(
                "TrainsPriceCalendar",
                () -> trainsSearch.priceCalendar(request, CommonHttpHeaders.get())
        );
    }
}
