package ru.yandex.travel.api.endpoints.trains_search;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.money.CurrencyUnit;

import com.google.common.base.Strings;
import com.google.protobuf.Message;
import com.google.protobuf.util.Timestamps;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.trains_search.exceptions.CanonicalNotFoundException;
import ru.yandex.travel.api.endpoints.trains_search.req_rsp.DirectionReq;
import ru.yandex.travel.api.endpoints.trains_search.req_rsp.DirectionRsp;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlock;
import ru.yandex.travel.api.services.crosslinks.CrosslinksService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;
import ru.yandex.travel.api.services.hotels.static_pages.RegionPagesStorage;
import ru.yandex.travel.api.services.train.TrainSeoConverter;
import ru.yandex.travel.api.services.train.canonical.TrainCanonicalIndex;
import ru.yandex.travel.api.services.train.canonical.TrainCanonicalItem;
import ru.yandex.travel.api.services.train.crosslinks.TrainCrossLinkItem;
import ru.yandex.travel.api.services.train.crosslinks.TrainCrossLinksIndex;
import ru.yandex.travel.api.spec.Currency;
import ru.yandex.travel.api.spec.MoneyAmount;
import ru.yandex.travel.api.spec.trains.canonical.HandleCanonicalRequest;
import ru.yandex.travel.api.spec.trains.canonical.HandleCanonicalResponse;
import ru.yandex.travel.api.spec.trains.crosslinks.CrossLinkItem;
import ru.yandex.travel.api.spec.trains.crosslinks.HandleCrossLinksRequest;
import ru.yandex.travel.api.spec.trains.crosslinks.HandleCrossLinksResponse;
import ru.yandex.travel.api.spec.trains.price_calendar.DayPrice;
import ru.yandex.travel.api.spec.trains.price_calendar.DirectionPrices;
import ru.yandex.travel.api.spec.trains.price_calendar.EmptyPriceReason;
import ru.yandex.travel.api.spec.trains.price_calendar.HandlePriceCalendarRequest;
import ru.yandex.travel.api.spec.trains.price_calendar.HandlePriceCalendarResponse;
import ru.yandex.travel.api.spec.trains.search.BrokenClassesCode;
import ru.yandex.travel.api.spec.trains.search.Company;
import ru.yandex.travel.api.spec.trains.search.Country;
import ru.yandex.travel.api.spec.trains.search.Features;
import ru.yandex.travel.api.spec.trains.search.FeaturesDynamicPricing;
import ru.yandex.travel.api.spec.trains.search.FeaturesETicket;
import ru.yandex.travel.api.spec.trains.search.FeaturesNamedTrain;
import ru.yandex.travel.api.spec.trains.search.FeaturesSubSegments;
import ru.yandex.travel.api.spec.trains.search.FeaturesSubSegmentsArrival;
import ru.yandex.travel.api.spec.trains.search.FeaturesSubtype;
import ru.yandex.travel.api.spec.trains.search.FeaturesThroughTrain;
import ru.yandex.travel.api.spec.trains.search.HandleSearchRequest;
import ru.yandex.travel.api.spec.trains.search.HandleSearchResponse;
import ru.yandex.travel.api.spec.trains.search.MinTariffs;
import ru.yandex.travel.api.spec.trains.search.MinTariffsClass;
import ru.yandex.travel.api.spec.trains.search.MinTariffsClasses;
import ru.yandex.travel.api.spec.trains.search.NearestDate;
import ru.yandex.travel.api.spec.trains.search.NearestDates;
import ru.yandex.travel.api.spec.trains.search.NearestDatesDirection;
import ru.yandex.travel.api.spec.trains.search.NearestDatesReason;
import ru.yandex.travel.api.spec.trains.search.PlaceDetails;
import ru.yandex.travel.api.spec.trains.search.PlacesDetails;
import ru.yandex.travel.api.spec.trains.search.ResponseStatus;
import ru.yandex.travel.api.spec.trains.search.SearchContext;
import ru.yandex.travel.api.spec.trains.search.SearchContextPoint;
import ru.yandex.travel.api.spec.trains.search.SearchContextPoints;
import ru.yandex.travel.api.spec.trains.search.Segment;
import ru.yandex.travel.api.spec.trains.search.Settlement;
import ru.yandex.travel.api.spec.trains.search.Station;
import ru.yandex.travel.api.spec.trains.search.Train;
import ru.yandex.travel.api.spec.trains.search.Variant;
import ru.yandex.travel.api.spec.trains.search.VariantUrl;
import ru.yandex.travel.api.spec.trains.search.VariantUrlOwner;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.proto.region_pages.TRegionPage;
import ru.yandex.travel.trains.proto.UserArgs;
import ru.yandex.travel.trains.search_api.api.Request;
import ru.yandex.travel.trains.search_api.api.TestContext;
import ru.yandex.travel.trains.search_api.api.price_calendar.PriceCalendarRequest;
import ru.yandex.travel.trains.search_api.api.seo_direction.SeoDirectionRequest;
import ru.yandex.travel.trains.search_api.api.seo_direction.SeoDirectionResponse;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrainsSearchImpl {
    private final CrosslinksService crosslinksService;
    private final RegionSlugService regionSlugService;
    private final RegionPagesStorage regionPagesStorage;
    private final ru.yandex.travel.api.services.train.search.TrainSearchService service;
    private final TrainSeoConverter seoConverter;

    private final TrainCrossLinksIndex index;

    private final TrainCanonicalIndex indexCanonical;

    private static final Map<CurrencyUnit, Currency.Enum> CURRENCY_MAP =
            Map.of(
                    ProtoCurrencyUnit.RUB, Currency.Enum.RUB,
                    ProtoCurrencyUnit.USD, Currency.Enum.USD,
                    ProtoCurrencyUnit.EUR, Currency.Enum.EUR
            );

    private static final Map<ru.yandex.travel.trains.search_api.api.ResponseStatus, ResponseStatus.Enum> STATUS_MAP =
            Map.of(
                    ru.yandex.travel.trains.search_api.api.ResponseStatus.STATUS_DONE, ResponseStatus.Enum.DONE,
                    ru.yandex.travel.trains.search_api.api.ResponseStatus.STATUS_QUERYING, ResponseStatus.Enum.QUERYING
            );

    private static final Map<ru.yandex.travel.trains.search_api.api.price_calendar.EmptyPriceReason, EmptyPriceReason.Enum> EMPTY_PRICE_REASON_MAP = Map.of(
            ru.yandex.travel.trains.search_api.api.price_calendar.EmptyPriceReason.EMPTY_PRICE_REASON_SOLD_OUT, EmptyPriceReason.Enum.SOLD_OUT,
            ru.yandex.travel.trains.search_api.api.price_calendar.EmptyPriceReason.EMPTY_PRICE_REASON_OTHER, EmptyPriceReason.Enum.OTHER,
            ru.yandex.travel.trains.search_api.api.price_calendar.EmptyPriceReason.EMPTY_PRICE_REASON_NO_DIRECT_TRAINS, EmptyPriceReason.Enum.NO_DIRECT_TRAINS
    );

    private ResponseStatus.Enum convertStatus(ru.yandex.travel.trains.search_api.api.ResponseStatus status) {
        var mapped = STATUS_MAP.get(status);
        if (mapped == null) {
            log.error("Cannot map ResponseStatus {}", status);
            return ResponseStatus.Enum.INVALID;
        }
        return mapped;
    }

    private NearestDate convertNearestDatesDate(@NotNull ru.yandex.travel.trains.search_api.api.NearestDate nearestDate) {
        return NearestDate.newBuilder().setDate(nearestDate.getDate()).build();
    }

    private NearestDatesDirection convertNearestDaysDirection(
            @NotNull ru.yandex.travel.trains.search_api.api.NearestDatesDirection nearestDatesDirection
    ) {
        return NearestDatesDirection.newBuilder()
                .setReason(NearestDatesReason.forNumber(nearestDatesDirection.getReasonValue()))
                .addAllDates(nearestDatesDirection.getDatesList().stream().map(this::convertNearestDatesDate)::iterator)
                .build();
    }

    private NearestDates convertNearestDays(@NotNull ru.yandex.travel.trains.search_api.api.NearestDates nearestDates) {
        var builder = NearestDates.newBuilder();
        if (nearestDates.hasForward()) {
            builder.setForward(convertNearestDaysDirection(nearestDates.getForward()));
        }
        if (nearestDates.hasBackward()) {
            builder.setBackward(convertNearestDaysDirection(nearestDates.getBackward()));
        }
        return builder.build();
    }

    public CompletableFuture<HandleSearchResponse> search(@NotNull HandleSearchRequest request,
                                                          CommonHttpHeaders headers) {
        UserArgs.Builder userArgs = createUserArgs(request.getIsBot(), headers);
        var testContext = TestContext.newBuilder()
                .setMockImAuto(request.getMockImAuto());
        if (!Strings.isNullOrEmpty(request.getMockImPath())) {
            testContext.setMockImPath(request.getMockImPath());
        }
        return service.search(
                Request.newBuilder()
                        .setPointFrom(request.getPointFrom())
                        .setPointTo(request.getPointTo())
                        .setWhen(request.getWhen())
                        .setReturnWhen(request.getReturnWhen())
                        .setPinForwardSegmentId(request.getPinForwardSegmentId())
                        .setPinBackwardSegmentId(request.getPinBackwardSegmentId())
                        .setIsBot(request.getIsBot())
                        .setUserArgs(userArgs.build())
                        .setOnlyDirect(request.getOnlyDirect())
                        .setOnlyOwnedPrices(request.getOnlyOwnedPrices())
                        .setTestContext(testContext.build())
                        .build()
        ).thenApply(
                serviceRsp -> {
                    var builder = HandleSearchResponse.newBuilder()
                            .setStatus(convertStatus(serviceRsp.getStatus()))
                            .addAllVariants(serviceRsp.getVariantsList().stream().map(this::convertVariant)::iterator)
                            .addAllActivePartners(serviceRsp.getActivePartnersList());

                    var searchContext = serviceRsp.getSearchContext();
                    if (ProtoUtils.hasFields(searchContext)) {
                        builder.setContext(convertSearchContext(searchContext));
                    }

                    var nearestDates = serviceRsp.getNearestDates();
                    if (ProtoUtils.hasFields(nearestDates)) {
                        builder.setNearestTrainDatesByDirection(convertNearestDays(nearestDates));
                    }

                    return builder.build();
                }
        );
    }

    public CompletableFuture<DirectionRsp> direction(DirectionReq request,
                                                     CommonHttpHeaders commonHttpHeaders,
                                                     UserCredentials userCredentials) {
        var trainSearchFuture = service.direction(
                SeoDirectionRequest.newBuilder()
                        .setFromSlug(request.getFromSlug())
                        .setToSlug(request.getToSlug())
                        .build()
        );

        var hotelsFuture = prepareHotelsRequest(request, commonHttpHeaders, userCredentials);

        return CompletableFuture.allOf(trainSearchFuture, hotelsFuture).thenApply(ignored -> {
            SeoDirectionResponse seoDirectionRsp = trainSearchFuture.join();

            CrosslinksHotelsBlock hotelsBlock;
            try {
                hotelsBlock = hotelsFuture.join();
            } catch (Exception exc) {
                log.error("Unable to complete hotel crosslinks request", exc);
                hotelsBlock = null;
            }

            return seoConverter.convertSeoDirection(seoDirectionRsp, hotelsBlock);
        });
    }

    private CompletableFuture<CrosslinksHotelsBlock> prepareHotelsRequest(DirectionReq request,
                                                                          CommonHttpHeaders commonHttpHeaders,
                                                                          UserCredentials userCredentials) {
        int geoId;
        try {
            geoId = regionSlugService.getGeoIdBySlug(request.getToSlug());
        } catch (Exception exc) {
            log.error("Unable to get geoId by slug {}. Request to hotel crosslinks cancelled.", request.getToSlug(), exc);
            return CompletableFuture.completedFuture(null);
        }

        Optional<TRegionPage> result = regionPagesStorage.tryGetRegionPage(geoId);
        if (!result.isPresent()) {
            return CompletableFuture.completedFuture(null);
        }

        return crosslinksService.getHotelsBlock(geoId, request.getDomain(), commonHttpHeaders, userCredentials);
    }

    public CompletableFuture<HandleCrossLinksResponse> crosslinks(HandleCrossLinksRequest request) {
        List<TrainCrossLinkItem> items = index.getCrossLinksByFromId(request.getFromKey(), request.getToKey());
        List<CrossLinkItem> crossLinksList = new ArrayList<>();

        if (items != null) {
            for (TrainCrossLinkItem item : items) {
                crossLinksList.add(CrossLinkItem.newBuilder()
                        .setFromTitleRuNominative(item.getFromTitleNominative())
                        .setToTitleRuNominative(item.getToTitleNominative())
                        .setFromKey(item.getFromKey())
                        .setToKey(item.getToKey())
                        .setFromSlug(item.getFromSlug())
                        .setToSlug(item.getToSlug())
                        .build()
                );
            }
        }

        return CompletableFuture.supplyAsync(() -> HandleCrossLinksResponse.newBuilder().addAllCrossLinks(crossLinksList).build());
    }

    public CompletableFuture<HandleCanonicalResponse> canonical(HandleCanonicalRequest request) {
        TrainCanonicalItem item = indexCanonical.getCanonicalBySlug(request.getFromSlug(), request.getToSlug());

        if (item == null) {
            throw new CanonicalNotFoundException("Canonical not found");
        }

        return CompletableFuture.supplyAsync(() -> HandleCanonicalResponse.newBuilder()
                .setFromSlug(item.getFromSlug())
                .setToSlug(item.getToSlug())
                .setFromTitle(item.getFromTitle())
                .setToTitle(item.getToTitle())
                .setFromPopularTitle(item.getFromPopularTitle())
                .setToPopularTitle(item.getToPopularTitle())
                .build());
    }

    public CompletableFuture<HandlePriceCalendarResponse> priceCalendar(HandlePriceCalendarRequest request,
                                                                        CommonHttpHeaders headers) {
        UserArgs.Builder userArgs = createUserArgs(request.getIsBot(), headers);
        return service.priceCalendar(PriceCalendarRequest.newBuilder()
                        .setPointFrom(request.getPointFrom())
                        .setPointTo(request.getPointTo())
                        .setUserArgs(userArgs)
                        .build())
                .thenApply(rsp -> HandlePriceCalendarResponse.newBuilder()
                        .setForward(convertDirectionPrices(rsp.getForward()))
                        .setBackward(convertDirectionPrices(rsp.getBackward()))
                        .build());
    }

    private DirectionPrices convertDirectionPrices(ru.yandex.travel.trains.search_api.api.price_calendar.DirectionPrices source) {
        return DirectionPrices.newBuilder()
                .addAllDates(source.getDatesList().stream().map(this::convertDayPrice).collect(Collectors.toList()))
                .build();
    }

    private DayPrice convertDayPrice(ru.yandex.travel.trains.search_api.api.price_calendar.DayPrice d) {
        DayPrice.Builder result = DayPrice.newBuilder()
                .setDate(d.getDate());
        if (d.hasPrice()) {
            result.setPrice(convertPrice(d.getPrice()));
        } else if (d.getEmptyPriceReason() != ru.yandex.travel.trains.search_api.api.price_calendar.EmptyPriceReason.EMPTY_PRICE_REASON_INVALID) {
            result.setEmptyPriceReason(EMPTY_PRICE_REASON_MAP.get(d.getEmptyPriceReason()));
        }
        return result.build();
    }

    private UserArgs.Builder createUserArgs(boolean isBot, CommonHttpHeaders headers) {
        UserArgs.Builder userArgs = UserArgs.newBuilder()
                .setIsBot(isBot);
        if (!Strings.isNullOrEmpty(headers.getICookie())) {
            userArgs.setIcookie(headers.getICookie());
        }
        if (!Strings.isNullOrEmpty(headers.getUserDevice())) {
            userArgs.setUserDevice(headers.getUserDevice());
        }
        if (!Strings.isNullOrEmpty(headers.getYandexUid())) {
            userArgs.setYandexUid(headers.getYandexUid());
        }
        if (!Strings.isNullOrEmpty(headers.getRawExperiments())) {
            userArgs.setUaasExperiments(headers.getRawExperiments());
        }
        return userArgs;
    }

    @NotNull
    private VariantUrl convertOrderUrl(@NotNull ru.yandex.travel.trains.search_api.api.OrderUrl orderUrl) {
        var builder = VariantUrl.newBuilder();

        var owner = convertUrlOwner(orderUrl.getOwner());
        if (owner != VariantUrlOwner.Enum.TRAINS) {
            builder.setUrl(orderUrl.getUrl());
        }
        builder.setOwner(owner);

        return builder.build();
    }

    private static final Map<ru.yandex.travel.trains.search_api.api.OrderOwner, VariantUrlOwner.Enum> URL_OWNER_MAP =
            Map.of(
                    ru.yandex.travel.trains.search_api.api.OrderOwner.ORDER_OWNER_TRAINS, VariantUrlOwner.Enum.TRAINS,
                    ru.yandex.travel.trains.search_api.api.OrderOwner.ORDER_OWNER_UFS, VariantUrlOwner.Enum.UFS
            );

    private VariantUrlOwner.Enum convertUrlOwner(ru.yandex.travel.trains.search_api.api.OrderOwner owner) {
        var mapped = URL_OWNER_MAP.get(owner);
        if (mapped == null) {
            log.error("Cannot map VariantUrlOwner {}", owner);
            return VariantUrlOwner.Enum.INVALID;
        }
        return mapped;
    }

    @NotNull
    private Variant convertVariant(@NotNull ru.yandex.travel.trains.search_api.api.Variant variant) {
        var builder = Variant.newBuilder()
                .setId(variant.getId())
                .addAllForward(
                        variant.getForwardList().stream().map(this::convertSegment)::iterator
                )
                .addAllBackward(
                        variant.getBackwardList().stream().map(this::convertSegment)::iterator
                );

        var orderUrl = variant.getOrderUrl();
        if (ProtoUtils.hasFields(orderUrl)) {
            builder.setOrderUrl(convertOrderUrl(orderUrl));
        }

        return builder.build();
    }

    @NotNull
    private Segment convertSegment(@NotNull ru.yandex.travel.trains.search_api.api.Segment segment) {
        var departure = segment.getDeparture();
        var arrival = segment.getArrival();
        var builder = Segment.newBuilder()
                .setId(segment.getId())
                .setDeparture(ProtoUtils.toInstant(departure).toString())
                .setArrival(ProtoUtils.toInstant(arrival).toString())
                .setDuration((int) Timestamps.between(departure, arrival).getSeconds())
                .setStationFrom(convertStation(segment.getStationFrom()))
                .setStationTo(convertStation(segment.getStationTo()))
                .setFeatures(convertFeatures(segment.getFeatures()))
                .setTariffs(convertTariffs(segment.getTariffs()));

        var company = segment.getCompany();
        if (ProtoUtils.hasFields(company)) {
            builder.setCompany(Company.newBuilder()
                    .setId(company.getId())
                    .setTitle(company.getTitle()));
        }
        var train = segment.getTrain();
        builder.setTrain(Train.newBuilder()
                .setTitle(train.getTitle())
                .setDisplayNumber(train.getDisplayNumber())
                .setNumber(train.getNumber()));
        var provider = segment.getProvider();
        if (!provider.isEmpty()) {
            builder.setProvider(provider);
        }

        return builder.build();
    }

    @NotNull
    private Features convertFeatures(@NotNull ru.yandex.travel.trains.search_api.api.Features features) {
        var builder = Features.newBuilder();

        if (features.getEticket()) {
            builder.setETicket(FeaturesETicket.newBuilder()
                    .setType("eTicket"));
        }
        if (features.getDynamicPricing()) {
            builder.setDynamicPricing(FeaturesDynamicPricing.newBuilder()
                    .setType("dynamicPricing"));
        }
        var subSegments = features.getSubsegments();
        if (features.getThroughTrain()) {
            builder.setThroughTrain(FeaturesThroughTrain.newBuilder()
                    .setType("throughTrain")
                    .setArrival(FeaturesSubSegmentsArrival.newBuilder()
                            .setMin(subSegments.getArrival().getMin())
                            .setMax(subSegments.getArrival().getMax())));
        }
        if (ProtoUtils.hasFields(subSegments)) {
            builder.setSubSegments(FeaturesSubSegments.newBuilder()
                    .setType("subSegments")
                    .setArrival(FeaturesSubSegmentsArrival.newBuilder()
                            .setMin(subSegments.getArrival().getMin())
                            .setMax(subSegments.getArrival().getMax())));
        }
        var namedTrain = features.getNamedTrain();
        if (ProtoUtils.hasFields(namedTrain)) {
            builder.setNamedTrain(FeaturesNamedTrain.newBuilder()
                    .setType("namedTrain")
                    .setId(namedTrain.getId())
                    .setTitle(namedTrain.getTitle())
                    .setIsDeluxe(namedTrain.getIsDeluxe())
                    .setIsHighSpeed(namedTrain.getIsHighSpeed()));
        }
        var subtype = features.getSubtype();
        if (ProtoUtils.hasFields(subtype)) {
            builder.setSubtype(FeaturesSubtype.newBuilder()
                    .setType("subtype")
                    .setId(subtype.getId())
                    .setTitle(subtype.getTitle()));
        }

        return builder.build();
    }

    private static final Map<ru.yandex.travel.trains.search_api.api.BrokenClassesCode, BrokenClassesCode.Enum> BROKEN_CLASSES_CODE_MAP =
            Map.of(
                    ru.yandex.travel.trains.search_api.api.BrokenClassesCode.BROKEN_CLASSES_CODE_SOLD_OUT,
                    BrokenClassesCode.Enum.SOLD_OUT,
                    ru.yandex.travel.trains.search_api.api.BrokenClassesCode.BROKEN_CLASSES_CODE_OTHER,
                    BrokenClassesCode.Enum.OTHER
            );

    private BrokenClassesCode.Enum convertBrokenClassesCode(ru.yandex.travel.trains.search_api.api.BrokenClassesCode code) {
        var mapped = BROKEN_CLASSES_CODE_MAP.get(code);
        if (mapped == null) {
            log.error("Cannot map BrokenClassesCode {}", code);
            return BrokenClassesCode.Enum.INVALID;
        }
        return mapped;
    }

    @NotNull
    private MinTariffsClass convertMinTariffClass(String type,
                                                  @NotNull ru.yandex.travel.trains.search_api.api.MinTariffsClass tariffsClass) {
        var result = MinTariffsClass.newBuilder()
                .setType(type)
                .setPrice(convertPrice(tariffsClass.getPrice()))
                .setSeats(tariffsClass.getSeats())
                .setHasNonRefundableTariff(tariffsClass.getHasNonRefundableTariff());
        if (tariffsClass.hasPlacesDetails()) {
            var placesDetails = PlacesDetails.newBuilder();
            if (tariffsClass.getPlacesDetails().hasLower()) {
                placesDetails.setLower(convertPlaceDetails(tariffsClass.getPlacesDetails().getLower()));
            }
            if (tariffsClass.getPlacesDetails().hasUpper()) {
                placesDetails.setUpper(convertPlaceDetails(tariffsClass.getPlacesDetails().getUpper()));
            }
            if (tariffsClass.getPlacesDetails().hasLowerSide()) {
                placesDetails.setLowerSide(convertPlaceDetails(tariffsClass.getPlacesDetails().getLowerSide()));
            }
            if (tariffsClass.getPlacesDetails().hasUpperSide()) {
                placesDetails.setUpperSide(convertPlaceDetails(tariffsClass.getPlacesDetails().getUpperSide()));
            }
            result.setPlacesDetails(placesDetails.build());
        }
        return result.build();
    }

    @NotNull
    private PlaceDetails convertPlaceDetails(@NotNull ru.yandex.travel.trains.search_api.api.PlaceDetails placeDetails) {
        return PlaceDetails.newBuilder()
                .setQuantity(placeDetails.getQuantity())
                .build();
    }

    @NotNull
    private MinTariffs convertTariffs(@NotNull ru.yandex.travel.trains.search_api.api.MinTariffs tariffs) {
        var builder = MinTariffs.newBuilder();
        var classes = tariffs.getClasses();
        var classesBuilder = MinTariffsClasses.newBuilder();
        ru.yandex.travel.trains.search_api.api.MinTariffsClass tariffsClass;

        tariffsClass = classes.getPlatzkarte();
        if (ProtoUtils.hasFields(tariffsClass)) {
            classesBuilder.setPlatzkarte(convertMinTariffClass("platzkarte", tariffsClass));
        }
        tariffsClass = classes.getCompartment();
        if (ProtoUtils.hasFields(tariffsClass)) {
            classesBuilder.setCompartment(convertMinTariffClass("compartment", tariffsClass));
        }
        tariffsClass = classes.getSuite();
        if (ProtoUtils.hasFields(tariffsClass)) {
            classesBuilder.setSuite(convertMinTariffClass("suite", tariffsClass));
        }
        tariffsClass = classes.getCommon();
        if (ProtoUtils.hasFields(tariffsClass)) {
            classesBuilder.setCommon(convertMinTariffClass("common", tariffsClass));
        }
        tariffsClass = classes.getSitting();
        if (ProtoUtils.hasFields(tariffsClass)) {
            classesBuilder.setSitting(convertMinTariffClass("sitting", tariffsClass));
        }
        tariffsClass = classes.getSoft();
        if (ProtoUtils.hasFields(tariffsClass)) {
            classesBuilder.setSoft(convertMinTariffClass("soft", tariffsClass));
        }

        builder.setClasses(classesBuilder);

        // TODO: replace hasField with `getOptionalFooCase().getNumber() == 0` to enable static checking
        if (hasField(tariffs, "broken_classes_code")) {
            builder.setBrokenClassesCode(convertBrokenClassesCode(tariffs.getBrokenClassesCode()));
        }

        return builder.build();
    }

    @NotNull
    private MoneyAmount convertPrice(TPrice price) {
        var money = ProtoUtils.fromTPrice(price);

        var mappedCurrency = CURRENCY_MAP.get(money.getCurrency());
        if (mappedCurrency == null) {
            log.error("Cannot map Currency {}", price.getCurrency());
            mappedCurrency = Currency.Enum.INVALID;
        }

        return MoneyAmount.newBuilder()
                .setValue(money.getNumber().floatValue())
                .setCurrency(mappedCurrency)
                .build();
    }

    @NotNull
    private Station convertStation(@NotNull ru.yandex.travel.trains.search_api.api.Station station) {
        var builder = Station.newBuilder()
                .setId(station.getId())
                .setTitle(station.getTitle())
                .setTimezone(station.getTimezone())
                .setRailwayTimezone(station.getRailwayTimezone());

        var country = station.getCountry();
        if (ProtoUtils.hasFields(country)) {
            builder.setCountry(Country.newBuilder()
                    .setId(country.getId())
                    .setCode(country.getCode()));
        }
        var settlement = station.getSettlement();
        if (ProtoUtils.hasFields(settlement)) {
            builder.setSettlement(Settlement.newBuilder()
                    .setId(settlement.getId())
                    .setPreposition(settlement.getPreposition())
                    .setTitle(settlement.getTitle())
                    .setTitleAccusative(settlement.getTitleAccusative())
                    .setTitleGenitive(settlement.getTitleGenitive())
                    .setTitleLocative(settlement.getTitleLocative()));
        }

        var platform = station.getPlatform();
        if (!platform.isEmpty()) {
            builder.setPlatform(platform);
        }

        return builder.build();
    }

    @NotNull
    private SearchContext convertSearchContext(@NotNull ru.yandex.travel.trains.search_api.api.SearchContext context) {
        return SearchContext.newBuilder()
                .setIsChanged(context.getIsChanged())
                .setOriginal(convertSearchContextPoints(context.getOriginal()))
                .setSearch(convertSearchContextPoints(context.getSearch()))
                .addAllTransportTypes(context.getTransportTypesList())
                .setLatestDatetime(ProtoUtils.toInstant(context.getLatestDatetime()).toString())
                .build();
    }

    @NotNull
    private SearchContextPoints convertSearchContextPoints(@NotNull ru.yandex.travel.trains.search_api.api.SearchContextPoints points) {
        return SearchContextPoints.newBuilder()
                .setNearest(points.getNearest())
                .setPointFrom(convertSearchContextPoint(points.getPointFrom()))
                .setPointTo(convertSearchContextPoint(points.getPointTo()))
                .build();
    }

    @NotNull
    private SearchContextPoint convertSearchContextPoint(@NotNull ru.yandex.travel.trains.search_api.api.SearchContextPoint point) {
        var builder = SearchContextPoint.newBuilder()
                .setKey(point.getKey())
                .setTitle(point.getTitle());

        // TODO: replace hasField with `getOptionalFooCase().getNumber() == 0` to enable static checking

        if (hasField(point, "title_with_type")) {
            builder.setTitleWithType(point.getTitleWithType());
        }
        if (hasField(point, "title_genitive")) {
            builder.setTitleGenitive(point.getTitleGenitive());
        }
        if (hasField(point, "title_accusative")) {
            builder.setTitleAccusative(point.getTitleAccusative());
        }
        if (hasField(point, "title_locative")) {
            builder.setTitleLocative(point.getTitleLocative());
        }
        if (hasField(point, "preposition")) {
            builder.setPreposition(point.getPreposition());
        }
        if (hasField(point, "popular_title")) {
            builder.setPopularTitle(point.getPopularTitle());
        }
        if (hasField(point, "short_title")) {
            builder.setShortTitle(point.getShortTitle());
        }
        return builder.build();
    }

    private <T extends Message> boolean hasField(@NotNull T message, @NotNull String fieldName) {
        var fieldDescriptor = Objects.requireNonNull(message.getDescriptorForType().findFieldByName(fieldName));
        return message.hasField(fieldDescriptor);
    }
}
