package ru.yandex.travel.api.endpoints.travel_orders;

import java.util.Set;
import java.util.UUID;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.booking_flow.model.AuthorizeOrderRequest;
import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.OrderAuthorizationRspV1;
import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.OrderHappyPageRspV1;
import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.TravelOrdersListRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.orders.OrderType;
import ru.yandex.travel.api.services.orders.authorization.AuthorizationService;
import ru.yandex.travel.api.services.orders.happy_page.HappyPageService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;

@RestController
@RequestMapping(value = "/api/orders")
@RequiredArgsConstructor
public class TravelOrdersController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final TravelOrdersImplV2 travelOrdersImplV2;
    private final AuthorizationService authorizationService;
    private final HappyPageService happyPageService;

    @RequestMapping(value = "/v2/list_orders", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "", response = TravelOrdersListRspV1.class)
    public DeferredResult<TravelOrdersListRspV1> listOrdersFirstPageV2(
            @RequestParam(name = "pageSize", defaultValue = "0", required = false) int pageSize,
            @RequestParam(name = "order_type", required = false) OrderType[] orderTypes,
            @RequestParam(name = "order_status", required = false) DisplayableOrderStatus status,
            @RequestParam(name = "search_term", required = false) String searchTerm
    ) {
        Set<OrderType> orderTypeSet;
        if (orderTypes != null) {
            orderTypeSet = Set.of(orderTypes);
        } else {
            orderTypeSet = Set.of();
        }
        return responseProcessor.replyWithFutureRetrying(
                "TravelOrdersListOrdersFirstPageV2",
                () -> travelOrdersImplV2.listOrdersFirstPage(pageSize, searchTerm, orderTypeSet, status),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    @RequestMapping(value = "/v2/list_orders_next_page", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "", response = TravelOrdersListRspV1.class)
    public DeferredResult<TravelOrdersListRspV1> listOrdersNextPageV2(
            @RequestParam(name = "nextPageToken") String nextPageToken) {
        return responseProcessor.replyWithFutureRetrying("TravelOrdersListOrdersNextPageV2",
                () -> travelOrdersImplV2.listNextPage(nextPageToken),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    @RequestMapping(value = "/v1/check_authorization", method = RequestMethod.GET, produces = "application/json")
    @SuppressWarnings("Duplicates")
    public DeferredResult<ResponseEntity<OrderAuthorizationRspV1>> checkAuthorization(
            @RequestParam(value = "id", required = false) String orderId,
            @RequestParam(value = "yandex_order_id", required = false) String prettyId) {
        DeferredResult<ResponseEntity<OrderAuthorizationRspV1>> result = new DeferredResult<>();
        authorizationService.checkAuthorization(orderId, prettyId)
                .thenAccept(r -> {
                            if (r == null) {
                                result.setResult(ResponseEntity.notFound().build());
                            } else {
                                if (r.getAuthorized()) {
                                    result.setResult(ResponseEntity.ok(r));
                                } else {
                                    result.setResult(ResponseEntity.status(HttpStatus.FORBIDDEN).body(r));
                                }
                            }
                        }
                )
                .exceptionally(e -> {
                    result.setErrorResult(e);
                    return null;
                });
        return result;
    }

    @RequestMapping(value = "/v1/request_authorization", method = RequestMethod.POST, produces = "application/json",
            consumes = "application/json")
    @SuppressWarnings("Duplicates")
    public DeferredResult<ResponseEntity<OrderAuthorizationRspV1>> requestAuthorization(
            @RequestParam(value = "id", required = false) String orderId,
            @RequestParam(value = "yandex_order_id", required = false) String prettyId,
            @RequestBody @Valid AuthorizeOrderRequest request) {
        DeferredResult<ResponseEntity<OrderAuthorizationRspV1>> result = new DeferredResult<>();
        authorizationService.requestAuthorization(orderId, prettyId, request.getSecret())
                .thenAccept(r -> {
                            if (r == null) {
                                result.setResult(ResponseEntity.notFound().build());
                            } else {
                                if (r.getAuthorized()) {
                                    result.setResult(ResponseEntity.ok(r));
                                } else {
                                    result.setResult(ResponseEntity.status(HttpStatus.FORBIDDEN).body(r));
                                }
                            }
                        }
                )
                .exceptionally(e -> {
                    result.setErrorResult(e);
                    return null;
                });
        return result;
    }

    @RequestMapping(value = "/v1/get_order_happy_page", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<OrderHappyPageRspV1> getOrderHappyPage(@RequestParam(value = "orderId") UUID orderId,
                                                                 @RequestParam(value = "geoId") String geoId) {
        return responseProcessor.replyWithFutureRetrying(
                "BookingFlowV1GetOrderHappyPage",
                () -> happyPageService.getOrderHappyPage(orderId, geoId, CommonHttpHeaders.get(), UserCredentials.get()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
