package ru.yandex.travel.api.endpoints.travel_orders;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.BaseEncoding;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.TravelOrdersListRspV1;
import ru.yandex.travel.api.models.travel_orders.TravelOrderListItem;
import ru.yandex.travel.api.proto.orders.TOrderListRequestV2;
import ru.yandex.travel.api.services.orders.OrderListService;
import ru.yandex.travel.api.services.orders.OrderListSource;
import ru.yandex.travel.api.services.orders.OrderType;
import ru.yandex.travel.api.services.orders.TrainOrderListServiceV2;
import ru.yandex.travel.api.services.orders.TravelOrderSubtypeList;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.encryption.EncryptionService;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;

@Service
@Slf4j
public class TravelOrdersImplV2 {

    private static final Set<OrderType> ALL_ORDER_TYPES = ImmutableSet.copyOf(OrderType.values());

    private final EncryptionService encryptionService;

    private final TrainOrderListServiceV2 trainOrderListService;
    private final OrderListService orderListService;

    public TravelOrdersImplV2(EncryptionService encryptionService, OrderListService orderListService,
                              TrainOrderListServiceV2 trainOrderListService) {
        this.encryptionService = encryptionService;
        this.orderListService = orderListService;
        this.trainOrderListService = trainOrderListService;
    }

    public CompletableFuture<TravelOrdersListRspV1> listOrdersFirstPage(int pageSize, String searchTerm,
                                                                        Set<OrderType> orderTypes,
                                                                        DisplayableOrderStatus status) {
        try {
            Set<OrderListSource> sources = new HashSet<>();
            Set<OrderType> usedOrderTypes = ImmutableSet.copyOf(orderTypes);
            if (usedOrderTypes.isEmpty()) {
                usedOrderTypes = ImmutableSet.copyOf(OrderType.values());
            }
            for (OrderType orderType : usedOrderTypes) {
                switch (orderType) {
                    case HOTEL:
                    case AVIA:
                        sources.add(OrderListSource.ORCHESTRATOR);
                        break;
                    case TRAIN:
                        sources.add(OrderListSource.ORCHESTRATOR);
                        sources.add(OrderListSource.TRAIN_API);
                        break;
                }
            }

            Preconditions.checkState(!sources.isEmpty(), "No sources to fetch orders from");

            Map<OrderListSource, Integer> offsets = new HashMap<>();
            for (OrderListSource source : sources) {
                offsets.put(source, 0);
            }


            return innerListOrders(new InternalRequest(offsets, status, usedOrderTypes, pageSize, searchTerm,
                    Instant.now()));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }


    public CompletableFuture<TravelOrdersListRspV1> listNextPage(String nextPageToken) {
        try {
            byte[] encryptedBytes = BaseEncoding.base64Url().decode(nextPageToken);
            byte[] decodedBytes = encryptionService.decrypt(encryptedBytes);
            TOrderListRequestV2 nextPageRequest = TOrderListRequestV2.parseFrom(decodedBytes);
            return innerListOrders(InternalRequest.fromTOrderListRequestV2(nextPageRequest));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private CompletableFuture<TravelOrdersListRspV1> innerListOrders(InternalRequest request) {


        log.info("List order params: {}", request);
        var requestFutures = new ArrayList<CompletableFuture<TravelOrderSubtypeList>>();

        request.offsets.forEach((k, v) -> {
            switch (k) {
                case ORCHESTRATOR:
                    requestFutures.add(
                            orderListService.listOrders(v, request.getPageSize(), request.getOrderTypes(),
                                    request.getDisplayableOrderStatus(), request.getSearchTerm()));
                    break;
                case TRAIN_API:
                    requestFutures.add(
                            trainOrderListService.listOrders(v, request.getPageSize(),
                                    request.getDisplayableOrderStatus(),
                                    request.getSearchTerm()));
            }
        });

        // ugly mambo-jumbo to get a completable future with a list of results from a list of futures
        return CompletableFuture.allOf(requestFutures.toArray(new CompletableFuture[0])).thenApply(ignored ->
                requestFutures.stream().map(CompletableFuture::join).collect(Collectors.toList())
        ).thenApply(resultsList -> {
                    LocalDateTime localRequestedAt = LocalDateTime.ofInstant(request.requestedAt,
                            ZoneId.systemDefault());
                    LocalDateTime requestedAtStartOfDay = localRequestedAt.toLocalDate().atStartOfDay();
                    List<TravelOrderListItem> orders = resultsList.stream().map(TravelOrderSubtypeList::getOrders)
                            .flatMap(List::stream)
                            .map(i -> setFulfilled(i, requestedAtStartOfDay))
                            .sorted((o1, o2) -> {
                                if (o1.getServicedAt().compareTo(o2.getServicedAt()) == 0) {
                                    return o1.getId().compareTo(o2.getId());
                                } else {
                                    if (o1.getServicedAt().compareTo(requestedAtStartOfDay) >= 0
                                            && o2.getServicedAt().compareTo(requestedAtStartOfDay) >= 0) {
                                        return o1.getServicedAt().compareTo(o2.getServicedAt());
                                    } else {
                                        return -o1.getServicedAt().compareTo(o2.getServicedAt());
                                    }
                                }
                            }).collect(Collectors.toList());

                    boolean hasMoreOrders = orders.size() > request.getPageSize() ||
                            resultsList.stream().anyMatch(TravelOrderSubtypeList::isHasMoreOrders);
                    if (hasMoreOrders) {
                        orders = orders.subList(0, request.getPageSize());
                    }

                    orders.forEach(o -> request.getOffsets().compute(o.getSource(), (k, v) -> v + 1));

                    TOrderListRequestV2 nextRequest = request.toOrderListRequestV2();

                    byte[] bytes = nextRequest.toByteArray();
                    byte[] encryptedBytes = encryptionService.encrypt(bytes);

                    TravelOrdersListRspV1 result = new TravelOrdersListRspV1();
                    result.setOrders(orders);
                    result.setHasMoreOrders(hasMoreOrders);
                    if (hasMoreOrders) {
                        result.setNextPageToken(BaseEncoding.base64Url().encode(encryptedBytes));
                    }
                    return result;
                }
        );
    }

    private TravelOrderListItem setFulfilled(TravelOrderListItem item, LocalDateTime requestDate) {
        item.setAlreadyFulfilled(item.getServicedAt().compareTo(requestDate) < 0);
        return item;
    }

    @Getter
    @RequiredArgsConstructor
    @VisibleForTesting
    public static final class InternalRequest {
        private final Map<OrderListSource, Integer> offsets;
        private final DisplayableOrderStatus displayableOrderStatus;
        private final Set<OrderType> orderTypes;
        private final int pageSize;
        private final String searchTerm;
        private final Instant requestedAt;

        public static InternalRequest fromTOrderListRequestV2(TOrderListRequestV2 proto) {

            DisplayableOrderStatus status = null;
            if (proto.getDisplayOrderState() != EDisplayOrderState.OS_UNKNOWN) {
                status = DisplayableOrderStatus.fromProto(proto.getDisplayOrderState());
            }

            Set<OrderType> orderTypes =
                    proto.getTypesList().stream().map(OrderType::fromProto).collect(Collectors.toUnmodifiableSet());
            if (orderTypes.isEmpty()) {
                orderTypes = ALL_ORDER_TYPES;
            }

            Map<OrderListSource, Integer> offsets = new HashMap<>();
            proto.getProviderOffsetsMap().forEach((k, v) -> offsets.put(OrderListSource.fromString(k), v));

            return new InternalRequest(offsets, status, orderTypes, proto.getPageSize(),
                    proto.getSearchTerm(), ProtoUtils.toInstant(proto.getRequestedAt()));
        }

        public TOrderListRequestV2 toOrderListRequestV2() {
            var builder = TOrderListRequestV2.newBuilder();
            builder.setRequestedAt(ProtoUtils.fromInstant(requestedAt))
                    .setPageSize(pageSize);
            if (displayableOrderStatus != null) {
                builder.addAllDisplayOrderStates(displayableOrderStatus.getProtoStates());
                builder.setDisplayOrderState(displayableOrderStatus.getProtoStates().get(0));
            }
            if (searchTerm != null) {
                builder.setSearchTerm(searchTerm);
            }
            offsets.forEach((k, v) -> builder.putProviderOffsets(k.getValue(), v));
            orderTypes.forEach((v) -> builder.addTypes(v.getOrderCommonType()));
            return builder.build();
        }
    }
}
