package ru.yandex.travel.api.endpoints.travel_orders_admin_idm;

import java.io.IOException;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmGetAllRolesResponse;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmInfoResponse;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmModifyRoleResponse;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmRole;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;

@RestController
@RequiredArgsConstructor
@Slf4j
@RequestMapping(value = "api/travel-orders-admin-idm")
public class IdmController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final IdmImpl idmImpl;
    private final ObjectMapper mapper = new ObjectMapper();

    @RequestMapping(value = "/info/", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get info about roles", response = IdmInfoResponse.class)
    public DeferredResult<IdmInfoResponse> info() {
        return responseProcessor.replyWithFuture( "OrdersAdminIdmInfo", () -> idmImpl.getInfo());
    }

    @RequestMapping(value = "/add-role/", method = RequestMethod.POST, consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE, produces = "application/json")
    @ApiOperation(value = "Add role to user", response = IdmModifyRoleResponse.class)
    public DeferredResult<IdmModifyRoleResponse> addRole(@RequestParam(value = "login") String login,
                                                      @RequestParam(value = "role") String role,
                                                      @RequestParam(value = "path", required = false) String path,
                                                      @RequestParam(value = "fields", required = false) String fields) {
        IdmRole idmRole;
        try {
            idmRole = mapper.readerFor(IdmRole.class).readValue(role);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
        return responseProcessor.replyWithFuture( "OrdersAdminIdmAddRole", () -> idmImpl.addRole(login, idmRole));
    }

    @RequestMapping(value = "/remove-role/", method = RequestMethod.POST, consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE, produces = "application/json")
    @ApiOperation(value = "Remove role from user", response = IdmModifyRoleResponse.class)
    public DeferredResult<IdmModifyRoleResponse> removeRole(@RequestParam(value = "login") String login,
                                                            @RequestParam(value = "role") String role,
                                                            @RequestParam(value = "path", required = false) String path,
                                                            @RequestParam(value = "fields", required = false) String fields) {
        IdmRole idmRole;
        try {
            idmRole = mapper.readerFor(IdmRole.class).readValue(role);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
        return responseProcessor.replyWithFuture( "OrdersAdminIdmRemoveRole", () -> idmImpl.removeRole(login, idmRole));
    }

    @RequestMapping(value = "/get-all-roles/", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get all users with their roles", response = IdmGetAllRolesResponse.class)
    public DeferredResult<IdmGetAllRolesResponse> getAllRoles() {
        return responseProcessor.replyWithFuture( "OrdersAdminIdmGetAllRoles", () -> idmImpl.getAllRoles());
    }
}
