package ru.yandex.travel.api.endpoints.travel_orders_admin_idm;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmAllRolesUser;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmGetAllRolesResponse;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmInfoName;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmInfoResponse;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmInfoRoleGroup;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmModifyRoleResponse;
import ru.yandex.travel.api.endpoints.travel_orders_admin_idm.model.IdmRole;
import ru.yandex.travel.api.services.orders.OrchestratorIdmClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.orders.commons.proto.EAdminRole;
import ru.yandex.travel.orders.idm.proto.TIdmAddRoleReq;
import ru.yandex.travel.orders.idm.proto.TIdmGetAllRolesReq;
import ru.yandex.travel.orders.idm.proto.TIdmRemoveRoleReq;
import ru.yandex.travel.orders.idm.proto.TIdmRole;
import ru.yandex.travel.orders.idm.proto.TIdmRoleGroup;
import ru.yandex.travel.orders.idm.proto.TIdmRolesInfoReq;

@Service
@Slf4j
@RequiredArgsConstructor
public class IdmImpl {
    private final static Map<String, EAdminRole> mapToAdminRole = Map.of(
            "developer", EAdminRole.AR_DEVELOPER,
            "operator", EAdminRole.AR_OPERATOR,
            "advancedOperator", EAdminRole.AR_ADVANCED_OPERATOR,
            "advancedBusOperator", EAdminRole.AR_ADV_BUS_OPERATOR,
            "advancedTrainsOperator", EAdminRole.AR_ADV_TRAINS_OPERATOR,
            "advancedAviaOperator", EAdminRole.AR_ADV_AVIA_OPERATOR,
            "advancedHotelsOperator", EAdminRole.AR_ADV_HOTEL_OPERATOR,
            "advancedSuburbanOperator", EAdminRole.AR_ADV_SUBURBAN_OPERATOR,
            "bizdev", EAdminRole.AR_BIZDEV
    );
    private final static Map<EAdminRole, String> mapToIdmRole = Map.of(
            EAdminRole.AR_DEVELOPER, "developer",
            EAdminRole.AR_OPERATOR, "operator",
            EAdminRole.AR_ADVANCED_OPERATOR, "advancedOperator",
            EAdminRole.AR_ADV_BUS_OPERATOR, "advancedBusOperator",
            EAdminRole.AR_ADV_TRAINS_OPERATOR, "advancedTrainsOperator",
            EAdminRole.AR_ADV_AVIA_OPERATOR, "advancedAviaOperator",
            EAdminRole.AR_ADV_HOTEL_OPERATOR, "advancedHotelsOperator",
            EAdminRole.AR_ADV_SUBURBAN_OPERATOR, "advancedSuburbanOperator",
            EAdminRole.AR_BIZDEV, "bizdev"
    );
    private final OrchestratorIdmClientFactory idmClientFactory;

    public CompletableFuture<IdmGetAllRolesResponse> getAllRoles() {
        return FutureUtils.buildCompletableFuture(idmClientFactory.createIdmFutureStub().getAllRoles(TIdmGetAllRolesReq.newBuilder().build()))
                .thenApply(rsp -> {
                    List<IdmAllRolesUser> users = rsp.getUsersList().stream()
                            .filter(Objects::nonNull)
                            .map(user -> {
                                List<IdmRole> roles = user.getRoleGroup().getRoleList().stream()
                                        .map(role -> {
                                            var idmRole = new IdmRole();
                                            idmRole.setRole(mapToIdmRole.get(role.getValue()));
                                            return idmRole;
                                        })
                                        .collect(Collectors.toList());
                                return IdmAllRolesUser.builder()
                                        .login(user.getLogin())
                                        .roles(roles)
                                        .build();
                            })
                            .collect(Collectors.toList());
                    return IdmGetAllRolesResponse.builder()
                            .code(0)
                            .users(users)
                            .build();
                });
    }

    public CompletableFuture<IdmModifyRoleResponse> addRole(String login, IdmRole role) {
        TIdmAddRoleReq request = TIdmAddRoleReq.newBuilder()
                .setLogin(login)
                .setRole(TIdmRole.newBuilder()
                        .setValue(fromIdmRole(role))
                        .build())
                .build();
        return FutureUtils.buildCompletableFuture(idmClientFactory.createIdmFutureStub().addRole(request))
                .handle(this::handleIdmExceptions);
    }

    public CompletableFuture<IdmModifyRoleResponse> removeRole(String login, IdmRole role) {
        TIdmRemoveRoleReq request = TIdmRemoveRoleReq.newBuilder()
                .setLogin(login)
                .setRole(TIdmRole.newBuilder()
                        .setValue(fromIdmRole(role))
                        .build())
                .build();
        return FutureUtils.buildCompletableFuture(idmClientFactory.createIdmFutureStub().removeRole(request))
                .handle(this::handleIdmExceptions);
    }

    public CompletableFuture<IdmInfoResponse> getInfo() {
        TIdmRolesInfoReq request = TIdmRolesInfoReq.newBuilder().build();

        return FutureUtils.buildCompletableFuture(idmClientFactory.createIdmFutureStub().rolesInfo(request))
                .thenApply(rsp -> {
                    TIdmRoleGroup roleGroup = rsp.getRoleGroups(0);
                    Map<String, String> roles = new HashMap<>();

                    roleGroup.getRoleList().forEach(idmRole -> {
                        if (mapToIdmRole.containsKey(idmRole.getValue())) {
                            roles.put(mapToIdmRole.get(idmRole.getValue()), idmRole.getName().getNameRu());
                        }
                    });

                    IdmInfoRoleGroup infoRoleGroup = IdmInfoRoleGroup.builder()
                            .slug(roleGroup.getSlug())
                            .name(IdmInfoName.builder()
                                    .en(roleGroup.getName().getNameEn())
                                    .ru(roleGroup.getName().getNameRu())
                                    .build())
                            .values(roles)
                            .build();

                    return IdmInfoResponse.builder()
                        .code(0)
                        .roles(infoRoleGroup)
                        .build();
                });
    }

    private EAdminRole fromIdmRole(IdmRole idmRole) {
        return mapToAdminRole.getOrDefault(idmRole.getRole(), EAdminRole.AR_UNKNOWN);
    }

    private <R> IdmModifyRoleResponse handleIdmExceptions(R r, Throwable t) {
        if (r != null) {
            return IdmModifyRoleResponse.builder()
                    .code(0)
                    .build();
        } else {
            if (t instanceof StatusRuntimeException) {
                StatusRuntimeException sre = (StatusRuntimeException) t;
                switch (sre.getStatus().getCode()) {
                    case ALREADY_EXISTS:
                    case NOT_FOUND:
                        return IdmModifyRoleResponse.builder()
                                .code(0)
                                .warning(sre.getMessage())
                                .build();
                    case INVALID_ARGUMENT:
                    case FAILED_PRECONDITION:
                        return IdmModifyRoleResponse.builder()
                                .code(2)
                                .fatal(sre.getMessage())
                                .build();
                    default:
                        throw ExceptionUtils.throwException(sre);
                }
            } else {
                throw ExceptionUtils.throwException(t);
            }
        }
    };
}
