package ru.yandex.travel.api.endpoints.trips;

import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.trips.exceptions.IziTravelCityNotFoundException;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.orders.happy_page.model.AfishaCrossSalePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.IziTravelCrossSalePayload;

@RestController
@RequestMapping(value = "/api/trips/activities")
@RequiredArgsConstructor
public class ActivitiesController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    private final ResponseProcessor responseProcessor;
    private final ActivitiesControllerImpl activitiesControllerImpl;

    @RequestMapping(value = "/v1/get_afisha_block", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<AfishaCrossSalePayload> getAfishaBlockV1(
            @RequestParam(value = "startDate") String startDate,
            @RequestParam(value = "resultsLimit") Integer resultsLimit,
            // rawItemsLimit - запрашиваем заданное число событий, потом отфильтровываем события без картинок
            @RequestParam(value = "rawItemsLimit", defaultValue="0") Integer rawItemsLimit,
            @RequestParam(value = "period") Integer period,
            @RequestParam(value = "geoId") Integer geoId) {
        return responseProcessor.replyWithFutureRetrying(
                "ActivitiesControllerGetAfishaBlockV1",
                () -> activitiesControllerImpl.getAfishaBlock(startDate, resultsLimit, rawItemsLimit, period, geoId),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    @RequestMapping(value = "/v1/get_izi_travel_block", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<IziTravelCrossSalePayload> getIziTravelBlockV1(@RequestParam(value = "resultsLimit") Integer resultsLimit,
                                                                       @RequestParam(value = "geoId") Integer geoId) {
        return responseProcessor.replyWithFutureRetrying(
                "ActivitiesControllerGetIziTravelBlockV1",
                () -> activitiesControllerImpl.getIziTravelBlock(resultsLimit, geoId),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    @ExceptionHandler(IziTravelCityNotFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<String> handleIziTravelCityNotFoundException(IziTravelCityNotFoundException e) {
        return ResponseEntity.status(HttpStatus.NOT_FOUND).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }
}
