package ru.yandex.travel.api.endpoints.trips;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.trips.exceptions.IziTravelCityNotFoundException;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.orders.happy_page.AfishaResponseMapper;
import ru.yandex.travel.api.services.orders.happy_page.IziTravelResponseMapper;
import ru.yandex.travel.api.services.orders.happy_page.afisha.AfishaService;
import ru.yandex.travel.api.services.orders.happy_page.afisha.AfishaServiceProperties;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaImageSize;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.IziTravelProperties;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.IziTravelService;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelCitiesChildrenResponse;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelHelper;
import ru.yandex.travel.api.services.orders.happy_page.model.AfishaCrossSalePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.IziTravelCrossSalePayload;

@Service
@RequiredArgsConstructor
@Slf4j
public class ActivitiesControllerImpl {
    private final AfishaService afishaService;
    private final AfishaServiceProperties afishaProperties;
    private final AfishaResponseMapper afishaResponseMapper;
    private final IziTravelService iziTravelService;
    private final IziTravelProperties iziTravelProperties;
    private final IziTravelResponseMapper iziTravelResponseMapper;
    private final RegionsService regionsService;

    public CompletableFuture<AfishaCrossSalePayload> getAfishaBlock(
            String startDate,
            Integer resultsLimit,
            // rawItemsLimit - запрашиваем заданное число событий, потом отфильтровываем события без картинок
            Integer rawItemsLimit,
            Integer period,
            Integer geoId) {
        LocalDate reqDate = LocalDate.parse(startDate, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        return afishaService.getClosestCityInfo(geoId).thenCompose(cityInfo ->
                afishaService.getActualEvents(
                        cityInfo.getGeoid(),
                        Math.max(resultsLimit, rawItemsLimit),
                        period,
                        reqDate,
                        AfishaImageSize.SMALL
                ).thenApply(response -> {
                    AfishaCrossSalePayload afishaPayload = new AfishaCrossSalePayload();
                    afishaPayload.setEvents(
                            afishaResponseMapper.getEventsFromResponse(
                                    afishaProperties.getLinkBaseUrl(),
                                    reqDate.format(DateTimeFormatter.ISO_DATE),
                                    response)
                                    .stream()
                                    .filter(e -> e.getImageUrl() != null && e.getImageUrl().length() > 0)
                                    .limit(resultsLimit)
                                    .collect(Collectors.toUnmodifiableList()));
                    afishaPayload.setRegionUrl(
                            afishaProperties.getLinkBaseUrl() + cityInfo.getUrl() + "/events");
                    afishaPayload.setCoordinates(cityInfo.getCoordinates());
                    return afishaPayload;
                }));
    }

    public CompletableFuture<IziTravelCrossSalePayload> getIziTravelBlock(Integer resultsLimit, Integer geoId) {
        UUID cityUuid = getCityUuid(geoId);
        if (cityUuid != null) {
            List<UUID> toursIds = iziTravelService.getCityChildren(cityUuid, null)
                    .thenApply(response -> response.stream()
                            .filter(IziTravelHelper::checkCityChildIsAllowed)
                            .map(IziTravelCitiesChildrenResponse::getUuid)
                            .collect(Collectors.toList()))
                    .join();
            if (toursIds.size() > resultsLimit) {
                toursIds = toursIds.subList(0, resultsLimit);
            }
            IziTravelCrossSalePayload iziPayload = new IziTravelCrossSalePayload();
            iziPayload.setDirectUrl(iziTravelProperties.getLinkBaseUrl() + "/city/" + cityUuid.toString());
            if (toursIds.size() == 0) {
                iziPayload.setTours(new ArrayList<>());
                return CompletableFuture.completedFuture(iziPayload);
            }
            return iziTravelService.getBatchOfObjects(toursIds)
                    .thenApply(response -> {
                        iziPayload.setTours(iziTravelResponseMapper.getToursFromResponse(iziTravelProperties.getLinkBaseUrl(), response));
                        return iziPayload;
                    });
        } else {
            return CompletableFuture.failedFuture(new IziTravelCityNotFoundException("City not found in IZI.TRAVEL"));
        }
    }

    private UUID getCityUuid(Integer geoId) {
        try {
            var destRegion = regionsService.getRegion(geoId, "en", "en");
            return iziTravelService.searchCityByNameAndCoords(destRegion.getLinguistics().getNominativeCase(), destRegion.getCoordinates()).join();
        } catch (RuntimeException ignored) {
            return null;
        }
    }
}
