package ru.yandex.travel.api.endpoints.trips;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrderReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrderRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrdersWithoutExcludedNextPageReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrdersWithoutExcludedReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrdersWithoutExcludedRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.SelectOrdersReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.SelectOrdersRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.credentials.UserCredentials;

@RestController
@RequestMapping(value = "/api/trips/orders")
@RequiredArgsConstructor
public class OrdersController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final OrdersControllerImpl ordersControllerImpl;

    @RequestMapping(value = "/v1/select_orders", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "", response = SelectOrdersRspV1.class)
    public DeferredResult<SelectOrdersRspV1> selectOrdersV1(@RequestBody SelectOrdersReqV1 req) {
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFutureRetrying(
                "OrdersControllerSelectOrdersV1",
                () -> ordersControllerImpl.selectOrders(req, userCredentials),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    @RequestMapping(value = "/v1/get_orders_without_excluded",
            method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "", response = GetOrdersWithoutExcludedRspV1.class)
    public DeferredResult<GetOrdersWithoutExcludedRspV1> getOrdersWithoutExcluded(@RequestBody GetOrdersWithoutExcludedReqV1 req) {
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFutureRetrying(
                "OrdersControllerGetOrdersWithoutExcludedV1",
                () -> ordersControllerImpl.getOrdersWithoutExcluded(req, userCredentials),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    @RequestMapping(value = "/v1/get_orders_without_excluded_next_page", method = RequestMethod.POST,
            produces = "application/json")
    @ApiOperation(value = "", response = GetOrdersWithoutExcludedRspV1.class)
    public DeferredResult<GetOrdersWithoutExcludedRspV1> getOrdersWithoutExcludedNextPage(
            @RequestBody GetOrdersWithoutExcludedNextPageReqV1 req) {
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFutureRetrying("OrdersControllerGetOrdersWithoutExcludedNextPageV1",
                () -> ordersControllerImpl.getOrdersWithoutExcludedNextPage(req.getNextPageToken(), userCredentials),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }


    @RequestMapping(value = "/v1/get_order", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "", response = GetOrderRspV1.class)
    public DeferredResult<GetOrderRspV1> getOrderV1(@BindFromQuery GetOrderReqV1 req) {
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFutureRetrying(
                "OrdersControllerSelectOrdersV1",
                () -> ordersControllerImpl.getOrder(req),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }
}
