package ru.yandex.travel.api.endpoints.trips;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.io.BaseEncoding;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrderReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrderRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrdersWithoutExcludedReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetOrdersWithoutExcludedRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.SelectOrdersReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.SelectOrdersRspV1;
import ru.yandex.travel.api.services.orders.OrderType;
import ru.yandex.travel.api.services.orders.trips.TripsOrdersService;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.encryption.EncryptionService;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.TOffsetPage;
import ru.yandex.travel.orders.proto.TGetOrdersInfoWithoutExcludedReq;

@Service
@RequiredArgsConstructor
@Slf4j
public class OrdersControllerImpl {
    private final EncryptionService encryptionService;
    private final TripsOrdersService ordersService;

    private static final Set<DisplayableOrderStatus> ALLOWED_ORDER_DISPLAYABLE_STATUSES =
            Set.of(DisplayableOrderStatus.FULFILLED);
    private static final Set<OrderType> ALLOWED_ORDER_TYPES = Set.of(OrderType.AVIA, OrderType.TRAIN, OrderType.HOTEL);

    public CompletableFuture<SelectOrdersRspV1> selectOrders(SelectOrdersReqV1 req, UserCredentials userCredentials) {
        return ordersService.selectOrders(req.getOrderIds(), userCredentials)
                .thenApply(orders -> SelectOrdersRspV1.builder().orders(orders).build());
    }

    public CompletableFuture<GetOrdersWithoutExcludedRspV1> getOrdersWithoutExcluded(GetOrdersWithoutExcludedReqV1 req, UserCredentials userCredentials) {
        var statuses = req.getDisplayableOrderStatuses();
        if (statuses.isEmpty()) {
            statuses = ALLOWED_ORDER_DISPLAYABLE_STATUSES;
        }
        var types = req.getOrderTypes();
        if (types.isEmpty()) {
            types = ALLOWED_ORDER_TYPES;
        }
        return innerGetOrdersWithoutExcluded(
                new InternalGetOrdersWithoutExcludedReq(
                        0,
                        req.getPageSize(),
                        mapOrderIds(req.getExcludedOrderIds()),
                        statuses.stream()
                                .flatMap(s -> s.getProtoStates().stream())
                                .collect(Collectors.toUnmodifiableSet()),
                        types.stream().map(OrderType::getOrderCommonType).collect(Collectors.toUnmodifiableSet()),
                        Instant.now()),
                userCredentials);
    }

    public CompletableFuture<GetOrdersWithoutExcludedRspV1> getOrdersWithoutExcludedNextPage(String nextPageToken,
                                                                                             UserCredentials userCredentials) {
        try {
            byte[] encryptedBytes = BaseEncoding.base64Url().decode(nextPageToken);
            byte[] decodedBytes = encryptionService.decrypt(encryptedBytes);
            var nextPageRequest = TGetOrdersInfoWithoutExcludedReq.parseFrom(decodedBytes);
            return innerGetOrdersWithoutExcluded(InternalGetOrdersWithoutExcludedReq.fromProto(nextPageRequest),
                    userCredentials);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<GetOrderRspV1> getOrder(GetOrderReqV1 req) {
        return ordersService.getOrder(req.getOrderId()).thenApply(order -> {
            var builder = GetOrderRspV1.builder();
            if (order != null) {
                builder.order(order);
            }
            return builder.build();
        });
    }

    private CompletableFuture<GetOrdersWithoutExcludedRspV1> innerGetOrdersWithoutExcluded(InternalGetOrdersWithoutExcludedReq request, UserCredentials userCredentials) {
        log.info("get orders without excluded params: {}", request);
        return ordersService.getOrdersWithoutExcluded(
                request.getExcludedOrderIds(),
                request.getDisplayOrderStates(),
                request.getOrderTypes(),
                request.getOffset(),
                request.getPageSize(),
                request.getRequestedAt(),
                userCredentials).thenApply(
                rsp -> {
                    LocalDateTime localRequestedAt = LocalDateTime.ofInstant(request.requestedAt,
                            ZoneId.systemDefault());
                    LocalDateTime requestedAtStartOfDay = localRequestedAt.toLocalDate().atStartOfDay();
                    var orders = rsp.getOrders();

                    boolean hasMoreOrders = orders.size() > request.getPageSize() || rsp.isHasMoreOrders();
                    if (hasMoreOrders) {
                        orders = orders.subList(0, request.getPageSize());
                    }

                    var nextPage =
                            TOffsetPage.newBuilder().setOffset(request.getOffset() + orders.size()).setLimit(request.pageSize).build();
                    var nextRequest = TGetOrdersInfoWithoutExcludedReq.newBuilder()
                            .setPage(nextPage)
                            .addAllExcludedOrderIds(request.getExcludedOrderIds())
                            .addAllDisplayOrderStates(request.getDisplayOrderStates())
                            .addAllTypes(request.getOrderTypes())
                            .setRequestedAt(ProtoUtils.fromInstant(request.getRequestedAt()))
                            .build();

                    byte[] bytes = nextRequest.toByteArray();
                    byte[] encryptedBytes = encryptionService.encrypt(bytes);
                    var resultBuilder = GetOrdersWithoutExcludedRspV1.builder()
                            .hasMoreOrders(hasMoreOrders)
                            .orders(orders);
                    if (hasMoreOrders) {
                        var nextPageToken = BaseEncoding.base64Url().encode(encryptedBytes);
                        resultBuilder.nextPageToken(nextPageToken);
                    }
                    return resultBuilder.build();
                }
        );
    }

    private List<String> mapOrderIds(List<UUID> orderIds) {
        return orderIds.stream().map(UUID::toString).collect(Collectors.toUnmodifiableList());
    }

    @Getter
    @RequiredArgsConstructor
    @VisibleForTesting
    public static final class InternalGetOrdersWithoutExcludedReq {
        private final Integer offset;
        private final int pageSize;
        private final List<String> excludedOrderIds;
        private final Set<EDisplayOrderState> displayOrderStates;
        private final Set<EDisplayOrderType> orderTypes;
        public final Instant requestedAt;

        public static InternalGetOrdersWithoutExcludedReq fromProto(TGetOrdersInfoWithoutExcludedReq proto) {

            return new InternalGetOrdersWithoutExcludedReq(
                    proto.getPage().getOffset(),
                    proto.getPage().getLimit(),
                    proto.getExcludedOrderIdsList(),
                    proto.getDisplayOrderStatesList().stream().collect(Collectors.toUnmodifiableSet()),
                    proto.getTypesList().stream().collect(Collectors.toUnmodifiableSet()),
                    ProtoUtils.toInstant(proto.getRequestedAt()));
        }
    }
}
