package ru.yandex.travel.api.endpoints.trips;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.trips.exceptions.TripNotFoundException;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetActiveTripsReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetActiveTripsRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetMoreTripsReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetMoreTripsRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripAsyncBlocksReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripAsyncBlocksRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripsReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripsRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;

@RestController
@RequestMapping(value = "/api/trips")
@RequiredArgsConstructor
public class TripsController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final TripsControllerImpl tripsControllerImpl;

    @ExceptionHandler(TripNotFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<String> handleTripNotFoundException(TripNotFoundException e) {
        return ResponseEntity.status(HttpStatus.NOT_FOUND).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @RequestMapping(value = "/v1/get_trips", method = RequestMethod.GET, produces =
            MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<GetTripsRspV1> getTrips(@Validated @BindFromQuery GetTripsReqV1 req) {
        var headers = CommonHttpHeaders.get();
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFuture("GetTripsV1", () -> tripsControllerImpl.getTrips(req, headers, userCredentials));
    }
    @RequestMapping(value = "/v1/get_more_trips", method = RequestMethod.GET, produces =
            MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<GetMoreTripsRspV1> getMoreTrips(@Validated @BindFromQuery GetMoreTripsReqV1 req) {
        var headers = CommonHttpHeaders.get();
        return responseProcessor.replyWithFuture("GetMoreTripsV1", () -> tripsControllerImpl.getMoreTrips(req, headers));
    }
    @RequestMapping(value = "/v1/get_active_trips", method = RequestMethod.GET, produces =
            MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<GetActiveTripsRspV1> getActiveTrips(@Validated @BindFromQuery GetActiveTripsReqV1 req) {
        var headers = CommonHttpHeaders.get();
        return responseProcessor.replyWithFuture("GetActiveTripsV1", () -> tripsControllerImpl.getActiveTrips(req, headers));
    }
    @RequestMapping(value = "/v1/get_trip", method = RequestMethod.GET, produces =
            MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<GetTripRspV1> getTrip(@Validated @BindFromQuery GetTripReqV1 req) {
        var headers = CommonHttpHeaders.get();
        return responseProcessor.replyWithFuture("GetTripV1", () -> tripsControllerImpl.getTrip(req, headers));
    }
    @RequestMapping(value = "/v1/get_trip_async_blocks", method = RequestMethod.GET, produces =
            MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<GetTripAsyncBlocksRspV1> getTripAsyncBlocks(@Validated @BindFromQuery GetTripAsyncBlocksReqV1 req) {
        var headers = CommonHttpHeaders.get();
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFuture("GetTripV1", () -> tripsControllerImpl.getTripAsyncBlocks(req, headers, userCredentials));
    }
}
