package ru.yandex.travel.api.endpoints.trips;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import io.grpc.Context;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.trips.exceptions.TripNotFoundException;
import ru.yandex.travel.api.endpoints.trips.model.PaginatedTripsList;
import ru.yandex.travel.api.endpoints.trips.model.RealTripListItem;
import ru.yandex.travel.api.endpoints.trips.model.TripListItem;
import ru.yandex.travel.api.endpoints.trips.model.TripListItemType;
import ru.yandex.travel.api.endpoints.trips.model.TripOrderState;
import ru.yandex.travel.api.endpoints.trips.model.TripState;
import ru.yandex.travel.api.endpoints.trips.model.blocks.ActivitiesBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.Activity;
import ru.yandex.travel.api.endpoints.trips.model.blocks.ActivityAfisha;
import ru.yandex.travel.api.endpoints.trips.model.blocks.ActivityIziTravel;
import ru.yandex.travel.api.endpoints.trips.model.blocks.ActivityType;
import ru.yandex.travel.api.endpoints.trips.model.blocks.Airline;
import ru.yandex.travel.api.endpoints.trips.model.blocks.AsyncInternalBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.AsyncTripBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.AviaCheckInNotification;
import ru.yandex.travel.api.endpoints.trips.model.blocks.AviaOrder;
import ru.yandex.travel.api.endpoints.trips.model.blocks.BusOrder;
import ru.yandex.travel.api.endpoints.trips.model.blocks.CrossSaleTripBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.ForecastBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.HotelDeferredPaymentNotification;
import ru.yandex.travel.api.endpoints.trips.model.blocks.HotelOrder;
import ru.yandex.travel.api.endpoints.trips.model.blocks.Notification;
import ru.yandex.travel.api.endpoints.trips.model.blocks.NotificationType;
import ru.yandex.travel.api.endpoints.trips.model.blocks.NotificationsBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.Order;
import ru.yandex.travel.api.endpoints.trips.model.blocks.OrderTripBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.OrderTripListItem;
import ru.yandex.travel.api.endpoints.trips.model.blocks.RestrictionsBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.TrainOrder;
import ru.yandex.travel.api.endpoints.trips.model.blocks.TripBlock;
import ru.yandex.travel.api.endpoints.trips.model.blocks.TripBlockType;
import ru.yandex.travel.api.endpoints.trips.model.blocks.TripHotelCrossSale;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetActiveTripsReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetActiveTripsRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetMoreTripsReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetMoreTripsRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripAsyncBlocksReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripAsyncBlocksRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripRspV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripsReqV1;
import ru.yandex.travel.api.endpoints.trips.req_rsp.GetTripsRspV1;
import ru.yandex.travel.api.endpoints.weather.models.ForecastItem;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.services.orders.happy_page.model.AfishaCrossSalePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.IziTravelCrossSalePayload;
import ru.yandex.travel.api.services.trips.TripsClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.TDate;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.komod.trips.api.trips.v1.AfishaEventInfo;
import ru.yandex.travel.komod.trips.api.trips.v1.AviaOfflineCheckIn;
import ru.yandex.travel.komod.trips.api.trips.v1.Block;
import ru.yandex.travel.komod.trips.api.trips.v1.GetActiveTripsReq;
import ru.yandex.travel.komod.trips.api.trips.v1.GetActiveTripsRsp;
import ru.yandex.travel.komod.trips.api.trips.v1.GetMoreTripsReq;
import ru.yandex.travel.komod.trips.api.trips.v1.GetMoreTripsRsp;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripAsyncBlocksReq;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripAsyncBlocksRsp;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripReq;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripRsp;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripsReq;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripsRsp;
import ru.yandex.travel.komod.trips.api.trips.v1.HotelsCrossSale;
import ru.yandex.travel.komod.trips.api.trips.v1.Trip;
import ru.yandex.travel.komod.trips.api.trips.v1.TripsListItem;

@Component
@RequiredArgsConstructor
@Slf4j
public class TripsControllerImpl {
    private final TripsClientFactory tripsClientFactory;

    public CompletableFuture<GetTripsRspV1> getTrips(GetTripsReqV1 req, CommonHttpHeaders headers,
                                                     UserCredentials userCredentials) {
        var passportId = headers.getPassportId();
        try {
            return Context.current().withValue(UserCredentials.KEY, userCredentials).call(() -> {
                var client = tripsClientFactory.createRoundRobinStub();
                var tripsFuture =
                        client.getTrips(GetTripsReq.newBuilder()
                                .setPassportId(passportId)
                                .setGeoId(req.getUserGeoId())
                                .setPastLimit(req.getPastLimit())
                                .build());
                return FutureUtils.buildCompletableFuture(tripsFuture)
                        .thenApply(this::mapResponse);
            });
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<GetMoreTripsRspV1> getMoreTrips(GetMoreTripsReqV1 req, CommonHttpHeaders headers) {
        var passportId = headers.getPassportId();
        var client = tripsClientFactory.createRoundRobinStub();
        var tripsFuture =
                client.getMoreTrips(GetMoreTripsReq.newBuilder()
                        .setContinuationToken(req.getContinuationToken())
                        .setPassportId(passportId)
                        .setLimit(req.getLimit())
                        .setGeoId(req.getUserGeoId())
                        .build());
        return FutureUtils.buildCompletableFuture(tripsFuture)
                .thenApply(this::mapResponse);
    }

    public CompletableFuture<GetActiveTripsRspV1> getActiveTrips(GetActiveTripsReqV1 req, CommonHttpHeaders headers) {
        var passportId = headers.getPassportId();
        var client = tripsClientFactory.createRoundRobinStub();
        var tripsFuture =
                client.getActiveTrips(GetActiveTripsReq.newBuilder()
                        .setPassportId(passportId)
                        .setGeoId(req.getUserGeoId())
                        .setLimit(req.getLimit())
                        .build());
        return FutureUtils.buildCompletableFuture(tripsFuture)
                .thenApply(this::mapResponse);
    }

    public CompletableFuture<GetTripRspV1> getTrip(GetTripReqV1 req, CommonHttpHeaders headers) {
        var passportId = Optional
                .ofNullable(headers.getPassportId())
                .orElse("");
        var client = tripsClientFactory.createRoundRobinStub();
        var tripsFuture = client.getTrip(GetTripReq.newBuilder()
                .setPassportId(passportId)
                .setTripId(req.getTripId())
                .setGeoId(req.getUserGeoId())
                .build());
        return FutureUtils.buildCompletableFuture(tripsFuture)
                .thenApply(this::mapResponse);
    }

    public CompletableFuture<GetTripAsyncBlocksRspV1> getTripAsyncBlocks(GetTripAsyncBlocksReqV1 req,
                                                                         CommonHttpHeaders headers,
                                                                         UserCredentials userCredentials) {
        try {
            return Context.current().withValue(UserCredentials.KEY, userCredentials).call(() -> {
                var client = tripsClientFactory.createRoundRobinStub();
                var tripsFuture =
                        client.getTripAsyncBlocks(GetTripAsyncBlocksReq.newBuilder()
                                .setTripId(req.getTripId())
                                .addAllBlockTypes(req.getBlockTypes().stream()
                                        .map(TripBlockType.BY_VALUE::getByValue)
                                        .map(TripBlockType::getProto)
                                        .collect(Collectors.toUnmodifiableList()))
                                .build());
                return FutureUtils.buildCompletableFuture(tripsFuture)
                        .thenApply(this::mapResponse);
            });
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private GetTripsRspV1 mapResponse(GetTripsRsp rsp) {
        return GetTripsRspV1.builder()
                .active(mapPaginatedList(rsp.getActive()))
                .past(mapPaginatedList(rsp.getPast()))
                .build();
    }

    private GetMoreTripsRspV1 mapResponse(GetMoreTripsRsp rsp) {
        return GetMoreTripsRspV1.builder()
                .active(mapPaginatedList(null))
                .past(mapPaginatedList(rsp.getTrips()))
                .build();
    }

    private GetActiveTripsRspV1 mapResponse(GetActiveTripsRsp rsp) {
        return GetActiveTripsRspV1.builder()
                .trips(rsp.getTripsList().stream()
                        .map(this::mapTripListItem)
                        .collect(Collectors.toUnmodifiableList()))
                .build();
    }

    private GetTripRspV1 mapResponse(GetTripRsp rsp) {
        if (!rsp.hasTrip()) {
            throw new TripNotFoundException("no trip has been found");
        }
        return GetTripRspV1.builder()
                .trip(mapTrip(rsp.getTrip()))
                .build();
    }

    private GetTripAsyncBlocksRspV1 mapResponse(GetTripAsyncBlocksRsp rsp) {
        return GetTripAsyncBlocksRspV1.builder()
                .blocks(mapTripBlocksList(rsp.getBlocksList()))
                .build();
    }

    private ru.yandex.travel.api.endpoints.trips.model.Trip mapTrip(Trip trip) {
        var tripState = TripState.BY_PROTO.getByValue(trip.getState());
        return ru.yandex.travel.api.endpoints.trips.model.Trip.builder()
                .id(trip.getId())
                .state(tripState)
                .title(trip.getTitle())
                .image(trip.getImage().getUrl())
                .beginDate(mapLocalDate(trip.getBeginDate()))
                .endDate(mapLocalDate(trip.getEndDate()))
                .displayDate(trip.getDisplayDate())
                .blocks(mapTripBlocksList(trip.getBlocksList()))
                .build();
    }

    private List<TripBlock> mapTripBlocksList(List<Block> list) {
        if (list == null) {
            return Collections.emptyList();
        }
        return list.stream()
                .map(this::mapTripBlock)
                .collect(Collectors.toUnmodifiableList());
    }

    private TripBlock mapTripBlock(Block block) {
        var blockType = TripBlockType.BY_PROTO.getByValue(block.getType());
        switch (block.getType()) {
            case AVIA_ORDER_BLOCK_TYPE:
                return OrderTripBlock.builder()
                        .type(blockType)
                        .orders(mapAviaOrders(block))
                        .build();
            case TRAIN_ORDER_BLOCK_TYPE:
                return OrderTripBlock.builder()
                        .type(blockType)
                        .orders(mapTrainOrders(block))
                        .build();
            case BUS_ORDER_BLOCK_TYPE:
                return OrderTripBlock.builder()
                        .type(blockType)
                        .orders(mapBusOrders(block))
                        .build();
            case HOTELS_ORDER_BLOCK_TYPE:
                return OrderTripBlock.builder()
                        .type(blockType)
                        .orders(mapHotelOrders(block))
                        .build();
            case HOTELS_CROSS_SALE_BLOCK_TYPE:
                return CrossSaleTripBlock.builder()
                        .type(blockType)
                        .block(mapTripHotelCrossSale(block.getHotelsCrossSale()))
                        .build();
            case ACTIVITIES_BLOCK_TYPE:
                var activitiesBlockProto = block.getActivitiesBlock();
                return AsyncTripBlock.builder()
                        .type(blockType)
                        .isLoaded(activitiesBlockProto.getIsLoaded())
                        .block(mapActivitiesInternalBlock(activitiesBlockProto))
                        .build();
            case RESTRICTIONS_BLOCK_TYPE:
                var restrictionsBlockProto = block.getRestrictionsBlock();
                return RestrictionsBlock.builder()
                        .type(blockType)
                        .block(mapRestrictionsBlock(restrictionsBlockProto))
                        .build();
            case NOTIFICATIONS_BLOCK_TYPE:
                var notificationsBlockProto = block.getNotifications();
                return NotificationsBlock.builder()
                        .type(blockType)
                        .block(NotificationsBlock.Notifications.builder().notifications(mapNotifications(notificationsBlockProto.getNotificationsList())).build())
                        .build();
            case FORECAST_BLOCK_TYPE:
                var forecastBlockProto = block.getForecast();
                return ForecastBlock.builder()
                        .type(blockType)
                        .block(ForecastBlock.BlockImpl.builder()
                                .items(mapForecastItems(forecastBlockProto.getItemsList()))
                                .build())
                        .build();

            default:
                throw new RuntimeException(String.format("Unexpected trip block type %s", block.getType()));
        }

    }

    private List<ForecastItem> mapForecastItems(List<ru.yandex.travel.komod.trips.api.trips.v1.ForecastItem> itemsList) {
        return itemsList.stream()
                .map(this::mapForecastItem)
                .filter(Objects::nonNull)
                .collect(Collectors.toUnmodifiableList());
    }

    private ForecastItem mapForecastItem(ru.yandex.travel.komod.trips.api.trips.v1.ForecastItem forecastItem) {
        return ForecastItem.builder()
                .description(forecastItem.getDescription())
                .title(forecastItem.getTitle())
                .imageUrl(forecastItem.getImageUrl())
                .url(forecastItem.getUrl())
                .itemType(forecastItem.getItemType())
                .build();
    }

    private List<Notification> mapNotifications(List<ru.yandex.travel.komod.trips.api.trips.v1.Notification> notificationsProto) {
        return notificationsProto.stream().map(this::mapNotification).filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    private Notification mapNotification(ru.yandex.travel.komod.trips.api.trips.v1.Notification notification) {
        switch (notification.getType()) {
            case NOTIFICATION_TYPE_AVIA_ONLINE_CHECK_IN:
                return mapAviaOnlineCheckInNotification(notification);
            case NOTIFICATION_TYPE_HOTEL_DEFERRED_PAYMENT:
                return mapHotelDeferredPaymentNotification(notification);
            default:
                return null;
        }
    }

    private Notification mapHotelDeferredPaymentNotification(ru.yandex.travel.komod.trips.api.trips.v1.Notification notification) {
        var hotelNotification = notification.getHotelDeferredPayment();
        return HotelDeferredPaymentNotification.builder()
                .type(NotificationType.HOTEL_DEFERRED_PAYMENT)
                .order(mapHotelOrder(hotelNotification.getOrder()))
                .build();
    }

    private AviaCheckInNotification mapAviaOnlineCheckInNotification(ru.yandex.travel.komod.trips.api.trips.v1.Notification notification) {
        var aviaNotification = notification.getAviaOnlineCheckIn();
        var airline = aviaNotification.getAirline();
        var notificationBuilder = AviaCheckInNotification.builder()
                .type(NotificationType.AVIA_CHECK_IN)
                .orderId(aviaNotification.getOrderId())
                .airline(Airline.builder()
                        .title(airline.getTitle())
                        .logo(airline.getSvgLogo())
                        .color(airline.getLogoBgColor())
                        .build())
                .flightNumber(aviaNotification.getFlightNumber())
                .flightTitle(aviaNotification.getFlightTitle())
                .pnr(aviaNotification.getPnr())
                .offlineCheckIn(mapAviaOfflineCheckIn(aviaNotification.getOfflineCheckIn()));
        if (!Strings.isNullOrEmpty(aviaNotification.getRegistrationUrl())) {
            notificationBuilder.registrationUrl(aviaNotification.getRegistrationUrl());
        }
        if (!Strings.isNullOrEmpty(aviaNotification.getUpdatedAt())) {
            notificationBuilder.updatedAt(aviaNotification.getUpdatedAt());
        }
        return notificationBuilder.build();
    }

    private AviaCheckInNotification.AviaOfflineCheckIn mapAviaOfflineCheckIn(AviaOfflineCheckIn offlineCheckIn) {
        var offlineCheckInBuilder = AviaCheckInNotification.AviaOfflineCheckIn.builder();
        if (!Strings.isNullOrEmpty(offlineCheckIn.getGate())) {
            offlineCheckInBuilder.gate(offlineCheckIn.getGate());
        }
        if (!Strings.isNullOrEmpty(offlineCheckIn.getCheckInCounters())) {
            offlineCheckInBuilder.gate(offlineCheckIn.getCheckInCounters());
        }
        return offlineCheckInBuilder.build();
    }

    private RestrictionsBlock.BlockImpl mapRestrictionsBlock(ru.yandex.travel.komod.trips.api.trips.v1.RestrictionsBlock restrictionsBlockProto) {
        var request = new RestrictionsBlock.Request();
        if (restrictionsBlockProto.getFromGeoId() != 0) {
            request.setFromGeoId((long) restrictionsBlockProto.getFromGeoId());
        }
        if (restrictionsBlockProto.getToGeoId() != 0) {
            request.setToGeoId((long) restrictionsBlockProto.getToGeoId());
        }
        if (!restrictionsBlockProto.getFromPointKey().isEmpty()) {
            request.setFromPointKey(restrictionsBlockProto.getFromPointKey());
        }
        if (!restrictionsBlockProto.getToPointKey().isEmpty()) {
            request.setToPointKey(restrictionsBlockProto.getToPointKey());
        }

        return RestrictionsBlock.BlockImpl.builder()
                .fromCountryTitle(restrictionsBlockProto.getFromCountryTitle())
                .toCountryTitle(restrictionsBlockProto.getToCountryTitle())
                .request(request)
                .build();
    }

    private AsyncInternalBlock mapActivitiesInternalBlock(ru.yandex.travel.komod.trips.api.trips.v1.ActivitiesBlock activitiesBlock) {
        if (!activitiesBlock.getIsLoaded()) {
            return null;
        }
        return ActivitiesBlock.builder()
                .activities(activitiesBlock.getActivitiesList().stream()
                        .map(this::mapActivity)
                        .collect(Collectors.toUnmodifiableList()))
                .build();
    }

    private Activity mapActivity(ru.yandex.travel.komod.trips.api.trips.v1.Activity activity) {
        var activityType = ActivityType.BY_PROTO.getByValue(activity.getType());
        switch (activityType) {
            case AFISHA:
                var protoEventInfo = activity.getAfishaEventInfo();
                var payloadEventInfo = new AfishaCrossSalePayload.EventInfo();
                payloadEventInfo.setName(protoEventInfo.getName());
                var eventMinPrice = protoEventInfo.getMinPrice();
                if (!eventMinPrice.equals(AfishaEventInfo.Price.getDefaultInstance())) {
                    payloadEventInfo.setMinPrice(
                            Money.of(eventMinPrice.getValue(), eventMinPrice.getCurrency()));
                }
                payloadEventInfo.setType(protoEventInfo.getType());
                if (!protoEventInfo.getDateTime().isEmpty()) {
                    payloadEventInfo.setDateTime(LocalDateTime.parse(protoEventInfo.getDateTime()));
                }
                payloadEventInfo.setDateText(protoEventInfo.getDateText());
                payloadEventInfo.setImageUrl(protoEventInfo.getImageUrl());
                payloadEventInfo.setTags(protoEventInfo.getTagsList());
                payloadEventInfo.setEventUrl(protoEventInfo.getEventUrl());
                return ActivityAfisha.builder()
                        .type(activityType)
                        .payload(payloadEventInfo)
                        .build();
            case IZI_TRAVEL:
                var protoTourInfo = activity.getIziTravelTourInfo();
                var payloadTourInfo = new IziTravelCrossSalePayload.TourInfo();
                payloadTourInfo.setName(protoTourInfo.getName());
                payloadTourInfo.setImageUrl(protoTourInfo.getImageUrl());
                payloadTourInfo.setTourUrl(protoTourInfo.getTourUrl());
                payloadTourInfo.setType(protoTourInfo.getType());
                payloadTourInfo.setCategory(protoTourInfo.getCategory());
                payloadTourInfo.setDuration(protoTourInfo.getDuration());
                return ActivityIziTravel.builder()
                        .type(activityType)
                        .payload(payloadTourInfo)
                        .build();
            default:
                throw new RuntimeException(String.format("Unexpected trip activity type %s", activity.getType()));
        }
    }

    private TripHotelCrossSale mapTripHotelCrossSale(HotelsCrossSale crossSale) {
        var request = new TripHotelCrossSale.Request();
        request.setAdults(crossSale.getAdults());
        request.setAdults(crossSale.getAdults());
        request.setPointKey(crossSale.getSettlementPointKey());
        request.setCheckinDate(mapLocalDate(crossSale.getCheckInDate()));
        request.setCheckoutDate(mapLocalDate(crossSale.getCheckOutDate()));
        request.setChildrenAges(crossSale.getChildrenAgesList());
        request.setTotalHotelLimit(crossSale.getTotalHotelLimit());

        return TripHotelCrossSale.builder()
                .title(crossSale.getTitle())
                .request(request)
                .build();
    }

    @NotNull
    private List<Order> mapHotelOrders(Block block) {
        return block.getHotelOrders().getValuesList().stream()
                .map(this::mapHotelOrder)
                .collect(Collectors.toUnmodifiableList());
    }

    private HotelOrder mapHotelOrder(ru.yandex.travel.komod.trips.api.trips.v1.HotelOrder hotelOrder) {
        var builder = HotelOrder.builder()
                .id(hotelOrder.getId())
                .title(hotelOrder.getTitle())
                .image(hotelOrder.getImage().getUrl())
                .displayCheckinCheckoutDates(hotelOrder.getDisplayDates())
                .address(hotelOrder.getAddress())
                .coordinates(Coordinates.of(
                        hotelOrder.getCoordinates().getLatitude(),
                        hotelOrder.getCoordinates().getLongitude()))
                .state(mapOrderState(hotelOrder.getTripOrderState()));
        if (hotelOrder.getStars() > 0) {
            builder.stars(hotelOrder.getStars());
        }
        return builder.build();
    }

    @NotNull
    private List<Order> mapTrainOrders(Block block) {
        return block.getTrainOrders().getValuesList().stream()
                .map(trainOrder -> TrainOrder.builder()
                        .id(trainOrder.getId())
                        .title(trainOrder.getTitle())
                        .displayDateForward(trainOrder.getDisplayDateForward())
                        .displayDateBackward(trainOrder.getDisplayDateBackward())
                        .trains(trainOrder.getTrainsList().stream()
                                .map(train -> TrainOrder.Train.builder()
                                        .number(train.getNumber())
                                        .description(train.getDescription())
                                        .build())
                                .collect(Collectors.toUnmodifiableList()))
                        .hasTransferWithStationChange(trainOrder.getHasTransferWithStationChange())
                        .refundedTicketsCount(trainOrder.getRefundedTicketsCount())
                        .state(mapOrderState(trainOrder.getTripOrderState()))
                        .build())
                .collect(Collectors.toUnmodifiableList());
    }

    @NotNull
    private List<Order> mapBusOrders(Block block) {
        return block.getBusOrders().getValuesList().stream()
                .map(busOrder -> {
                    var builder = BusOrder.builder()
                            .id(busOrder.getId())
                            .title(busOrder.getTitle())
                            .description(busOrder.getDescription())
                            .carrierName(busOrder.getCarrierName())
                            .refundedTicketsCount(busOrder.getRefundedTicketsCount())
                            .displayDateForward(busOrder.getDisplayDateForward())
                            .state(mapOrderState(busOrder.getTripOrderState()));
                    if (!Strings.isNullOrEmpty(busOrder.getDownloadBlankToken())) {
                        builder.downloadBlankToken(busOrder.getDownloadBlankToken());
                    }
                    return builder.build();
                })
                .collect(Collectors.toUnmodifiableList());
    }

    @NotNull
    private List<Order> mapAviaOrders(Block block) {
        return block.getAviaOrders().getValuesList().stream()
                .map(aviaOrder -> {
                    var builder = AviaOrder.builder()
                            .id(aviaOrder.getId())
                            .title(aviaOrder.getTitle())
                            .displayDateForward(aviaOrder.getDisplayDateForward())
                            .displayDateBackward(aviaOrder.getDisplayDateBackward())
                            .pnr(aviaOrder.getPnr())
                            .state(mapOrderState(aviaOrder.getTripOrderState()))
                            .airlines(aviaOrder.getAirlinesList().stream()
                                    .map(airline -> Airline.builder()
                                            .title(airline.getTitle())
                                            .logo(airline.getSvgLogo())
                                            .color(airline.getLogoBgColor())
                                            .build())
                                    .collect(Collectors.toUnmodifiableList()));
                    if (!Strings.isNullOrEmpty(aviaOrder.getRegistrationUrl())) {
                        builder.registrationUrl(aviaOrder.getRegistrationUrl());
                    }
                    return builder.build();
                })
                .collect(Collectors.toUnmodifiableList());
    }

    private LocalDate mapLocalDate(TDate date) {
        if (date == null) {
            return null;
        }
        return LocalDate.of(date.getYear(), date.getMonth(), date.getDay());
    }

    private PaginatedTripsList mapPaginatedList(ru.yandex.travel.komod.trips.api.trips.v1.PaginatedTripsList list) {
        if (list == null) {
            return PaginatedTripsList.builder().trips(Collections.emptyList()).continuationToken(null).build();
        }
        return PaginatedTripsList.builder()
                .trips(list.getTripsList().stream()
                        .map(this::mapTripListItem)
                        .collect(Collectors.toUnmodifiableList()))
                .continuationToken(list.getContinuationToken())
                .build();
    }

    private TripListItem mapTripListItem(TripsListItem trip) {
        var protoItemType = TripListItemType.BY_PROTO.getByValue(trip.getType());
        var tripState = TripState.BY_PROTO.getByValue(trip.getState());
        switch (trip.getType()) {
            case REAL_TYPE:
                var realItem = trip.getRealItem();
                return RealTripListItem.builder()
                        .id(realItem.getId())
                        .type(protoItemType)
                        .displayDate(realItem.getDisplayDate())
                        .image(realItem.getImage())
                        .title(realItem.getTitle())
                        .state(tripState)
                        .build();
            case ORDER_TYPE:
                var orderItem = trip.getOrderItem();
                return OrderTripListItem.builder()
                        .orderId(orderItem.getOrderId())
                        .type(protoItemType)
                        .state(tripState)
                        .displayDate(orderItem.getDisplayDate())
                        .image(orderItem.getImage())
                        .title(orderItem.getTitle())
                        .build();
            default:
                throw new RuntimeException(String.format("Unexpected trip list item type %s", trip.getType()));
        }
    }

    private TripOrderState mapOrderState(ru.yandex.travel.komod.trips.api.trips.v1.TripOrderState orderStateProto) {
        return TripOrderState.BY_PROTO.getByValue(orderStateProto);
    }
}
