package ru.yandex.travel.api.endpoints.trust_callback;

import javax.servlet.http.HttpServletRequest;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;

@RestController
@RequiredArgsConstructor
@Slf4j
@RequestMapping(value = "/api/travel_orders_trust_callback")
public class TrustCallbackController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final TrustCallbackImpl callbackImpl;

    @RequestMapping(value = "/v1/basket_status_changed/", method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE)
    @ApiOperation(value = "Trust basket/refund status change callback")
    public DeferredResult<Void> basketStatusChangedV1(HttpServletRequest request) {
        return responseProcessor.replyWithFutureRetrying(
                "TrustCallbackBasketStatusChangedV1",
                () -> callbackImpl.basketStatusChanged(request.getParameterMap()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
