package ru.yandex.travel.api.endpoints.weather;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.TextStyle;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.weather.models.Forecast;
import ru.yandex.travel.api.endpoints.weather.models.ForecastItem;
import ru.yandex.travel.api.endpoints.weather.models.WeatherByGeoIdResponse;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.weather.MonthDisplayNameProvider;
import ru.yandex.travel.api.services.weather.WeatherService;
import ru.yandex.travel.api.services.weather.WeatherServiceProperties;
import ru.yandex.travel.api.services.weather.model.LinguisticCase;
import ru.yandex.travel.api.services.weather.model.WeatherByGeoIdData;

@RestController
@RequestMapping(value = "/api/weather")
@RequiredArgsConstructor
@EnableConfigurationProperties({WeatherServiceProperties.class})
@Slf4j
public class WeatherController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final WeatherService weatherService;
    private final MonthDisplayNameProvider monthDisplayNameProvider;
    private final String defaultWeatherImageUrl = "https://avatars.mds.yandex.net/get-avia/1586578/weather-logo/orig";
    private final Locale RU = Locale.forLanguageTag("ru");

    @RequestMapping(value = "/v1/get_by_geo_id", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<WeatherByGeoIdResponse> getByGeoId(
            @RequestParam(value = "startDate") String startDate,
            @RequestParam(value = "days") Integer days,
            @RequestParam(value = "geoId") Integer geoId) {
        LocalDate reqDate = LocalDate.parse(startDate, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        return responseProcessor.replyWithFutureRetrying(
                "WeatherV1GetByGeoId",
                () -> getWeatherByGeoId(geoId, reqDate, days),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private CompletableFuture<WeatherByGeoIdResponse> getWeatherByGeoId(Integer geoId, LocalDate reqDate, Integer days) {
        return weatherService.getWeatherByGeoId(geoId, reqDate, reqDate.plusDays(days - 1)).thenApply(
                weatherByGeoIdData -> WeatherByGeoIdResponse.builder()
                        .data(mapForecast(weatherByGeoIdData))
                        .build()
        );
    }

    private Forecast mapForecast(Optional<WeatherByGeoIdData> weatherData) {
        List<ForecastItem> items = new LinkedList<>();

        if (weatherData.isEmpty()) {
            return Forecast.builder()
                    .items(items)
                    .build();
        }
        var data = weatherData.get();
        var countItems = data.forecastItems.size() + data.climateItems.size();
        for (WeatherByGeoIdData.ForecastItem item : data.forecastItems) {
            items.add(mapForecastItem(item, data.url, countItems));
        }
        var firstClimateBlock = true;
        for (WeatherByGeoIdData.ClimateItem item : data.climateItems) {
            items.add(mapClimateItem(item, data.url, firstClimateBlock));
            firstClimateBlock = false;
        }

        return Forecast.builder()
                .items(items)
                .build();
    }

    private ForecastItem mapClimateItem(WeatherByGeoIdData.ClimateItem item, String url, boolean firstClimateBlock) {
        return ForecastItem.builder()
                .url(url)
                .imageUrl(defaultWeatherImageUrl)
                .title(mapClimateTitle(item, firstClimateBlock))
                .description(mapClimateDescription(item))
                .itemType("CLIMATE")
                .build();
    }

    private String mapClimateDescription(WeatherByGeoIdData.ClimateItem item) {
        var days = item.getPrecipitationDays();
        var dayWord = days == 1 ? "день" : "дней";
        return String.format("%d %s с осадками", days, dayWord);
    }

    private String mapClimateTitle(WeatherByGeoIdData.ClimateItem item, boolean firstClimateBlock) {
        var format = firstClimateBlock ? "Обычно в %s %+d...%+d°" : "В %s %+d...%+d°";
        return String.format(
                format,
                monthDisplayNameProvider.getDisplayName(item.getMonth(), LinguisticCase.PREPOSITIONAL),
                item.getMaxDayTemperature(),
                item.getMinNightTemperature());

    }

    private ForecastItem mapForecastItem(WeatherByGeoIdData.ForecastItem item, String url, Integer countItems) {
        return ForecastItem.builder()
                .url(url)
                .imageUrl(item.getIcon())
                .title(mapForecastTitle(item.getDate(), countItems))
                .description(String.format("%+d...%+d°", item.getTemperature(), item.getMinTemperature()))
                .itemType("FORECAST")
                .build();
    }

    private String mapForecastTitle(LocalDate date, Integer countItems) {
        return String.format("%s, %d %s",
                mapWeekDay(date, countItems),
                date.getDayOfMonth(),
                mapMonth(date, countItems));
    }

    private String mapMonth(LocalDate date, Integer countItems) {
        var displayName = monthDisplayNameProvider.getDisplayName(date.getMonth(), LinguisticCase.GENITIVE);
        if (countItems != 1) {
            displayName = displayName.substring(0, 3);
        }
        return displayName;
    }

    private String mapWeekDay(LocalDate date, Integer countItems) {
        var day = date.getDayOfWeek();
        if (countItems == 1) {
            return toCamelCase(day.getDisplayName(TextStyle.FULL, RU));
        } else {
            return day.getDisplayName(TextStyle.SHORT, RU);
        }
    }

    private String toCamelCase(String value) {
        if (value.isEmpty()) {
            return value;
        }
        return value.substring(0, 1).toUpperCase() + value.substring(1);
    }
}
