package ru.yandex.travel.api.endpoints.workflow_management;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.workflow_management.req_rsp.PauseWorkflowReqV1;
import ru.yandex.travel.api.endpoints.workflow_management.req_rsp.PauseWorkflowRspV1;
import ru.yandex.travel.api.endpoints.workflow_management.req_rsp.ResumeWorkflowReqV1;
import ru.yandex.travel.api.endpoints.workflow_management.req_rsp.ResumeWorkflowRspV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;

@RestController
@RequestMapping(value = "/api/workflow_management")
@RequiredArgsConstructor
@Slf4j
@ConditionalOnProperty("workflow-management.enabled")
public class WorkflowManagementController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final WorkflowManagementImpl workflowManagementImpl;

    @RequestMapping(value = "/v1/pause_workflow", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Pause workflow with ID", response = PauseWorkflowRspV1.class)
    public DeferredResult<PauseWorkflowRspV1> pauseWorkflow(@RequestBody @Valid PauseWorkflowReqV1 request) {
        return responseProcessor.replyWithFuture(
                "WorkflowManagement::PauseWorkflow",
                () -> workflowManagementImpl.pauseWorkflow(request)
        );
    }

    @RequestMapping(value = "/v1/resume_workflow", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Resume workflow with ID", response = ResumeWorkflowRspV1.class)
    public DeferredResult<ResumeWorkflowRspV1> resumeWorkflow(@RequestBody @Valid ResumeWorkflowReqV1 request) {
        return responseProcessor.replyWithFuture(
                "WorkflowManagement::ResumeWorkflow",
                () -> workflowManagementImpl.resumeWorkflow(request)
        );
    }
}
