package ru.yandex.travel.api.grpc

import io.grpc.Status
import io.grpc.StatusException
import io.grpc.stub.StreamObserver
import lombok.RequiredArgsConstructor
import lombok.extern.slf4j.Slf4j
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.travel.api.bizdev_admin.BizdevAdminApiGrpc.BizdevAdminApiImplBase
import ru.yandex.travel.api.config.hotels.administrator.AdministratorAdminInterfaceProvider
import ru.yandex.travel.api.endpoints.administrator.req_rsp.HotelConnectionState
import ru.yandex.travel.api.endpoints.administrator.req_rsp.UnpublishedReason
import ru.yandex.travel.api.services.orders.OrchestratorAdminClientFactory
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.commons.grpc.ServerUtils
import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.administrator.grpc.proto.GetHotelConnectionReq
import ru.yandex.travel.hotels.administrator.grpc.proto.GetHotelConnectionRsp
import ru.yandex.travel.hotels.proto.EPartnerId
import ru.yandex.travel.orders.admin.proto.TGetPaymentsStateReq
import java.util.concurrent.CompletableFuture
import java.util.function.Function

@GrpcService(authenticateUser = true, authenticateService = true)
@Slf4j
@RequiredArgsConstructor
class BizdevAdminService @Autowired constructor(
    private val administratorApiFactory: AdministratorAdminInterfaceProvider,
    private val orchestratorApiFactory: OrchestratorAdminClientFactory,
): BizdevAdminApiImplBase()  {
    val log: Logger = LoggerFactory.getLogger(javaClass)

    fun <ReqT, RspT> callAsync(request: ReqT, observer: StreamObserver<RspT>, errorTag: String,
                               handler: Function<ReqT, CompletableFuture<RspT>>
    ) {
        ServerUtils.asynchronously(log, request, observer,
            { rq: ReqT ->
                try {
                    return@asynchronously handler.apply(rq)
                } catch (e: Exception) {
//                    meters.incrementCounter(errorTag)
                    throw e
                }
            },
            {
                err: Throwable -> return@asynchronously StatusException(Status.INTERNAL)
            }
        )
    }

    override fun getHotelConnection(request: GetHotelConnectionReq, responseObserver: StreamObserver<GetHotelConnectionRsp>) {
        callAsync(request, responseObserver, "tag42", {
            val partnerId: EPartnerId = request.partnerId
            val hotelCode: String = request.hotelCode

            val hotelConnRsp = FutureUtils.buildCompletableFuture(
                administratorApiFactory.provideInternalInterface()
                        .getHotelConnection(
                            GetHotelConnectionReq.newBuilder()
                                .setPartnerId(partnerId)
                                .setHotelCode(hotelCode)
                                .build()))
                    .thenCompose { hotelConnRsp ->
                        if (!hotelConnRsp.hasBillingClientId()) {
                            return@thenCompose CompletableFuture.completedFuture(hotelConnRsp)
                        }

                        return@thenCompose FutureUtils.buildCompletableFuture(
                            orchestratorApiFactory.createAdminFutureStub(false)
                                .getPaymentsState(
                                    TGetPaymentsStateReq.newBuilder()
                                        .setBillingClientId(hotelConnRsp.billingClientId)
                                        .build())
                        ).handle{ getPaymentsStateRsp, ex ->
                            // TODO
                            val paymentsEnabled = if (ex == null) getPaymentsStateRsp.paymentsEnabled else false
                            return@handle hotelConnRsp
                        }
                    }
            return@callAsync hotelConnRsp
        })
    }

    private fun createGetHotelConnectionRsp(hotelConn: GetHotelConnectionRsp, paymentsEnabled: Boolean): ru.yandex.travel.api.endpoints.administrator.req_rsp.GetHotelConnectionRsp {
        return ru.yandex.travel.api.endpoints.administrator.req_rsp.GetHotelConnectionRsp(
            hotelConnectionState = HotelConnectionState.convertFromProto(hotelConn.connectionState),
            permalink = if (hotelConn.hasPermalink()) hotelConn.permalink else null,
            slug = if (hotelConn.hasPermalink()) "permalink/${hotelConn.permalink}" else "",
            stTicket = hotelConn.stTicket,
            paperAgreement = hotelConn.paperAgreement,
            paymentsEnabled = paymentsEnabled,
            hotelName = hotelConn.hotelName,
            billingClientId = if (hotelConn.hasBillingClientId()) hotelConn.billingClientId else null,
            billingContractId = if (hotelConn.hasBillingClientId()) hotelConn.billingContractId else null,
            unpublishedReason = UnpublishedReason.convertFromProto(hotelConn.unpublishedReason),
            accountantEmails = hotelConn.accountantEmailsList,
            contractPersonEmails = hotelConn.contractPersonEmailsList,
        )
    }
}
