package ru.yandex.travel.api.health.hotels;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.autoconfigure.health.ConditionalOnEnabledHealthIndicator;
import org.springframework.boot.actuate.health.CompositeHealthContributor;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthContributor;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.actuate.health.NamedContributor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import ru.yandex.travel.tvm.TvmWrapper;

@Component
@ConditionalOnProperty("tvm.enabled")
@ConditionalOnEnabledHealthIndicator("tvm")
public class TvmHealthContributor implements CompositeHealthContributor {

    @RequiredArgsConstructor
    private static class TvmWrapperHealthIndicator implements HealthIndicator {
        private final TvmWrapper tvmWrapper;

        @Override
        public Health health() {
            if (tvmWrapper.isInitialized()) {
                return Health.up().build();
            } else {
                return Health.down().build();
            }
        }
    }

    private final Map<String, HealthContributor> contributors = new HashMap<>();

    public TvmHealthContributor(@Autowired(required = false) Map<String, TvmWrapper> wrapperMap) {
        if (Objects.nonNull(wrapperMap) && !wrapperMap.isEmpty()) {
            for (Map.Entry<String, TvmWrapper> tvmWrapperEntry : wrapperMap.entrySet()) {
                contributors.put(tvmWrapperEntry.getKey(), new TvmWrapperHealthIndicator(tvmWrapperEntry.getValue()));
            }
        }
    }

    @Override
    public HealthContributor getContributor(String name) {
        return contributors.get(name);
    }

    @NotNull
    @Override
    public Iterator<NamedContributor<HealthContributor>> iterator() {
        return contributors.entrySet().stream()
                .map((entry) -> NamedContributor.of(entry.getKey(), entry.getValue()))
                .iterator();
    }
}
