package ru.yandex.travel.api.infrastucture;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.config.common.EncryptionConfigurationProperties;
import ru.yandex.travel.orders.commons.proto.TAviaPaymentTestContext;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.orders.commons.proto.TBusTestContext;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.commons.proto.TSuburbanTestContext;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;

@Slf4j
@RequiredArgsConstructor
@Service
@EnableConfigurationProperties(EncryptionConfigurationProperties.class)
public class ApiTokenEncrypter {
    private static final String HMAC_SHA256_ALGORITHM = "HmacSHA256";

    private final EncryptionConfigurationProperties encryptionConfigurationProperties;

    public String toAviaTestContextToken(TAviaTestContext aviaTestContext) {
        return toToken(aviaTestContext);
    }

    public String toAviaPaymentTestContextToken(TAviaPaymentTestContext aviaPaymentTestContext) {
        return toToken(aviaPaymentTestContext);
    }

    public String toTrainTestContextToken(TTrainTestContext trainTestContext) {
        return toToken(trainTestContext);
    }

    public String toBusTestContextToken(TBusTestContext busTestContext) {
        return toToken(busTestContext);
    }

    public String toDownloadBlankToken(TDownloadBlankToken proto) {
        return toToken(proto);
    }

    public String toPaymentTestContextToken(TPaymentTestContext paymentTestContext) {
        return toToken(paymentTestContext);
    }

    public String toSuburbanTestContextToken(TSuburbanTestContext suburbanTestContext) {
        return toToken(suburbanTestContext);
    }

    public TAviaTestContext fromAviaTestContextToken(String aviaTestContextToken) {
        var dataPart = validateTokenAndExtractData(aviaTestContextToken);
        try {
            return TAviaTestContext.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted test context token", e);
        }
    }

    public TAviaPaymentTestContext fromAviaPaymentTestContextToken(String aviaPaymentTestContextToken) {
        var dataPart = validateTokenAndExtractData(aviaPaymentTestContextToken);
        try {
            return TAviaPaymentTestContext.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted payment test context token", e);
        }
    }

    public TTrainTestContext fromTrainTestContextToken(String trainTestContextToken) {
        var dataPart = validateTokenAndExtractData(trainTestContextToken);
        try {
            return TTrainTestContext.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted test context token", e);
        }
    }

    public TBusTestContext fromBusTestContextToken(String testContextToken) {
        var dataPart = validateTokenAndExtractData(testContextToken);
        try {
            return TBusTestContext.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted test context token", e);
        }
    }

    public TDownloadBlankToken fromDownloadBlankToken(String token) {
        var dataPart = validateTokenAndExtractData(token);
        try {
            return TDownloadBlankToken.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted download blank token", e);
        }
    }

    public TPaymentTestContext fromPaymentTestContextToken(String trainTestContextToken) {
        var dataPart = validateTokenAndExtractData(trainTestContextToken);
        try {
            return TPaymentTestContext.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted payment test context token", e);
        }
    }

    public TSuburbanTestContext fromSuburbanTestContextToken(String suburbanTestContextToken) {
        var dataPart = validateTokenAndExtractData(suburbanTestContextToken);
        try {
            return TSuburbanTestContext.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Corrupted test context token", e);
        }
    }

    private <T extends Message> String toToken(T token) {
        String dataPart = dataPartToString(token);
        String dataHmac = calculateHmac(dataPart);
        return dataPart + ":" + dataHmac;
    }

    private String validateTokenAndExtractData(String token) {
        String[] stringParts = token.split(":");
        Preconditions.checkState(stringParts.length == 2, "Token must have exactly 2 parts");

        String dataPart = stringParts[0];
        String messageHmac = stringParts[1];

        String calculatedHmac = calculateHmac(dataPart);
        if (!calculatedHmac.equals(messageHmac)) {
            throw new IllegalArgumentException("Corrupted token");
        }
        return dataPart;
    }

    private String dataPartToString(Message message) {
        return BaseEncoding.base64Url().encode(message.toByteArray());
    }

    private String calculateHmac(String data) {
        try {
            SecretKeySpec signingKey =
                    new SecretKeySpec(encryptionConfigurationProperties.getEncryptionKey().getBytes(),
                            HMAC_SHA256_ALGORITHM);
            Mac mac = Mac.getInstance(HMAC_SHA256_ALGORITHM);
            mac.init(signingKey);
            return BaseEncoding.base64Url().encode(mac.doFinal(data.getBytes()));
        } catch (Exception e) {
            throw new IllegalStateException("Error encoding token", e);
        }
    }
}
