package ru.yandex.travel.api.infrastucture;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import lombok.extern.slf4j.Slf4j;
import org.springframework.core.MethodParameter;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.servlet.mvc.method.annotation.ServletModelAttributeMethodProcessor;

/**
 * Method processor supports {@link ParamName} parameters renaming
 */

@Slf4j
public class RenamingProcessor extends ServletModelAttributeMethodProcessor {

    private final Map<Class<?>, Map<String, String>> replaceMap = new ConcurrentHashMap<>();

    public RenamingProcessor(boolean annotationNotRequired) {
        super(annotationNotRequired);
    }


    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return parameter.hasParameterAnnotation(BindFromQuery.class);
    }

    @Override
    protected void bindRequestParameters(WebDataBinder binder, NativeWebRequest nativeWebRequest) {
        if (binder instanceof ParamNameDataBinder) {
            Object target = binder.getTarget();
            Class<?> targetClass = target.getClass();
            if (!replaceMap.containsKey(targetClass)) {
                Map<String, String> renameMap = new HashMap<>();
                analyzeClass(targetClass, renameMap);
                if (!renameMap.isEmpty()) {
                    replaceMap.put(targetClass, renameMap);
                }
            }
            Map<String, String> mapping = replaceMap.get(targetClass);
            ((ParamNameDataBinder)binder).setRenameMapping(mapping);
        }
        super.bindRequestParameters(binder, nativeWebRequest);
    }

    private static void analyzeClass(Class<?> targetClass, Map<String, String> renameMap) {
        Field[] fields = targetClass.getDeclaredFields();
        for (Field field : fields) {
            ParamName paramNameAnnotation = field.getAnnotation(ParamName.class);
            if (paramNameAnnotation != null && !paramNameAnnotation.value().isEmpty()) {
                renameMap.put(paramNameAnnotation.value(), field.getName());
            }
        }
        Class<?> superClass = targetClass.getSuperclass();
        if (superClass != null) {
            analyzeClass(superClass, renameMap);
        }
    }
}
