package ru.yandex.travel.api.infrastucture;

import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.function.Supplier;

import com.google.common.base.Strings;
import io.grpc.Context;
import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryStrategy;
import ru.yandex.travel.grpc.AppCallIdGenerator;

import static ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers.defaultRetryStrategy;

@Service
@Slf4j
@RequiredArgsConstructor
public class ResponseProcessor {
    private final Tracer tracer;

    private final Retry retryHelper;

    public <Rsp> DeferredResult<Rsp> replyWithFuture(String operationName, Supplier<CompletableFuture<Rsp>> supplier) {
        return replyWithFuture(operationName, supplier, null);
    }

    public <Rsp> DeferredResult<Rsp> replyWithFuture(String operationName, Supplier<CompletableFuture<Rsp>> supplier,
                                                     Duration timeout) {
        return replyWithFutureRetrying(operationName, supplier, defaultRetryStrategy(), timeout);
    }

    public <Rsp> DeferredResult<Rsp> replyWithFutureRetrying(String operationName,
                                                             Supplier<CompletableFuture<Rsp>> supplier,
                                                             RetryStrategy<Rsp> retryStrategy) {
        return replyWithFutureRetrying(operationName, supplier, retryStrategy, null);
    }

    public <Rsp> DeferredResult<Rsp> replyWithFutureRetrying(String operationName,
                                                             Supplier<CompletableFuture<Rsp>> supplier,
                                                             RetryStrategy<Rsp> retryStrategy,
                                                             Duration timeout) {

        String requestId = CommonHttpHeaders.get().getRequestId();
        if (Strings.isNullOrEmpty(requestId)) {
            // falling back to awacs request id
            requestId = CommonHttpHeaders.get().getAwacsRequestId();
        }
        return replyWithFutureRetrying(operationName, supplier, retryStrategy, null, requestId);
    }

    public <Rsp> DeferredResult<Rsp> replyWithFutureRetrying(String operationName,
                                                             Supplier<CompletableFuture<Rsp>> supplier,
                                                             RetryStrategy<Rsp> retryStrategy,
                                                             Duration timeout, String callIdSeed) {

        Span span = tracer.buildSpan(operationName).start();
        String retryOperationName = operationName + "::ImplCall";
        try (Scope scope = tracer.scopeManager().activate(span)) {
            DeferredResult<Rsp> result;
            result = new DeferredResult<>(timeout != null ? timeout.toMillis() : null);
            // using request id header content as app call id generator seed
            final AppCallIdGenerator appCallIdGenerator;
            if (!Strings.isNullOrEmpty(callIdSeed)) {
                appCallIdGenerator = AppCallIdGenerator.newInstance(callIdSeed);
            } else {
                appCallIdGenerator = AppCallIdGenerator.newInstance();
            }
            Context.current().withValue(AppCallIdGenerator.KEY, appCallIdGenerator).run(() -> {
                CompletableFuture<Rsp> responseFuture;
                if (retryStrategy != null) {
                    responseFuture = retryHelper.withRetry(retryOperationName, supplier, retryStrategy);
                } else {
                    responseFuture = supplier.get();
                }
                responseFuture.whenComplete((r, t) -> {
                    try (Scope asyncScope = tracer.scopeManager().activate(span)) {
                        if (t == null) {
                            result.setResult(r);
                        } else {
                            Throwable realException = t;
                            // Dropping extra wrappers
                            while (realException instanceof CompletionException) {
                                realException = Objects.requireNonNull(realException.getCause());
                            }
                            result.setErrorResult(realException);
                            span.log("An exception has occurred and has been successfully handled. Detailed " +
                                    "information: " +
                                    realException.getMessage());
                        }
                    } finally {
                        span.finish();
                    }
                });
            });
            return result;
        }
    }
}
