package ru.yandex.travel.api.infrastucture.shutdown;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

@Component
@EnableConfigurationProperties(ShutdownProperties.class)
@Slf4j
public class ShutdownManager implements ApplicationContextAware, HealthIndicator {
    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS");

    private final AtomicBoolean shutdownInitiated = new AtomicBoolean(false);

    private Instant shutdownStartedAt;

    private ApplicationContext ctx;

    private ScheduledExecutorService shutdownSequenceExecutor;

    private Duration shutdownTimeout;

    public ShutdownManager(ShutdownProperties properties) {
        this.shutdownTimeout = properties.getShutdownTimeout();
        shutdownSequenceExecutor = new ScheduledThreadPoolExecutor(
                1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("shutdown-thread-%s").build()
        );

    }

    public void initShutdown() {
        if (!shutdownInitiated.getAndSet(true)) {
            log.info("Shutdown initiated, health check flag switched off. Shutting down in {} ms",
                    shutdownTimeout.toMillis());
            shutdownStartedAt = Instant.now();
            shutdownSequenceExecutor.schedule(this::runShutdownSequence, shutdownTimeout.toMillis(),
                    TimeUnit.MILLISECONDS);
        }
    }

    private void runShutdownSequence() {
        log.info("Shutting down application context");
        ((ConfigurableApplicationContext) ctx).close();
        log.info("Everything shut down. Exiting");
        System.exit(0);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.ctx = applicationContext;
    }

    @Override
    public Health health() {
        if (shutdownInitiated.get()) {
            return Health.down()
                    .withDetail("Shutdown started at", dateTimeFormatter.format(shutdownStartedAt.atZone(ZoneId.systemDefault())))
                    .build();
        } else {
            return Health.up().build();
        }
    }
}
